<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\User;
use App\Models\Publicacao;
use App\Models\Criativo;
use App\Models\Empresa;
use App\Models\Hotspot;
use App\Models\PublicacaoImpressao;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;

class AnalyticsFlowTest extends TestCase
{
    use RefreshDatabase;

    private User $user;
    private Publicacao $publication;
    private Criativo $creative;
    private Hotspot $hotspot;

    protected function setUp(): void
    {
        parent::setUp();

        // Arrange: Create common necessary models
        $this->user = User::factory()->create(['is_superadmin' => true]);
        $this->actingAs($this->user);

        $this->creative = Criativo::factory()->create([
            'path' => 'midias/test_creative.jpg'
        ]);

        $this->hotspot = Hotspot::factory()->create([
            'serial' => 'TEST_SERIAL_123'
        ]);

        $this->publication = Publicacao::factory()->create([
            'criativo_id' => $this->creative->id,
            'data_inicio' => now()->subDay(),
            'data_fim' => now()->addDay(),
        ]);

        // Arrange: Simulate raw audit logs from the external script
        DB::table('auditoria_splash')->insert([
            [
                'hs_serial' => $this->hotspot->serial,
                'imagem' => $this->creative->path,
                'cl_mac_address' => 'AA:BB:CC:DD:EE:01',
                'created_at' => now(),
                'processed' => false
            ],
            [
                'hs_serial' => $this->hotspot->serial,
                'imagem' => $this->creative->path,
                'cl_mac_address' => 'AA:BB:CC:DD:EE:02',
                'created_at' => now(),
                'processed' => false
            ],
            [
                'hs_serial' => $this->hotspot->serial,
                'imagem' => 'midias/unknown_creative.jpg', // This one should not be processed
                'cl_mac_address' => 'AA:BB:CC:DD:EE:03',
                'created_at' => now(),
                'processed' => false
            ]
        ]);
    }

    /**
     * @test
     * Tests if the ETL command correctly processes raw logs and aggregates the data.
     */
    public function etl_command_processes_logs_correctly(): void
    {
        // Assertions before ETL
        $this->assertDatabaseCount('auditoria_splash', 3);
        $this->assertDatabaseCount('publicacao_impressoes', 0);
        $this->assertDatabaseHas('auditoria_splash', ['processed' => false]);

        // Act: Run the ETL command
        Artisan::call('etl:process-splash-audits');

        // Assertions after ETL
        $this->assertEquals(2, DB::table('auditoria_splash')->where('processed', true)->count());
        $this->assertEquals(1, DB::table('auditoria_splash')->where('processed', false)->count()); // The unknown creative log
        
        $this->assertDatabaseCount('publicacao_impressoes', 1);
        $this->assertDatabaseHas('publicacao_impressoes', [
            'publicacao_id' => $this->publication->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->creative->id,
            'impressoes' => 2,
            'usuarios_unicos' => 2,
            'data' => now()->toDateString()
        ]);
    }

    /**
     * @test
     * Tests if the analytics endpoints return the correct processed data after the ETL has run.
     */
    public function analytics_endpoints_return_correct_data(): void
    {
        // Arrange: Run the ETL first to process the data
        Artisan::call('etl:process-splash-audits');

        // === Test Main Statistics Page ===
        // Act: Visit the main statistics page
        $response = $this->get(route('publicacoes.estatisticas', $this->publication));

        // Assert: Check for correct data on the page
        $response->assertStatus(200);
        $response->assertSee('2</span>', false); // Check for total impressions
        $response->assertSee('2</span>', false); // Check for unique users

        // === Test Analytics Modal JSON Endpoint ===
        // Act: Call the JSON endpoint for the modal
        $jsonResponse = $this->get(route('publicacoes.analyticsJson', $this->publication));

        // Assert: Check the JSON structure and values
        $jsonResponse->assertStatus(200);
        $jsonResponse->assertJsonPath('stats.total_impressoes', 2);
        $jsonResponse->assertJsonPath('stats.total_usuarios_unicos', 2);
        $jsonResponse->assertJsonCount(2, 'logs_acesso'); // It should fetch the raw logs
    }
}
