<?php

use App\Models\Hotspot;
use App\Models\Publicacao;
use App\Models\Empresa;
use App\Models\Criativo;
use App\Models\WalledGarden;
use App\Models\PublicacaoImpressao;
use App\Models\PublicacaoClique;
use Illuminate\Foundation\Testing\RefreshDatabase;

uses(RefreshDatabase::class);

beforeEach(function () {
    // Criar dados de teste
    $this->franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
    $this->cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
    
    $this->hotspot = Hotspot::factory()->create([
        'serial' => 'TEST123',
        'franquia_id' => $this->franquia->id,
        'cliente_id' => $this->cliente->id,
        'status' => true,
    ]);
    
    $this->criativo = Criativo::factory()->aprovado()->imagem()->create();
    
    $this->publicacao = Publicacao::factory()->create([
        'criativo_id' => $this->criativo->id,
        'franquia_id' => $this->franquia->id,
        'cliente_id' => $this->cliente->id,
        'tipo_alcance' => 'todos',
        'status' => 'ativa',
        'data_inicio' => now()->subDays(5),
        'data_fim' => now()->addDays(25),
        'link_destino' => 'https://exemplo.com',
        'habilitar_clique' => true,
        'track_impressoes' => true,
        'track_cliques' => true,
    ]);
    
    WalledGarden::factory()->domain()->create([
        'publicacao_id' => $this->publicacao->id,
        'endereco' => 'example.com',
    ]);
});

test('api test endpoint retorna sucesso', function () {
    $response = $this->getJson('/api/test');
    
    $response->assertStatus(200)
        ->assertJson([
            'success' => true,
            'message' => 'API FreeFi Admin funcionando!',
        ]);
});

test('heartbeat atualiza last activity do hotspot', function () {
    $response = $this->postJson('/api/hotspot/heartbeat', [
        'serial' => 'TEST123',
        'vpn_ip' => '10.0.0.1',
    ]);
    
    $response->assertStatus(200)
        ->assertJson([
            'success' => true,
            'message' => 'Heartbeat registrado.',
        ]);
    
    $this->hotspot->refresh();
    expect($this->hotspot->last_activity_at)->not->toBeNull();
    expect($this->hotspot->vpn_ip)->toBe('10.0.0.1');
});

test('heartbeat retorna 404 para hotspot inexistente', function () {
    $response = $this->postJson('/api/hotspot/heartbeat', [
        'serial' => 'INEXISTENTE',
    ]);
    
    $response->assertStatus(404)
        ->assertJson([
            'success' => false,
            'message' => 'Hotspot não encontrado.',
        ]);
});

test('buscar publicacoes retorna publicacoes vigentes', function () {
    $response = $this->getJson('/api/hotspot/TEST123/publicacoes');
    
    $response->assertStatus(200)
        ->assertJson([
            'success' => true,
            'count' => 1,
        ])
        ->assertJsonStructure([
            'data' => [
                '*' => [
                    'id',
                    'titulo',
                    'posicao',
                    'link_destino',
                    'habilitar_clique',
                    'track_impressoes',
                    'track_cliques',
                    'criativo' => ['id', 'title', 'tipo', 'formato'],
                    'walled_gardens',
                ],
            ],
        ]);
});

test('buscar walled gardens retorna lista correta', function () {
    $response = $this->getJson("/api/hotspot/publicacao/{$this->publicacao->id}/walled-gardens");
    
    $response->assertStatus(200)
        ->assertJson([
            'success' => true,
            'count' => 1,
        ])
        ->assertJsonStructure([
            'data' => [
                '*' => ['tipo', 'endereco', 'descricao'],
            ],
        ]);
    
    expect($response->json('data')[0]['endereco'])->toBe('example.com');
});

test('registrar impressao incrementa contador', function () {
    $response = $this->postJson('/api/hotspot/impressao', [
        'serial' => 'TEST123',
        'publicacao_id' => $this->publicacao->id,
        'criativo_id' => $this->criativo->id,
        'usuarios_unicos' => 5,
    ]);
    
    $response->assertStatus(200)
        ->assertJson([
            'success' => true,
            'message' => 'Impressão registrada.',
        ]);
    
    $impressao = PublicacaoImpressao::where('hotspot_id', $this->hotspot->id)
        ->where('publicacao_id', $this->publicacao->id)
        ->first();
    
    expect($impressao)->not->toBeNull();
    expect($impressao->impressoes)->toBe(1);
    expect($impressao->usuarios_unicos)->toBe(5);
});

test('registrar impressao multiplas vezes incrementa corretamente', function () {
    // Primeira impressão
    $this->postJson('/api/hotspot/impressao', [
        'serial' => 'TEST123',
        'publicacao_id' => $this->publicacao->id,
        'criativo_id' => $this->criativo->id,
        'usuarios_unicos' => 3,
    ]);
    
    // Segunda impressão
    $this->postJson('/api/hotspot/impressao', [
        'serial' => 'TEST123',
        'publicacao_id' => $this->publicacao->id,
        'criativo_id' => $this->criativo->id,
        'usuarios_unicos' => 7,
    ]);
    
    $impressao = PublicacaoImpressao::where('hotspot_id', $this->hotspot->id)
        ->where('publicacao_id', $this->publicacao->id)
        ->first();
    
    expect($impressao->impressoes)->toBe(2);
    expect($impressao->usuarios_unicos)->toBe(7); // Deve pegar o máximo
});

test('registrar clique cria registro e retorna redirect url', function () {
    $response = $this->postJson('/api/hotspot/clique', [
        'serial' => 'TEST123',
        'publicacao_id' => $this->publicacao->id,
        'criativo_id' => $this->criativo->id,
        'ip_usuario' => '192.168.1.100',
        'user_agent' => 'Mozilla/5.0',
    ]);
    
    $response->assertStatus(200)
        ->assertJson([
            'success' => true,
            'message' => 'Clique registrado.',
            'redirect_url' => 'https://exemplo.com',
        ]);
    
    $clique = PublicacaoClique::where('hotspot_id', $this->hotspot->id)
        ->where('publicacao_id', $this->publicacao->id)
        ->first();
    
    expect($clique)->not->toBeNull();
    expect($clique->ip_usuario)->toBe('192.168.1.100');
    expect($clique->user_agent)->toBe('Mozilla/5.0');
});

test('registrar clique falha se tracking desabilitado', function () {
    $this->publicacao->update(['track_cliques' => false]);
    
    $response = $this->postJson('/api/hotspot/clique', [
        'serial' => 'TEST123',
        'publicacao_id' => $this->publicacao->id,
        'criativo_id' => $this->criativo->id,
        'ip_usuario' => '192.168.1.100',
    ]);
    
    $response->assertStatus(400)
        ->assertJson([
            'success' => false,
            'message' => 'Tracking de cliques desabilitado.',
        ]);
});

test('registrar clique falha se link nao habilitado', function () {
    $this->publicacao->update(['habilitar_clique' => false]);
    
    $response = $this->postJson('/api/hotspot/clique', [
        'serial' => 'TEST123',
        'publicacao_id' => $this->publicacao->id,
        'criativo_id' => $this->criativo->id,
        'ip_usuario' => '192.168.1.100',
    ]);
    
    $response->assertStatus(400)
        ->assertJson([
            'success' => false,
            'message' => 'Clique não habilitado para esta publicação.',
        ]);
});

test('validacao requer campos obrigatorios', function () {
    $response = $this->postJson('/api/hotspot/impressao', []);
    
    $response->assertStatus(422)
        ->assertJsonValidationErrors(['serial', 'publicacao_id', 'criativo_id']);
});
