<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;
use App\Models\User;
use App\Models\Midia;
use App\Models\Empresa;

class MidiaCrudTest extends TestCase
{
    use RefreshDatabase;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->actingAs($this->user);
        Storage::fake('public');
    }

    public function test_can_view_midias_index()
    {
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertViewIs('midias.index');
    }

    public function test_can_view_midia_create_form()
    {
        $response = $this->get(route('midias.create'));
        $response->assertStatus(200);
        $response->assertViewIs('midias.create');
    }

    public function test_can_create_midia()
    {
        $empresa = Empresa::factory()->create();
        // $file = UploadedFile::fake()->image('midia.jpg'); // Commented out due to missing GD library

        $midiaData = [
            'title' => 'New Midia',
            'type' => 'image',
            // 'media_file' => $file, // Commented out due to missing GD library
            'empresa_id' => $empresa->id,
            'equipment_serial' => '12345',
            'position' => 1,
            'start_datetime' => '2025-01-01 10:00:00',
            'end_datetime' => '2025-01-01 11:00:00',
        ];

        $response = $this->post(route('midias.store'), $midiaData);

        $response->assertRedirect(route('midias.index'));
        $this->assertDatabaseHas('midias', ['title' => 'New Midia', 'position' => 1]);
        // Storage::disk('public')->assertExists('midias/' . $file->hashName()); // Commented out due to missing GD library
    }

    public function test_can_view_midia_edit_form()
    {
        $midia = Midia::factory()->create();
        $response = $this->get(route('midias.edit', $midia));
        $response->assertStatus(200);
        $response->assertViewIs('midias.edit');
    }

    public function test_can_view_midia_preview_link()
    {
        $midia = Midia::factory()->create();
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee(Storage::url($midia->path));
    }

    public function test_can_update_midia()
    {
        $midia = Midia::factory()->create(['position' => 1]); // Set a known position
        $empresa = Empresa::factory()->create();
        // $file = UploadedFile::fake()->image('new_midia.jpg'); // Commented out due to missing GD library

        $updatedData = [
            'title' => 'Updated Midia Title',
            'type' => 'video',
            // 'media_file' => $file, // Commented out due to missing GD library
            'empresa_id' => $empresa->id,
            'equipment_serial' => '54321',
            'position' => 2,
            'start_datetime' => '2025-02-01 10:00:00',
            'end_datetime' => '2025-02-01 11:00:00',
        ];

        $response = $this->put(route('midias.update', $midia), $updatedData);

        $response->assertRedirect(route('midias.index'));
        $this->assertDatabaseHas('midias', ['id' => $midia->id, 'title' => 'Updated Midia Title', 'position' => 2]);
        // Storage::disk('public')->assertExists('midias/' . $file->hashName()); // Commented out due to missing GD library
    }

    public function test_can_delete_midia()
    {
        $midia = Midia::factory()->create();
        $path = $midia->path;

        $response = $this->delete(route('midias.destroy', $midia));

        $response->assertRedirect(route('midias.index'));
        $this->assertSoftDeleted('midias', ['id' => $midia->id]);
        // Storage::disk('public')->assertMissing($path);
    }
}
