<?php

namespace Tests\Unit;

use App\Models\HotspotCustomization;
use App\Models\Hotspot;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class HotspotCustomizationTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Testa criação de customização
     */
    public function test_can_create_hotspot_customization(): void
    {
        $hotspot = Hotspot::factory()->create();
        $user = User::factory()->create();

        $customization = HotspotCustomization::create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Test WiFi',
            'logo_url' => 'img/test-logo.png',
            'primary_color' => '#FF5722',
            'is_active' => true,
            'created_by' => $user->id,
            'updated_by' => $user->id,
        ]);

        $this->assertDatabaseHas('hotspot_customizations', [
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Test WiFi',
            'primary_color' => '#FF5722',
        ]);

        $this->assertTrue($customization->is_active);
    }

    /**
     * Testa relacionamento com Hotspot
     */
    public function test_belongs_to_hotspot(): void
    {
        $customization = HotspotCustomization::factory()->create();

        $this->assertInstanceOf(Hotspot::class, $customization->hotspot);
        $this->assertEquals($customization->hotspot_id, $customization->hotspot->id);
    }

    /**
     * Testa relacionamento com User (creator)
     */
    public function test_has_creator_relationship(): void
    {
        $user = User::factory()->create();
        $customization = HotspotCustomization::factory()->create([
            'created_by' => $user->id
        ]);

        $this->assertInstanceOf(User::class, $customization->creator);
        $this->assertEquals($user->id, $customization->creator->id);
    }

    /**
     * Testa geração de JavaScript object
     */
    public function test_generates_javascript_object(): void
    {
        $customization = HotspotCustomization::factory()->create([
            'brand_name' => 'Test Brand',
            'primary_color' => '#123456',
            'is_active' => true,
        ]);

        $jsObject = $customization->toJavaScriptObject();

        $this->assertJson($jsObject);

        $decoded = json_decode($jsObject, true);
        $this->assertEquals('Test Brand', $decoded['brand_name']);
        $this->assertEquals('#123456', $decoded['primary_color']);

        // Não deve incluir campos internos
        $this->assertArrayNotHasKey('id', $decoded);
        $this->assertArrayNotHasKey('created_at', $decoded);
        $this->assertArrayNotHasKey('updated_at', $decoded);
    }

    /**
     * Testa geração do loader script
     */
    public function test_generates_loader_script(): void
    {
        $customization = HotspotCustomization::factory()->create([
            'brand_name' => 'My Custom WiFi',
            'logo_url' => 'img/custom-logo.png',
            'primary_color' => '#00FF00',
        ]);

        $script = $customization->generateLoaderScript();

        // Deve conter tags script
        $this->assertStringContainsString('<script>', $script);
        $this->assertStringContainsString('</script>', $script);

        // Deve conter o objeto de customização
        $this->assertStringContainsString('window.freefiCustomization', $script);

        // Deve conter a função de aplicação
        $this->assertStringContainsString('applyCustomization', $script);

        // Deve conter dados específicos
        $this->assertStringContainsString('My Custom WiFi', $script);
        $this->assertStringContainsString('#00FF00', $script);
    }

    /**
     * Testa geração de script com CSS customizado
     */
    public function test_generates_script_with_custom_css(): void
    {
        $customization = HotspotCustomization::factory()->create([
            'custom_css' => '.my-class { color: red; }',
        ]);

        $script = $customization->generateLoaderScript();

        $this->assertStringContainsString('custom_css', $script);
        $this->assertStringContainsString('.my-class { color: red; }', $script);
    }

    /**
     * Testa scope activeBySerial
     */
    public function test_scope_active_by_serial(): void
    {
        // Criar hotspot com serial específico
        $hotspot = Hotspot::factory()->create(['serial' => 'TEST-SERIAL-123']);

        // Customização ativa para este hotspot
        $activeCustomization = HotspotCustomization::factory()->active()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Active Custom',
        ]);

        // Customização inativa para o mesmo hotspot
        $inactiveCustomization = HotspotCustomization::factory()->inactive()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Inactive Custom',
        ]);

        // Buscar por serial
        $result = HotspotCustomization::activeBySerial('TEST-SERIAL-123')->first();

        $this->assertNotNull($result);
        $this->assertEquals($activeCustomization->id, $result->id);
        $this->assertEquals('Active Custom', $result->brand_name);
    }

    /**
     * Testa que scope activeBySerial não retorna inativos
     */
    public function test_scope_active_by_serial_excludes_inactive(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'SERIAL-456']);

        // Apenas customização inativa
        HotspotCustomization::factory()->inactive()->create([
            'hotspot_id' => $hotspot->id,
        ]);

        $result = HotspotCustomization::activeBySerial('SERIAL-456')->first();

        $this->assertNull($result);
    }

    /**
     * Testa scope activeBySerial com serial inexistente
     */
    public function test_scope_active_by_serial_with_nonexistent_serial(): void
    {
        $result = HotspotCustomization::activeBySerial('NONEXISTENT-SERIAL')->first();

        $this->assertNull($result);
    }

    /**
     * Testa campos booleanos
     */
    public function test_boolean_fields_are_cast_correctly(): void
    {
        $customization = HotspotCustomization::factory()->create([
            'enable_advertisement' => true,
            'require_name' => false,
            'enable_cpf_mask' => true,
            'show_mac_address' => false,
        ]);

        $this->assertTrue($customization->enable_advertisement);
        $this->assertFalse($customization->require_name);
        $this->assertTrue($customization->enable_cpf_mask);
        $this->assertFalse($customization->show_mac_address);

        $this->assertIsBool($customization->enable_advertisement);
        $this->assertIsBool($customization->require_name);
    }

    /**
     * Testa campos de tempo são integers
     */
    public function test_time_fields_are_cast_to_integers(): void
    {
        $customization = HotspotCustomization::factory()->create([
            'modal_countdown_seconds' => 15,
            'banner_rotation_seconds' => 8,
            'status_refresh_seconds' => 90,
        ]);

        $this->assertIsInt($customization->modal_countdown_seconds);
        $this->assertIsInt($customization->banner_rotation_seconds);
        $this->assertIsInt($customization->status_refresh_seconds);

        $this->assertEquals(15, $customization->modal_countdown_seconds);
        $this->assertEquals(8, $customization->banner_rotation_seconds);
        $this->assertEquals(90, $customization->status_refresh_seconds);
    }

    /**
     * Testa soft delete
     */
    public function test_uses_soft_delete(): void
    {
        $customization = HotspotCustomization::factory()->create();
        $id = $customization->id;

        $customization->delete();

        // Não deve estar no banco quando consulta normalmente
        $this->assertNull(HotspotCustomization::find($id));

        // Deve estar com soft delete
        $this->assertNotNull(HotspotCustomization::withTrashed()->find($id));
    }

    /**
     * Testa valores padrão
     */
    public function test_has_default_values(): void
    {
        $hotspot = Hotspot::factory()->create();
        $user = User::factory()->create();

        $customization = HotspotCustomization::create([
            'hotspot_id' => $hotspot->id,
            'created_by' => $user->id,
            'updated_by' => $user->id,
        ]);

        // Verificar alguns valores padrão da migration
        $this->assertEquals('freefi.com.br', $customization->brand_name);
        $this->assertEquals('img/logo.png', $customization->logo_url);
        $this->assertEquals('#3e4d59', $customization->primary_color);
        $this->assertEquals(10, $customization->modal_countdown_seconds);
        $this->assertTrue($customization->enable_advertisement);
        $this->assertTrue($customization->is_active);
    }

    /**
     * Testa constraint unique (um hotspot só pode ter uma customização)
     */
    public function test_hotspot_can_have_only_one_customization(): void
    {
        $this->expectException(\Exception::class);

        $hotspot = Hotspot::factory()->create();

        // Primeira customização - OK
        HotspotCustomization::factory()->create(['hotspot_id' => $hotspot->id]);

        // Segunda customização para o mesmo hotspot - deve falhar
        HotspotCustomization::factory()->create(['hotspot_id' => $hotspot->id]);
    }
}
