<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\Midia;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class MidiaTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $empresa;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->empresa = Empresa::factory()->create();
        $this->actingAs($this->user);
    }

    public function test_midia_belongs_to_empresa()
    {
        $midia = Midia::factory()->create(['empresa_legado_id' => $this->empresa->id]);

        $this->assertInstanceOf(Empresa::class, $midia->empresa);
        $this->assertEquals($this->empresa->id, $midia->empresa->id);
    }

    public function test_midia_belongs_to_creator()
    {
        $midia = Midia::factory()->create(['created_by' => $this->user->id]);

        $this->assertInstanceOf(User::class, $midia->creator);
        $this->assertEquals($this->user->id, $midia->creator->id);
    }

    public function test_midia_belongs_to_updater()
    {
        $midia = Midia::factory()->create(['updated_by' => $this->user->id]);

        $this->assertInstanceOf(User::class, $midia->updater);
        $this->assertEquals($this->user->id, $midia->updater->id);
    }

    public function test_midia_belongs_to_destroyer()
    {
        $midia = Midia::factory()->create();
        $midia->delete(); // This should set deleted_by through Userstamps trait

        $midia->refresh();
        $this->assertInstanceOf(User::class, $midia->destroyer);
        $this->assertEquals($this->user->id, $midia->destroyer->id);
    }

    public function test_midia_has_fillable_attributes()
    {
        $fillable = [
            'title', 'type', 'path', 'empresa_id', 'equipment_serial', 
            'status', 'views_count', 'position', 'start_datetime', 'end_datetime'
        ];

        $midia = new Midia();
        $this->assertEquals($fillable, $midia->getFillable());
    }

    public function test_midia_uses_soft_deletes()
    {
        $midia = Midia::factory()->create();
        $midia->delete();

        $this->assertSoftDeleted('midias', ['id' => $midia->id]);
        $this->assertNotNull($midia->fresh()->deleted_at);
    }

    public function test_midia_default_status_is_active()
    {
        $midia = Midia::factory()->create();
        
        // Status 0 = active (default)
        $this->assertEquals(0, $midia->status);
    }

    public function test_midia_default_views_count_is_zero()
    {
        $midia = Midia::factory()->create();
        
        $this->assertEquals(0, $midia->views_count);
    }

    public function test_midia_position_is_between_1_and_6()
    {
        $midia = Midia::factory()->create(['position' => 3]);
        
        $this->assertGreaterThanOrEqual(1, $midia->position);
        $this->assertLessThanOrEqual(6, $midia->position);
    }

    public function test_midia_type_is_valid()
    {
        $imageMedia = Midia::factory()->create(['type' => 'image']);
        $videoMedia = Midia::factory()->create(['type' => 'video']);
        
        $this->assertEquals('image', $imageMedia->type);
        $this->assertEquals('video', $videoMedia->type);
    }

    public function test_midia_can_have_datetime_range()
    {
        $startDate = now();
        $endDate = now()->addDays(7);
        
        $midia = Midia::factory()->create([
            'start_datetime' => $startDate,
            'end_datetime' => $endDate
        ]);
        
        $this->assertEquals($startDate->format('Y-m-d H:i:s'), $midia->start_datetime->format('Y-m-d H:i:s'));
        $this->assertEquals($endDate->format('Y-m-d H:i:s'), $midia->end_datetime->format('Y-m-d H:i:s'));
    }
}