<?php

/**
 * Script de Testes para Sistema de Registro com Código de Convite
 * Executa validações diretas sem dependência do PHPUnit
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

// Cores para output
class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 60) . "\n"; }
}

$passed = 0;
$failed = 0;

// ═══════════════════════════════════════════════════════════════
TestOutput::title('🧪 TESTES DE VALIDAÇÃO DE ROTAS');

// Teste 1: Rota de registro existe
try {
    if (\Illuminate\Support\Facades\Route::has('register')) {
        TestOutput::success('Rota GET /register existe');
        $passed++;
    } else {
        TestOutput::error('Rota GET /register não encontrada');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar rota: ' . $e->getMessage());
    $failed++;
}

// Teste 2: Rotas de validação e registro POST existem
try {
    $routes = \Illuminate\Support\Facades\Route::getRoutes();
    $hasValidateCode = false;
    $hasRegisterPost = false;
    
    foreach ($routes as $route) {
        if ($route->uri() === 'register/validate-code' && in_array('POST', $route->methods())) {
            $hasValidateCode = true;
        }
        if ($route->uri() === 'register' && in_array('POST', $route->methods())) {
            $hasRegisterPost = true;
        }
    }
    
    if ($hasValidateCode) {
        TestOutput::success('Rota POST /register/validate-code existe');
        $passed++;
    } else {
        TestOutput::error('Rota POST /register/validate-code não encontrada');
        $failed++;
    }
    
    if ($hasRegisterPost) {
        TestOutput::success('Rota POST /register existe');
        $passed++;
    } else {
        TestOutput::error('Rota POST /register não encontrada');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar rotas POST: ' . $e->getMessage());
    $failed += 2;
}

// ═══════════════════════════════════════════════════════════════
TestOutput::title('📊 TESTES DE ESTRUTURA DE BANCO DE DADOS');

// Teste 3: Tabela users tem campos necessários
try {
    $columns = \Illuminate\Support\Facades\Schema::getColumnListing('users');
    $requiredColumns = ['status_cadastro', 'empresa_id', 'aprovado_por', 'aprovado_em'];
    $missing = [];
    
    foreach ($requiredColumns as $col) {
        if (!in_array($col, $columns)) {
            $missing[] = $col;
        }
    }
    
    if (empty($missing)) {
        TestOutput::success('Tabela users tem todos os campos necessários: ' . implode(', ', $requiredColumns));
        $passed++;
    } else {
        TestOutput::error('Tabela users está faltando campos: ' . implode(', ', $missing));
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar tabela users: ' . $e->getMessage());
    $failed++;
}

// Teste 4: Tabela empresas tem campo codigo_convite
try {
    $columns = \Illuminate\Support\Facades\Schema::getColumnListing('empresas');
    
    if (in_array('codigo_convite', $columns)) {
        TestOutput::success('Tabela empresas tem campo codigo_convite');
        $passed++;
    } else {
        TestOutput::error('Tabela empresas não tem campo codigo_convite');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar tabela empresas: ' . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════
TestOutput::title('🔍 TESTES DE DADOS E VALIDAÇÃO');

// Teste 5: Existem franquias com código de convite
try {
    $franquiasCount = \App\Models\Empresa::where('tipo_empresa', 'franquia')
        ->whereNotNull('codigo_convite')
        ->count();
    
    if ($franquiasCount > 0) {
        TestOutput::success("Existem {$franquiasCount} franquias com código de convite");
        $passed++;
    } else {
        TestOutput::error('Nenhuma franquia com código de convite encontrada');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao buscar franquias: ' . $e->getMessage());
    $failed++;
}

// Teste 6: Código de convite tem 10 caracteres
try {
    $franquia = \App\Models\Empresa::where('tipo_empresa', 'franquia')
        ->whereNotNull('codigo_convite')
        ->first();
    
    if ($franquia && strlen($franquia->codigo_convite) === 10) {
        TestOutput::success("Código de convite tem 10 caracteres: {$franquia->codigo_convite}");
        $passed++;
    } else {
        $len = $franquia ? strlen($franquia->codigo_convite) : 0;
        TestOutput::error("Código de convite não tem 10 caracteres (tem {$len})");
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar tamanho do código: ' . $e->getMessage());
    $failed++;
}

// Teste 7: Código de convite é único
try {
    $totalCodigos = \App\Models\Empresa::whereNotNull('codigo_convite')->count();
    $codigosUnicos = \App\Models\Empresa::whereNotNull('codigo_convite')
        ->distinct('codigo_convite')
        ->count('codigo_convite');
    
    if ($totalCodigos === $codigosUnicos) {
        TestOutput::success("Todos os {$totalCodigos} códigos de convite são únicos");
        $passed++;
    } else {
        TestOutput::error("Códigos duplicados encontrados: {$totalCodigos} total, {$codigosUnicos} únicos");
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar unicidade: ' . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════
TestOutput::title('🔒 TESTES DE SEGURANÇA');

// Teste 8: Senhas são hasheadas
try {
    $user = \App\Models\User::first();
    
    if ($user && str_starts_with($user->password, '$2y$')) {
        TestOutput::success('Senhas são hasheadas com bcrypt (começam com $2y$)');
        $passed++;
    } else {
        TestOutput::error('Senha não está no formato bcrypt correto');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar hash de senha: ' . $e->getMessage());
    $failed++;
}

// Teste 9: UserFactory existe e funciona
try {
    $testUser = \App\Models\User::factory()->make();
    
    if ($testUser && $testUser->name && $testUser->email) {
        TestOutput::success('UserFactory cria usuários corretamente');
        $passed++;
    } else {
        TestOutput::error('UserFactory não criou usuário com dados válidos');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao testar UserFactory: ' . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════
TestOutput::title('🧬 TESTES DE LÓGICA DE NEGÓCIO');

// Teste 10: Validação de código - franquia ativa vs inativa
try {
    $franquiaAtiva = \App\Models\Empresa::where('tipo_empresa', 'franquia')
        ->where('status', 0)
        ->whereNotNull('codigo_convite')
        ->first();
    
    $franquiaInativa = \App\Models\Empresa::where('tipo_empresa', 'franquia')
        ->where('status', 1)
        ->whereNotNull('codigo_convite')
        ->first();
    
    if ($franquiaAtiva) {
        TestOutput::success("Franquia ativa encontrada: {$franquiaAtiva->name} (código: {$franquiaAtiva->codigo_convite})");
        $passed++;
    } else {
        TestOutput::error('Nenhuma franquia ativa com código encontrada');
        $failed++;
    }
    
    if ($franquiaInativa) {
        TestOutput::success("Franquia inativa encontrada: {$franquiaInativa->name} (código: {$franquiaInativa->codigo_convite})");
        $passed++;
    } else {
        TestOutput::info('Nenhuma franquia inativa encontrada (opcional para testes)');
        // Não conta como falha
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar status de franquias: ' . $e->getMessage());
    $failed++;
}

// Teste 11: Enum status_cadastro tem valores corretos
try {
    // Criar usuário de teste para verificar enum
    $testUser = \App\Models\User::factory()->make();
    $testUser->status_cadastro = 'pendente';
    
    // Verificar se o valor é aceito
    if (in_array($testUser->status_cadastro, ['pendente', 'aprovado', 'rejeitado'])) {
        TestOutput::success('Enum status_cadastro aceita valores corretos (pendente, aprovado, rejeitado)');
        $passed++;
    } else {
        TestOutput::error('Enum status_cadastro não aceita valores esperados');
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao verificar enum: ' . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════
TestOutput::title('📋 RESUMO DOS TESTES');

echo "\n";
TestOutput::info("Total de testes executados: " . ($passed + $failed));
if ($passed > 0) {
    echo "\033[32m✓ Testes passados: {$passed}\033[0m\n";
}
if ($failed > 0) {
    echo "\033[31m✗ Testes falhados: {$failed}\033[0m\n";
}

$percentage = $passed + $failed > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;
echo "\n\033[1m Taxa de sucesso: {$percentage}%\033[0m\n\n";

// ═══════════════════════════════════════════════════════════════
TestOutput::title('📝 EXEMPLOS DE CÓDIGOS DE CONVITE DISPONÍVEIS');

try {
    $franquias = \App\Models\Empresa::where('tipo_empresa', 'franquia')
        ->whereNotNull('codigo_convite')
        ->where('status', 0)
        ->take(5)
        ->get(['name', 'codigo_convite', 'status']);
    
    echo "\nFranquias ativas para teste:\n";
    foreach ($franquias as $f) {
        echo "  • {$f->name}: \033[1;32m{$f->codigo_convite}\033[0m\n";
    }
} catch (Exception $e) {
    TestOutput::error('Erro ao listar franquias: ' . $e->getMessage());
}

echo "\n";

// Retornar código de saída apropriado
exit($failed > 0 ? 1 : 0);
