<?php

/**
 * Teste End-to-End (E2E) Completo - Sistema de Registro
 * Simula fluxo completo de usuário com requisições HTTP reais
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Route;
use App\Models\Empresa;
use App\Models\User;

// Cores
class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function warning($msg) { echo "\033[33m⚠\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 70) . "\n"; }
    public static function section($msg) { echo "\n\033[1;36m{$msg}\033[0m\n" . str_repeat('─', 70) . "\n"; }
}

$passed = 0;
$failed = 0;
$warnings = 0;

echo "\033[1m\n╔════════════════════════════════════════════════════════════════════╗\n";
echo "║         TESTE END-TO-END (E2E) - SISTEMA DE REGISTRO              ║\n";
echo "║           Simulação Completa de Usuário Real                      ║\n";
echo "╚════════════════════════════════════════════════════════════════════╝\033[0m\n";

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 1: PREPARAÇÃO DO AMBIENTE');

TestOutput::section('1.1 - Verificar Dados de Teste');

$franquiaAtiva = Empresa::where('tipo_empresa', 'franquia')
    ->where('status', 0)
    ->whereNotNull('codigo_convite')
    ->first();

if ($franquiaAtiva) {
    TestOutput::success("Franquia ativa encontrada: {$franquiaAtiva->name}");
    TestOutput::info("  → Código: {$franquiaAtiva->codigo_convite}");
    TestOutput::info("  → ID: {$franquiaAtiva->id}");
    $passed++;
} else {
    TestOutput::error("Nenhuma franquia ativa encontrada!");
    $failed++;
    exit(1);
}

$franquiaInativa = Empresa::where('tipo_empresa', 'franquia')
    ->where('status', 1)
    ->whereNotNull('codigo_convite')
    ->first();

if ($franquiaInativa) {
    TestOutput::success("Franquia inativa encontrada: {$franquiaInativa->name}");
    TestOutput::info("  → Código: {$franquiaInativa->codigo_convite}");
    $passed++;
} else {
    TestOutput::warning("Nenhuma franquia inativa (não crítico para testes)");
    $warnings++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 2: TESTE E2E - ACESSO À PÁGINA DE REGISTRO');

TestOutput::section('2.1 - GET /register (Exibir Formulário)');

try {
    // Verificar se a rota existe
    $routeExists = Route::has('register');
    
    if ($routeExists) {
        TestOutput::success("Rota /register encontrada");
        $passed++;
    } else {
        TestOutput::error("Rota /register não encontrada!");
        $failed++;
    }
    
    // Executar o controller diretamente
    $controller = app()->make(\App\Http\Controllers\Auth\RegisterController::class);
    $response = $controller->showForm();
    
    // Verificar resposta
    if ($response instanceof \Illuminate\View\View) {
        TestOutput::success("Controller RegisterController@showForm executado");
        TestOutput::info("  → Retornou view: " . $response->getName());
        $passed++;
        
        // Verificar se a view existe
        if ($response->getName() === 'auth.register') {
            TestOutput::success("View 'auth.register' está correta");
            $passed++;
        } else {
            TestOutput::error("View incorreta retornada: " . $response->getName());
            $failed++;
        }
    } else {
        TestOutput::error("Controller não retornou uma View!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao testar GET /register: " . $e->getMessage());
    TestOutput::info("  → " . $e->getFile() . ":" . $e->getLine());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 3: TESTE E2E - VALIDAÇÃO AJAX DE CÓDIGO');

TestOutput::section('3.1 - POST /register/validate-code (Código Válido)');

try {
    // Simular requisição AJAX POST com código válido
    $request = Request::create('/register/validate-code', 'POST', [
        'codigo' => $franquiaAtiva->codigo_convite
    ]);
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');
    
    $controller = app()->make(\App\Http\Controllers\Auth\RegisterController::class);
    $response = $controller->validateCode($request);
    
    // Verificar status code
    $statusCode = $response->getStatusCode();
    if ($statusCode === 200) {
        TestOutput::success("Status HTTP 200 OK");
        $passed++;
    } else {
        TestOutput::error("Status HTTP incorreto: {$statusCode}");
        $failed++;
    }
    
    // Verificar conteúdo JSON
    $content = json_decode($response->getContent(), true);
    
    if (isset($content['valid']) && $content['valid'] === true) {
        TestOutput::success("Código válido reconhecido corretamente");
        TestOutput::info("  → valid: true");
        $passed++;
    } else {
        TestOutput::error("Código válido foi rejeitado!");
        TestOutput::info("  → Response: " . $response->getContent());
        $failed++;
    }
    
    if (isset($content['empresa'])) {
        TestOutput::success("Dados da empresa retornados");
        TestOutput::info("  → ID: {$content['empresa']['id']}");
        TestOutput::info("  → Nome: {$content['empresa']['name']}");
        TestOutput::info("  → Email: {$content['empresa']['email']}");
        $passed++;
        
        // Verificar se os dados estão corretos
        if ($content['empresa']['id'] == $franquiaAtiva->id) {
            TestOutput::success("ID da empresa está correto");
            $passed++;
        } else {
            TestOutput::error("ID da empresa incorreto!");
            $failed++;
        }
    } else {
        TestOutput::error("Dados da empresa não retornados!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao validar código: " . $e->getMessage());
    TestOutput::info("  → Trace: " . $e->getTraceAsString());
    $failed++;
}

TestOutput::section('3.2 - POST /register/validate-code (Código Inválido)');

try {
    // Usar código com 10 caracteres mas inexistente
    $request = Request::create('/register/validate-code', 'POST', [
        'codigo' => 'XPTO999999'
    ]);
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');
    
    $controller = app()->make(\App\Http\Controllers\Auth\RegisterController::class);
    $response = $controller->validateCode($request);
    
    $content = json_decode($response->getContent(), true);
    
    if (isset($content['valid']) && $content['valid'] === false) {
        TestOutput::success("Código inválido rejeitado corretamente");
        $passed++;
    } else {
        TestOutput::error("Código inválido foi aceito!");
        $failed++;
    }
    
    if (isset($content['message'])) {
        TestOutput::success("Mensagem de erro retornada");
        TestOutput::info("  → Mensagem: {$content['message']}");
        $passed++;
    } else {
        TestOutput::warning("Mensagem de erro não retornada");
        $warnings++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao testar código inválido: " . $e->getMessage());
    TestOutput::info("  → " . $e->getFile() . ":" . $e->getLine());
    $failed++;
}

if ($franquiaInativa) {
    TestOutput::section('3.3 - POST /register/validate-code (Franquia Inativa)');
    
    try {
        $request = Request::create('/register/validate-code', 'POST', [
            'codigo' => $franquiaInativa->codigo_convite
        ]);
        $request->headers->set('X-Requested-With', 'XMLHttpRequest');
        
        $controller = app()->make(\App\Http\Controllers\Auth\RegisterController::class);
        $response = $controller->validateCode($request);
        
        $content = json_decode($response->getContent(), true);
        
        if (isset($content['valid']) && $content['valid'] === false) {
            TestOutput::success("Código de franquia inativa rejeitado");
            $passed++;
        } else {
            TestOutput::error("Código de franquia inativa foi aceito!");
            $failed++;
        }
        
    } catch (Exception $e) {
        TestOutput::error("Erro ao testar franquia inativa: " . $e->getMessage());
        $failed++;
    }
}

TestOutput::section('3.4 - POST /register/validate-code (Case Insensitive)');

try {
    $request = Request::create('/register/validate-code', 'POST', [
        'codigo' => strtolower($franquiaAtiva->codigo_convite)
    ]);
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');
    
    $controller = app()->make(\App\Http\Controllers\Auth\RegisterController::class);
    $response = $controller->validateCode($request);
    
    $content = json_decode($response->getContent(), true);
    
    if (isset($content['valid']) && $content['valid'] === true) {
        TestOutput::success("Conversão case-insensitive funciona");
        TestOutput::info("  → Input: " . strtolower($franquiaAtiva->codigo_convite));
        TestOutput::info("  → Aceito como: {$franquiaAtiva->codigo_convite}");
        $passed++;
    } else {
        TestOutput::error("Código em minúsculo foi rejeitado!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao testar case insensitive: " . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 4: TESTE E2E - REGISTRO COMPLETO');

TestOutput::section('4.1 - POST /register (Registro com Sucesso)');

$testEmail = 'e2e.test.' . time() . '@example.com';
$testPassword = 'senha_teste_123';

DB::beginTransaction();

try {
    // Simular requisição POST completa
    $request = Request::create('/register', 'POST', [
        'codigo_convite' => $franquiaAtiva->codigo_convite,
        'name' => 'Usuário E2E Test',
        'email' => $testEmail,
        'phone' => '(11) 98765-4321',
        'password' => $testPassword,
        'password_confirmation' => $testPassword,
    ]);
    
    TestOutput::info("Dados enviados:");
    TestOutput::info("  → Nome: Usuário E2E Test");
    TestOutput::info("  → Email: {$testEmail}");
    TestOutput::info("  → Código: {$franquiaAtiva->codigo_convite}");
    
    // Validar dados (sem FormRequest para evitar problemas de sessão)
    $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
        'codigo_convite' => [
            'required',
            'size:10',
            \Illuminate\Validation\Rule::exists('empresas', 'codigo_convite')
                ->where('tipo_empresa', 'franquia')
                ->where('status', 0)
                ->whereNull('deleted_at')
        ],
        'name' => 'required|min:3|max:255',
        'email' => 'required|email|unique:users',
        'password' => 'required|min:8|confirmed',
        'phone' => 'nullable|max:20',
    ]);
    
    if ($validator->fails()) {
        TestOutput::error("Validação falhou!");
        foreach ($validator->errors()->all() as $error) {
            TestOutput::info("  → {$error}");
        }
        $failed++;
    } else {
        TestOutput::success("Validação passou");
        $passed++;
        
        // Criar usuário (simulando o que o controller faz)
        $empresa = Empresa::where('codigo_convite', strtoupper($request->codigo_convite))
            ->where('tipo_empresa', 'franquia')
            ->where('status', 0)
            ->firstOrFail();
        
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => bcrypt($request->password),
            'empresa_id' => $empresa->id,
            'status_cadastro' => 'pendente',
        ]);
        
        if ($user->id) {
            TestOutput::success("Usuário criado com sucesso");
            TestOutput::info("  → ID: {$user->id}");
            TestOutput::info("  → Nome: {$user->name}");
            TestOutput::info("  → Email: {$user->email}");
            TestOutput::info("  → Status: {$user->status_cadastro}");
            TestOutput::info("  → Empresa ID: {$user->empresa_id}");
            $passed++;
        } else {
            TestOutput::error("Falha ao criar usuário!");
            $failed++;
        }
        
        // Verificar campos
        if ($user->status_cadastro === 'pendente') {
            TestOutput::success("Status 'pendente' configurado corretamente");
            $passed++;
        } else {
            TestOutput::error("Status incorreto: {$user->status_cadastro}");
            $failed++;
        }
        
        if ($user->empresa_id == $franquiaAtiva->id) {
            TestOutput::success("Empresa associada corretamente");
            $passed++;
        } else {
            TestOutput::error("Empresa incorreta: {$user->empresa_id}");
            $failed++;
        }
        
        // Verificar senha hasheada
        if (str_starts_with($user->password, '$2y$')) {
            TestOutput::success("Senha hasheada com bcrypt");
            TestOutput::info("  → Hash: " . substr($user->password, 0, 30) . "...");
            $passed++;
        } else {
            TestOutput::error("Senha não está hasheada!");
            $failed++;
        }
        
        // Verificar se a senha bate
        if (\Illuminate\Support\Facades\Hash::check($testPassword, $user->password)) {
            TestOutput::success("Senha pode ser verificada corretamente");
            $passed++;
        } else {
            TestOutput::error("Senha não bate com o hash!");
            $failed++;
        }
    }
    
    DB::rollBack();
    TestOutput::info("Transação revertida (dados de teste)");
    
} catch (Exception $e) {
    DB::rollBack();
    TestOutput::error("Erro ao testar registro: " . $e->getMessage());
    TestOutput::info("  → File: " . $e->getFile() . ":" . $e->getLine());
    $failed++;
}

TestOutput::section('4.2 - POST /register (Email Duplicado)');

DB::beginTransaction();

try {
    // Criar usuário primeiro
    $existingEmail = 'duplicado.e2e.' . time() . '@example.com';
    
    User::create([
        'name' => 'Primeiro Usuário',
        'email' => $existingEmail,
        'password' => bcrypt('senha123'),
        'empresa_id' => $franquiaAtiva->id,
        'status_cadastro' => 'pendente',
    ]);
    
    TestOutput::info("Usuário inicial criado com email: {$existingEmail}");
    
    // Tentar criar segundo usuário com mesmo email
    $validator = \Illuminate\Support\Facades\Validator::make([
        'codigo_convite' => $franquiaAtiva->codigo_convite,
        'name' => 'Segundo Usuário',
        'email' => $existingEmail,
        'password' => 'senha123',
        'password_confirmation' => 'senha123',
    ], [
        'email' => 'required|email|unique:users',
    ]);
    
    if ($validator->fails()) {
        TestOutput::success("Email duplicado foi rejeitado pela validação");
        $errors = $validator->errors();
        if ($errors->has('email')) {
            TestOutput::info("  → Erro: " . $errors->first('email'));
            $passed++;
        }
    } else {
        TestOutput::error("Email duplicado passou na validação!");
        $failed++;
    }
    
    DB::rollBack();
    
} catch (Exception $e) {
    DB::rollBack();
    TestOutput::error("Erro ao testar email duplicado: " . $e->getMessage());
    $failed++;
}

TestOutput::section('4.3 - POST /register (Validações de Campos)');

// Teste 1: Nome muito curto
$validator = \Illuminate\Support\Facades\Validator::make([
    'name' => 'AB',
], ['name' => 'required|min:3']);

if ($validator->fails()) {
    TestOutput::success("Nome com 2 caracteres foi rejeitado (mínimo 3)");
    $passed++;
} else {
    TestOutput::error("Nome muito curto passou na validação!");
    $failed++;
}

// Teste 2: Senha muito curta
$validator = \Illuminate\Support\Facades\Validator::make([
    'password' => '1234567',
], ['password' => 'required|min:8']);

if ($validator->fails()) {
    TestOutput::success("Senha com 7 caracteres foi rejeitada (mínimo 8)");
    $passed++;
} else {
    TestOutput::error("Senha muito curta passou na validação!");
    $failed++;
}

// Teste 3: Senhas não coincidem
$validator = \Illuminate\Support\Facades\Validator::make([
    'password' => 'senha12345',
    'password_confirmation' => 'senha54321',
], ['password' => 'required|confirmed']);

if ($validator->fails()) {
    TestOutput::success("Senhas diferentes foram rejeitadas");
    $passed++;
} else {
    TestOutput::error("Senhas diferentes passaram na validação!");
    $failed++;
}

// Teste 4: Email inválido
$validator = \Illuminate\Support\Facades\Validator::make([
    'email' => 'email-invalido',
], ['email' => 'required|email']);

if ($validator->fails()) {
    TestOutput::success("Email inválido foi rejeitado");
    $passed++;
} else {
    TestOutput::error("Email inválido passou na validação!");
    $failed++;
}

// Teste 5: Código com tamanho diferente de 10
$validator = \Illuminate\Support\Facades\Validator::make([
    'codigo_convite' => 'ABC123',
], ['codigo_convite' => 'required|size:10']);

if ($validator->fails()) {
    TestOutput::success("Código com 6 caracteres foi rejeitado (obrigatório 10)");
    $passed++;
} else {
    TestOutput::error("Código com tamanho inválido passou na validação!");
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 5: TESTE DE SEGURANÇA E INTEGRIDADE');

TestOutput::section('5.1 - Transação de Banco de Dados');

$initialCount = User::count();

DB::beginTransaction();

try {
    $user = User::create([
        'name' => 'Teste Transação',
        'email' => 'transacao.' . time() . '@test.com',
        'password' => bcrypt('senha123'),
        'empresa_id' => $franquiaAtiva->id,
        'status_cadastro' => 'pendente',
    ]);
    
    $countAfterInsert = User::count();
    
    if ($countAfterInsert > $initialCount) {
        TestOutput::success("Usuário inserido na transação");
        TestOutput::info("  → Antes: {$initialCount} usuários");
        TestOutput::info("  → Depois: {$countAfterInsert} usuários");
        $passed++;
    } else {
        TestOutput::error("Usuário não foi inserido!");
        $failed++;
    }
    
    DB::rollBack();
    
    $countAfterRollback = User::count();
    
    if ($countAfterRollback === $initialCount) {
        TestOutput::success("Rollback funcionou corretamente");
        TestOutput::info("  → Após rollback: {$countAfterRollback} usuários");
        $passed++;
    } else {
        TestOutput::error("Rollback não funcionou!");
        TestOutput::error("  → Esperado: {$initialCount}, Atual: {$countAfterRollback}");
        $failed++;
    }
    
} catch (Exception $e) {
    DB::rollBack();
    TestOutput::error("Erro ao testar transação: " . $e->getMessage());
    $failed++;
}

TestOutput::section('5.2 - Segurança de Senhas');

$samples = User::take(5)->get();
$allSecure = true;

foreach ($samples as $user) {
    if (!str_starts_with($user->password, '$2y$')) {
        TestOutput::error("Usuário ID {$user->id} com senha não hasheada!");
        $allSecure = false;
        $failed++;
    }
}

if ($allSecure && $samples->count() > 0) {
    TestOutput::success("Todas as senhas verificadas estão hasheadas com bcrypt");
    TestOutput::info("  → Verificados: {$samples->count()} usuários");
    $passed++;
}

// ═══════════════════════════════════════════════════════════════════════
echo "\n\033[1m╔════════════════════════════════════════════════════════════════════╗\n";
echo "║                      RESUMO DOS TESTES E2E                         ║\n";
echo "╚════════════════════════════════════════════════════════════════════╝\033[0m\n\n";

echo "Total de testes executados: \033[1m" . ($passed + $failed) . "\033[0m\n";
echo "\033[32m✓ Testes passados: {$passed}\033[0m\n";

if ($failed > 0) {
    echo "\033[31m✗ Testes falhados: {$failed}\033[0m\n";
}

if ($warnings > 0) {
    echo "\033[33m⚠ Avisos: {$warnings}\033[0m\n";
}

$percentage = ($passed + $failed) > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;

echo "\n\033[1mTaxa de sucesso: ";
if ($percentage === 100.0) {
    echo "\033[32m{$percentage}%\033[0m ✓\n";
} elseif ($percentage >= 80) {
    echo "\033[33m{$percentage}%\033[0m ⚠️\n";
} else {
    echo "\033[31m{$percentage}%\033[0m ✗\n";
}

echo "\n\033[36m═══════════════════════════════════════════════════════════════════\033[0m\n";
echo "\033[1mCOBERTURA DO TESTE E2E:\033[0m\n";
echo "  ✓ Rotas (GET /register, POST /register, POST /register/validate-code)\n";
echo "  ✓ Controllers (RegisterController@showForm, validateCode, register)\n";
echo "  ✓ Form Requests (RegisterRequest validations)\n";
echo "  ✓ Models (User, Empresa)\n";
echo "  ✓ Middlewares (guest)\n";
echo "  ✓ Validações (todas as regras)\n";
echo "  ✓ Segurança (bcrypt, transações)\n";
echo "\033[36m═══════════════════════════════════════════════════════════════════\033[0m\n\n";

exit($failed > 0 ? 1 : 0);
