<?php

/**
 * Teste E2E - Registro + Listagem de Usuários
 * Simula fluxo completo: criar usuário → listar usuários → verificar presença
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Models\Empresa;
use App\Models\User;
use App\Http\Controllers\UserController;

// Cores
class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function warning($msg) { echo "\033[33m⚠\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 70) . "\n"; }
    public static function section($msg) { echo "\n\033[1;36m{$msg}\033[0m\n" . str_repeat('─', 70) . "\n"; }
}

$passed = 0;
$failed = 0;

echo "\033[1m\n╔════════════════════════════════════════════════════════════════════╗\n";
echo "║    TESTE E2E - REGISTRO + LISTAGEM DE USUÁRIOS                    ║\n";
echo "║      Simulação: Criar Usuário → Listar → Verificar                ║\n";
echo "╚════════════════════════════════════════════════════════════════════╝\033[0m\n";

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 1: PREPARAÇÃO E CRIAÇÃO DE USUÁRIO DE TESTE');

TestOutput::section('1.1 - Buscar Franquia para Teste');

$franquia = Empresa::where('tipo_empresa', 'franquia')
    ->where('status', 0)
    ->whereNotNull('codigo_convite')
    ->first();

if ($franquia) {
    TestOutput::success("Franquia encontrada: {$franquia->name}");
    TestOutput::info("  → Código: {$franquia->codigo_convite}");
    $passed++;
} else {
    TestOutput::error("Nenhuma franquia encontrada!");
    $failed++;
    exit(1);
}

TestOutput::section('1.2 - Criar Usuário de Teste');

$testEmail = 'usuario.listagem.' . time() . '@test.com';
$testUser = null;

DB::beginTransaction();

try {
    $testUser = User::create([
        'name' => 'Usuário Teste Listagem',
        'email' => $testEmail,
        'password' => bcrypt('senha12345'),
        'empresa_id' => $franquia->id,
        'status_cadastro' => 'pendente',
    ]);
    
    if ($testUser->id) {
        TestOutput::success("Usuário criado para teste");
        TestOutput::info("  → ID: {$testUser->id}");
        TestOutput::info("  → Nome: {$testUser->name}");
        TestOutput::info("  → Email: {$testUser->email}");
        TestOutput::info("  → Status: {$testUser->status_cadastro}");
        $passed++;
    } else {
        TestOutput::error("Falha ao criar usuário!");
        $failed++;
        DB::rollBack();
        exit(1);
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao criar usuário: " . $e->getMessage());
    $failed++;
    DB::rollBack();
    exit(1);
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 2: TESTE DE LISTAGEM DE USUÁRIOS');

TestOutput::section('2.1 - Verificar Rota de Listagem');

try {
    if (\Illuminate\Support\Facades\Route::has('users.index')) {
        TestOutput::success("Rota 'users.index' encontrada");
        $passed++;
    } else {
        TestOutput::error("Rota 'users.index' não encontrada!");
        $failed++;
    }
} catch (Exception $e) {
    TestOutput::error("Erro ao verificar rota: " . $e->getMessage());
    $failed++;
}

TestOutput::section('2.2 - Executar Controller de Listagem');

try {
    // Simular requisição GET /users
    $request = Request::create('/users', 'GET');
    
    $controller = app()->make(UserController::class);
    $response = $controller->index($request);
    
    // Verificar tipo de resposta
    if ($response instanceof \Illuminate\View\View) {
        TestOutput::success("Controller UserController@index executado");
        TestOutput::info("  → Tipo: View");
        TestOutput::info("  → View: " . $response->getName());
        $passed++;
    } elseif ($response instanceof \Illuminate\Http\JsonResponse) {
        TestOutput::success("Controller UserController@index executado");
        TestOutput::info("  → Tipo: JSON Response");
        $passed++;
    } else {
        TestOutput::warning("Controller retornou tipo inesperado: " . get_class($response));
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao executar controller: " . $e->getMessage());
    TestOutput::info("  → " . $e->getFile() . ":" . $e->getLine());
    $failed++;
}

TestOutput::section('2.3 - Listar Todos os Usuários Diretamente do Banco');

try {
    $users = User::with('empresa')->get();
    
    TestOutput::success("Usuários listados do banco de dados");
    TestOutput::info("  → Total de usuários: {$users->count()}");
    $passed++;
    
    // Verificar se existem usuários
    if ($users->count() > 0) {
        TestOutput::success("Banco de dados contém usuários");
        $passed++;
    } else {
        TestOutput::warning("Nenhum usuário encontrado no banco!");
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao listar usuários: " . $e->getMessage());
    $failed++;
}

TestOutput::section('2.4 - Verificar Presença do Usuário de Teste na Listagem');

try {
    $foundUser = User::where('email', $testEmail)->first();
    
    if ($foundUser) {
        TestOutput::success("Usuário de teste encontrado na listagem");
        TestOutput::info("  → ID: {$foundUser->id}");
        TestOutput::info("  → Nome: {$foundUser->name}");
        TestOutput::info("  → Email: {$foundUser->email}");
        $passed++;
        
        // Verificar dados do usuário
        if ($foundUser->id === $testUser->id) {
            TestOutput::success("ID do usuário está correto");
            $passed++;
        } else {
            TestOutput::error("ID do usuário incorreto!");
            $failed++;
        }
        
        if ($foundUser->name === $testUser->name) {
            TestOutput::success("Nome do usuário está correto");
            $passed++;
        } else {
            TestOutput::error("Nome do usuário incorreto!");
            $failed++;
        }
        
        if ($foundUser->email === $testUser->email) {
            TestOutput::success("Email do usuário está correto");
            $passed++;
        } else {
            TestOutput::error("Email do usuário incorreto!");
            $failed++;
        }
        
    } else {
        TestOutput::error("Usuário de teste NÃO encontrado na listagem!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao buscar usuário: " . $e->getMessage());
    $failed++;
}

TestOutput::section('2.5 - Verificar Relacionamento com Empresa');

try {
    $user = User::with('empresa')->find($testUser->id);
    
    if ($user && $user->empresa) {
        TestOutput::success("Relacionamento User → Empresa funciona");
        TestOutput::info("  → Empresa ID: {$user->empresa->id}");
        TestOutput::info("  → Empresa Nome: {$user->empresa->name}");
        TestOutput::info("  → Empresa Tipo: {$user->empresa->tipo_empresa}");
        $passed++;
        
        // Verificar se é a empresa correta
        if ($user->empresa->id === $franquia->id) {
            TestOutput::success("Empresa associada está correta");
            $passed++;
        } else {
            TestOutput::error("Empresa associada incorreta!");
            $failed++;
        }
        
    } else {
        TestOutput::error("Relacionamento não carregou a empresa!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao verificar relacionamento: " . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 3: TESTES DE FILTROS E PAGINAÇÃO');

TestOutput::section('3.1 - Listar Usuários com Status Pendente');

try {
    $pendentes = User::where('status_cadastro', 'pendente')->get();
    
    TestOutput::success("Filtro por status funcionando");
    TestOutput::info("  → Usuários pendentes: {$pendentes->count()}");
    $passed++;
    
    // Verificar se nosso usuário está na lista
    $userInList = $pendentes->contains('id', $testUser->id);
    
    if ($userInList) {
        TestOutput::success("Usuário de teste está na lista de pendentes");
        $passed++;
    } else {
        TestOutput::error("Usuário de teste não está na lista de pendentes!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao filtrar por status: " . $e->getMessage());
    $failed++;
}

TestOutput::section('3.2 - Listar Usuários por Empresa');

try {
    $usersDaEmpresa = User::where('empresa_id', $franquia->id)->get();
    
    TestOutput::success("Filtro por empresa funcionando");
    TestOutput::info("  → Usuários da empresa '{$franquia->name}': {$usersDaEmpresa->count()}");
    $passed++;
    
    // Verificar se nosso usuário está na lista
    $userInList = $usersDaEmpresa->contains('id', $testUser->id);
    
    if ($userInList) {
        TestOutput::success("Usuário de teste está na lista da empresa");
        $passed++;
    } else {
        TestOutput::error("Usuário de teste não está na lista da empresa!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao filtrar por empresa: " . $e->getMessage());
    $failed++;
}

TestOutput::section('3.3 - Buscar Usuário por Email');

try {
    $user = User::where('email', $testEmail)->first();
    
    if ($user) {
        TestOutput::success("Busca por email funcionando");
        TestOutput::info("  → Usuário encontrado: {$user->name}");
        $passed++;
    } else {
        TestOutput::error("Busca por email falhou!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao buscar por email: " . $e->getMessage());
    $failed++;
}

TestOutput::section('3.4 - Buscar Usuário por Nome (LIKE)');

try {
    $users = User::where('name', 'like', '%Teste Listagem%')->get();
    
    if ($users->count() > 0) {
        TestOutput::success("Busca por nome (LIKE) funcionando");
        TestOutput::info("  → Usuários encontrados: {$users->count()}");
        $passed++;
        
        $userInList = $users->contains('id', $testUser->id);
        
        if ($userInList) {
            TestOutput::success("Usuário de teste encontrado na busca");
            $passed++;
        } else {
            TestOutput::error("Usuário de teste não encontrado na busca!");
            $failed++;
        }
    } else {
        TestOutput::error("Busca por nome não retornou resultados!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao buscar por nome: " . $e->getMessage());
    $failed++;
}

TestOutput::section('3.5 - Teste de Paginação');

try {
    $perPage = 10;
    $paginated = User::paginate($perPage);
    
    TestOutput::success("Paginação funcionando");
    TestOutput::info("  → Total de usuários: {$paginated->total()}");
    TestOutput::info("  → Por página: {$paginated->perPage()}");
    TestOutput::info("  → Página atual: {$paginated->currentPage()}");
    TestOutput::info("  → Total de páginas: {$paginated->lastPage()}");
    $passed++;
    
    // Verificar estrutura da paginação
    if ($paginated->total() > 0) {
        TestOutput::success("Paginação contém registros");
        $passed++;
    } else {
        TestOutput::warning("Paginação está vazia");
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao testar paginação: " . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 4: TESTES DE ORDENAÇÃO');

TestOutput::section('4.1 - Ordenar por Nome (A-Z)');

try {
    $users = User::orderBy('name', 'asc')->take(5)->get();
    
    TestOutput::success("Ordenação por nome (ASC) funcionando");
    TestOutput::info("  → Primeiros 5 usuários:");
    
    foreach ($users as $user) {
        TestOutput::info("      • {$user->name}");
    }
    
    $passed++;
    
} catch (Exception $e) {
    TestOutput::error("Erro ao ordenar por nome: " . $e->getMessage());
    $failed++;
}

TestOutput::section('4.2 - Ordenar por Data de Criação (Mais Recentes)');

try {
    $users = User::orderBy('created_at', 'desc')->take(3)->get();
    
    TestOutput::success("Ordenação por data (DESC) funcionando");
    TestOutput::info("  → 3 usuários mais recentes:");
    
    foreach ($users as $user) {
        TestOutput::info("      • {$user->name} ({$user->created_at->format('d/m/Y H:i')})");
    }
    
    // Verificar se nosso usuário está entre os mais recentes
    $userInList = $users->contains('id', $testUser->id);
    
    if ($userInList) {
        TestOutput::success("Usuário de teste está entre os mais recentes");
        $passed++;
    } else {
        TestOutput::info("Usuário de teste não está entre os 3 mais recentes (normal)");
    }
    
    $passed++;
    
} catch (Exception $e) {
    TestOutput::error("Erro ao ordenar por data: " . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 5: TESTES DE CONTAGEM E AGREGAÇÃO');

TestOutput::section('5.1 - Contar Total de Usuários');

try {
    $total = User::count();
    
    TestOutput::success("Contagem de usuários funcionando");
    TestOutput::info("  → Total: {$total} usuários");
    $passed++;
    
} catch (Exception $e) {
    TestOutput::error("Erro ao contar usuários: " . $e->getMessage());
    $failed++;
}

TestOutput::section('5.2 - Contar Usuários por Status');

try {
    $pendentes = User::where('status_cadastro', 'pendente')->count();
    $aprovados = User::where('status_cadastro', 'aprovado')->count();
    $rejeitados = User::where('status_cadastro', 'rejeitado')->count();
    
    TestOutput::success("Contagem por status funcionando");
    TestOutput::info("  → Pendentes: {$pendentes}");
    TestOutput::info("  → Aprovados: {$aprovados}");
    TestOutput::info("  → Rejeitados: {$rejeitados}");
    TestOutput::info("  → Total: " . ($pendentes + $aprovados + $rejeitados));
    $passed++;
    
} catch (Exception $e) {
    TestOutput::error("Erro ao contar por status: " . $e->getMessage());
    $failed++;
}

TestOutput::section('5.3 - Contar Usuários por Empresa');

try {
    $empresas = Empresa::withCount('users')->get();
    
    TestOutput::success("Contagem de usuários por empresa funcionando");
    TestOutput::info("  → Empresas com usuários:");
    
    $count = 0;
    foreach ($empresas as $empresa) {
        if ($empresa->users_count > 0) {
            TestOutput::info("      • {$empresa->name}: {$empresa->users_count} usuários");
            $count++;
        }
    }
    
    if ($count > 0) {
        TestOutput::success("Encontradas {$count} empresas com usuários");
        $passed++;
    } else {
        TestOutput::warning("Nenhuma empresa com usuários");
    }
    
    $passed++;
    
} catch (Exception $e) {
    TestOutput::error("Erro ao contar por empresa: " . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 6: LIMPEZA');

TestOutput::section('6.1 - Remover Usuário de Teste');

try {
    DB::rollBack(); // Reverter toda a transação
    
    TestOutput::success("Transação revertida - usuário de teste removido");
    TestOutput::info("  → Dados de teste limpos");
    $passed++;
    
    // Verificar se foi realmente removido
    $userExists = User::where('email', $testEmail)->exists();
    
    if (!$userExists) {
        TestOutput::success("Confirmado: usuário não existe mais no banco");
        $passed++;
    } else {
        TestOutput::error("Usuário ainda existe no banco!");
        $failed++;
    }
    
} catch (Exception $e) {
    TestOutput::error("Erro ao limpar dados: " . $e->getMessage());
    $failed++;
}

// ═══════════════════════════════════════════════════════════════════════
echo "\n\033[1m╔════════════════════════════════════════════════════════════════════╗\n";
echo "║                 RESUMO - TESTE DE LISTAGEM E2E                    ║\n";
echo "╚════════════════════════════════════════════════════════════════════╝\033[0m\n\n";

echo "Total de testes executados: \033[1m" . ($passed + $failed) . "\033[0m\n";
echo "\033[32m✓ Testes passados: {$passed}\033[0m\n";

if ($failed > 0) {
    echo "\033[31m✗ Testes falhados: {$failed}\033[0m\n";
}

$percentage = ($passed + $failed) > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;

echo "\n\033[1mTaxa de sucesso: ";
if ($percentage === 100.0) {
    echo "\033[32m{$percentage}%\033[0m ✓\n";
} elseif ($percentage >= 80) {
    echo "\033[33m{$percentage}%\033[0m ⚠️\n";
} else {
    echo "\033[31m{$percentage}%\033[0m ✗\n";
}

echo "\n\033[36m═══════════════════════════════════════════════════════════════════\033[0m\n";
echo "\033[1mCOBERTURA DO TESTE:\033[0m\n";
echo "  ✓ Criação de usuário\n";
echo "  ✓ Listagem de usuários (GET /users)\n";
echo "  ✓ Verificação de presença na listagem\n";
echo "  ✓ Relacionamento User → Empresa\n";
echo "  ✓ Filtros (status, empresa, email, nome)\n";
echo "  ✓ Paginação\n";
echo "  ✓ Ordenação (nome, data)\n";
echo "  ✓ Contagem e agregação\n";
echo "  ✓ Limpeza de dados de teste\n";
echo "\033[36m═══════════════════════════════════════════════════════════════════\033[0m\n\n";

exit($failed > 0 ? 1 : 0);
