<?php

/**
 * Teste de Integração - Fluxo Completo de Registro com Código de Convite
 * Simula requisições HTTP para testar todo o sistema
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Http\Controllers\Auth\RegisterController;
use App\Models\Empresa;
use App\Models\User;

// Cores
class TestColor {
    public static function green($text) { return "\033[32m{$text}\033[0m"; }
    public static function red($text) { return "\033[31m{$text}\033[0m"; }
    public static function yellow($text) { return "\033[33m{$text}\033[0m"; }
    public static function blue($text) { return "\033[36m{$text}\033[0m"; }
    public static function bold($text) { return "\033[1m{$text}\033[0m"; }
}

$passed = 0;
$failed = 0;
$total = 0;

function testCase($name, $callback) {
    global $passed, $failed, $total;
    $total++;
    
    echo "\n" . TestColor::blue("🧪 Teste {$total}: {$name}");
    
    try {
        $result = $callback();
        
        if ($result === true || $result === null) {
            echo " " . TestColor::green("✓ PASSOU") . "\n";
            $passed++;
            return true;
        } else {
            echo " " . TestColor::red("✗ FALHOU") . "\n";
            if (is_string($result)) {
                echo "   Motivo: {$result}\n";
            }
            $failed++;
            return false;
        }
    } catch (Exception $e) {
        echo " " . TestColor::red("✗ ERRO") . "\n";
        echo "   " . $e->getMessage() . "\n";
        $failed++;
        return false;
    }
}

echo TestColor::bold("\n╔═══════════════════════════════════════════════════════════════╗\n");
echo TestColor::bold("║  TESTE DE INTEGRAÇÃO - REGISTRO COM CÓDIGO DE CONVITE        ║\n");
echo TestColor::bold("╚═══════════════════════════════════════════════════════════════╝\n");

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::yellow("═══ FASE 1: PREPARAÇÃO DE DADOS ═══") . "\n";

$franquiaAtiva = null;
$franquiaInativa = null;
$codigoValido = null;
$codigoInvalido = 'XPTO999999';

testCase("Buscar franquia ativa para teste", function() use (&$franquiaAtiva, &$codigoValido) {
    $franquiaAtiva = Empresa::where('tipo_empresa', 'franquia')
        ->where('status', 0)
        ->whereNotNull('codigo_convite')
        ->first();
    
    if ($franquiaAtiva) {
        $codigoValido = $franquiaAtiva->codigo_convite;
        echo "   Código válido: " . TestColor::green($codigoValido) . " ({$franquiaAtiva->name})\n";
        return true;
    }
    
    return "Nenhuma franquia ativa encontrada";
});

testCase("Buscar franquia inativa para teste", function() use (&$franquiaInativa) {
    $franquiaInativa = Empresa::where('tipo_empresa', 'franquia')
        ->where('status', 1)
        ->whereNotNull('codigo_convite')
        ->first();
    
    if ($franquiaInativa) {
        echo "   Código inativo: " . TestColor::yellow($franquiaInativa->codigo_convite) . " ({$franquiaInativa->name})\n";
        return true;
    }
    
    echo "   " . TestColor::blue("(Nenhuma franquia inativa - pulando testes relacionados)") . "\n";
    return true; // Não é erro crítico
});

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::yellow("═══ FASE 2: TESTES DE VALIDAÇÃO DE CÓDIGO (AJAX) ═══") . "\n";

testCase("Validar código existente e ativo", function() use ($codigoValido) {
    $request = Request::create('/register/validate-code', 'POST', [
        'codigo' => $codigoValido
    ]);
    
    $controller = new RegisterController();
    $response = $controller->validateCode($request);
    $data = json_decode($response->getContent(), true);
    
    if (!isset($data['valid']) || $data['valid'] !== true) {
        return "Código válido foi rejeitado";
    }
    
    if (!isset($data['empresa']['id']) || !isset($data['empresa']['name'])) {
        return "Dados da empresa não retornados corretamente";
    }
    
    echo "   Empresa retornada: {$data['empresa']['name']}\n";
    return true;
});

testCase("Rejeitar código inexistente", function() use ($codigoInvalido) {
    $request = Request::create('/register/validate-code', 'POST', [
        'codigo' => $codigoInvalido
    ]);
    
    $controller = new RegisterController();
    $response = $controller->validateCode($request);
    $data = json_decode($response->getContent(), true);
    
    if ($data['valid'] === true) {
        return "Código inexistente foi aceito incorretamente";
    }
    
    if (!isset($data['message'])) {
        return "Mensagem de erro não retornada";
    }
    
    echo "   Mensagem: {$data['message']}\n";
    return true;
});

if ($franquiaInativa) {
    testCase("Rejeitar código de franquia inativa", function() use ($franquiaInativa) {
        $request = Request::create('/register/validate-code', 'POST', [
            'codigo' => $franquiaInativa->codigo_convite
        ]);
        
        $controller = new RegisterController();
        $response = $controller->validateCode($request);
        $data = json_decode($response->getContent(), true);
        
        if ($data['valid'] === true) {
            return "Código de franquia inativa foi aceito incorretamente";
        }
        
        return true;
    });
}

testCase("Validação converte código para maiúsculo", function() use ($codigoValido) {
    $request = Request::create('/register/validate-code', 'POST', [
        'codigo' => strtolower($codigoValido)
    ]);
    
    $controller = new RegisterController();
    $response = $controller->validateCode($request);
    $data = json_decode($response->getContent(), true);
    
    if ($data['valid'] !== true) {
        return "Código em minúsculo não foi convertido corretamente";
    }
    
    echo "   Testado com: " . strtolower($codigoValido) . " → aceito\n";
    return true;
});

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::yellow("═══ FASE 3: TESTES DE REGISTRO COMPLETO ═══") . "\n";

$testEmail = 'teste_' . time() . '@example.com';
$testUser = null;

testCase("Registrar usuário com código válido", function() use ($codigoValido, $testEmail, &$testUser, $franquiaAtiva) {
    // Usar transação de teste para não poluir o banco
    DB::beginTransaction();
    
    try {
        $userData = [
            'codigo_convite' => $codigoValido,
            'name' => 'Usuário de Teste',
            'email' => $testEmail,
            'phone' => '11999999999',
            'password' => 'senha12345',
            'password_confirmation' => 'senha12345',
        ];
        
        // Simular requisição POST
        $request = Request::create('/register', 'POST', $userData);
        
        // Criar usuário manualmente (simulando o que o controller faz)
        $testUser = User::create([
            'name' => $userData['name'],
            'email' => $userData['email'],
            'phone' => $userData['phone'],
            'password' => bcrypt($userData['password']),
            'empresa_id' => $franquiaAtiva->id,
            'status_cadastro' => 'pendente',
        ]);
        
        if (!$testUser->id) {
            DB::rollBack();
            return "Usuário não foi criado";
        }
        
        if ($testUser->status_cadastro !== 'pendente') {
            DB::rollBack();
            return "Status de cadastro incorreto: {$testUser->status_cadastro}";
        }
        
        if ($testUser->empresa_id != $franquiaAtiva->id) {
            DB::rollBack();
            return "Empresa não associada corretamente";
        }
        
        echo "   Usuário criado: ID {$testUser->id}, Status: {$testUser->status_cadastro}\n";
        
        DB::rollBack(); // Limpar teste
        return true;
        
    } catch (Exception $e) {
        DB::rollBack();
        throw $e;
    }
});

testCase("Senha é hasheada corretamente", function() use ($testEmail) {
    DB::beginTransaction();
    
    try {
        $password = 'senha_teste_123';
        
        $testUser = User::create([
            'name' => 'Teste Hash',
            'email' => 'hash_' . time() . '@test.com',
            'password' => bcrypt($password),
            'empresa_id' => 1,
            'status_cadastro' => 'pendente',
        ]);
        
        if (!str_starts_with($testUser->password, '$2y$')) {
            DB::rollBack();
            return "Senha não está no formato bcrypt";
        }
        
        if ($testUser->password === $password) {
            DB::rollBack();
            return "Senha armazenada em texto plano!";
        }
        
        if (strlen($testUser->password) < 30) {
            DB::rollBack();
            return "Hash de senha muito curto";
        }
        
        echo "   Hash gerado: {$testUser->password}\n";
        
        DB::rollBack();
        return true;
        
    } catch (Exception $e) {
        DB::rollBack();
        throw $e;
    }
});

testCase("Email duplicado é rejeitado", function() {
    DB::beginTransaction();
    
    try {
        $email = 'duplicado_' . time() . '@test.com';
        
        // Criar primeiro usuário
        User::create([
            'name' => 'Primeiro Usuário',
            'email' => $email,
            'password' => bcrypt('senha123'),
            'empresa_id' => 1,
            'status_cadastro' => 'pendente',
        ]);
        
        // Tentar criar segundo com mesmo email
        try {
            User::create([
                'name' => 'Segundo Usuário',
                'email' => $email,
                'password' => bcrypt('senha123'),
                'empresa_id' => 1,
                'status_cadastro' => 'pendente',
            ]);
            
            DB::rollBack();
            return "Email duplicado foi aceito incorretamente";
            
        } catch (\Illuminate\Database\QueryException $e) {
            // Esperado - violação de unique constraint
            DB::rollBack();
            echo "   " . TestColor::green("Violação de constraint detectada corretamente") . "\n";
            return true;
        }
        
    } catch (Exception $e) {
        DB::rollBack();
        throw $e;
    }
});

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::yellow("═══ FASE 4: TESTES DE SEGURANÇA ═══") . "\n";

testCase("Verificar que senhas nunca são armazenadas em texto plano", function() {
    $users = User::take(10)->get();
    
    foreach ($users as $user) {
        if (!str_starts_with($user->password, '$2y$')) {
            return "Usuário {$user->id} tem senha não hasheada";
        }
    }
    
    echo "   Verificados {$users->count()} usuários - todos com hash bcrypt\n";
    return true;
});

testCase("Transação protege integridade dos dados", function() use ($codigoValido, $franquiaAtiva) {
    $initialCount = User::count();
    
    DB::beginTransaction();
    
    try {
        // Criar usuário
        User::create([
            'name' => 'Teste Transação',
            'email' => 'transacao_' . time() . '@test.com',
            'password' => bcrypt('senha123'),
            'empresa_id' => $franquiaAtiva->id,
            'status_cadastro' => 'pendente',
        ]);
        
        $countAfterInsert = User::count();
        
        // Rollback
        DB::rollBack();
        
        $countAfterRollback = User::count();
        
        if ($countAfterInsert <= $initialCount) {
            return "Usuário não foi inserido na transação";
        }
        
        if ($countAfterRollback !== $initialCount) {
            return "Rollback não funcionou - dados persistiram";
        }
        
        echo "   Inicial: {$initialCount} → Após insert: {$countAfterInsert} → Após rollback: {$countAfterRollback}\n";
        return true;
        
    } catch (Exception $e) {
        DB::rollBack();
        throw $e;
    }
});

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::yellow("═══ FASE 5: TESTES DE CASOS EXTREMOS ═══") . "\n";

testCase("Código com tamanho diferente de 10 caracteres", function() {
    try {
        $request = Request::create('/register/validate-code', 'POST', [
            'codigo' => 'ABC123'
        ]);
        
        $controller = new RegisterController();
        $response = $controller->validateCode($request);
        $data = json_decode($response->getContent(), true);
        
        if ($data['valid'] === true) {
            return "Código com tamanho inválido foi aceito";
        }
        
        echo "   " . TestColor::green("Validação rejeitou código curto corretamente") . "\n";
        return true;
        
    } catch (\Illuminate\Validation\ValidationException $e) {
        // Exceção de validação é o comportamento esperado
        echo "   " . TestColor::green("Validação de tamanho funcionou: ") . $e->getMessage() . "\n";
        return true;
    }
});

testCase("Telefone é opcional no registro", function() use ($codigoValido, $franquiaAtiva) {
    DB::beginTransaction();
    
    try {
        $user = User::create([
            'name' => 'Sem Telefone',
            'email' => 'sem_tel_' . time() . '@test.com',
            'password' => bcrypt('senha123'),
            'empresa_id' => $franquiaAtiva->id,
            'status_cadastro' => 'pendente',
            // phone não fornecido
        ]);
        
        if (!$user->id) {
            DB::rollBack();
            return "Não foi possível criar usuário sem telefone";
        }
        
        DB::rollBack();
        return true;
        
    } catch (Exception $e) {
        DB::rollBack();
        throw $e;
    }
});

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::bold("╔═══════════════════════════════════════════════════════════════╗\n");
echo TestColor::bold("║  RESUMO FINAL DOS TESTES                                      ║\n");
echo TestColor::bold("╚═══════════════════════════════════════════════════════════════╝\n\n");

echo "Total de testes: " . TestColor::bold($total) . "\n";
echo TestColor::green("✓ Passou: {$passed}") . "\n";

if ($failed > 0) {
    echo TestColor::red("✗ Falhou: {$failed}") . "\n";
}

$percentage = $total > 0 ? round(($passed / $total) * 100, 2) : 0;

echo "\n" . TestColor::bold("Taxa de sucesso: ");
if ($percentage === 100.0) {
    echo TestColor::green("{$percentage}%") . "\n";
} elseif ($percentage >= 80) {
    echo TestColor::yellow("{$percentage}%") . "\n";
} else {
    echo TestColor::red("{$percentage}%") . "\n";
}

// ═══════════════════════════════════════════════════════════════
echo "\n" . TestColor::yellow("═══ CÓDIGOS DE CONVITE DISPONÍVEIS PARA TESTES MANUAIS ═══") . "\n\n";

$franquias = Empresa::where('tipo_empresa', 'franquia')
    ->whereNotNull('codigo_convite')
    ->where('status', 0)
    ->get(['name', 'codigo_convite', 'status']);

foreach ($franquias as $f) {
    echo "  🎫 " . TestColor::green($f->codigo_convite) . " - {$f->name}\n";
}

echo "\n" . TestColor::blue("💡 Acesse: http://localhost:8082/register para testar manualmente") . "\n\n";

exit($failed > 0 ? 1 : 0);
