<?php

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\Empresa;
use App\Models\Midia;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║                                                                ║\n";
echo "║   🧪 TESTE DE UPLOAD DE CRIATIVO COM MD5                      ║\n";
echo "║                                                                ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

// 1. Buscar empresa para o teste
echo "1️⃣  BUSCANDO EMPRESA PARA O TESTE...\n";
$empresa = Empresa::first();

if (!$empresa) {
    echo "❌ Nenhuma empresa encontrada!\n";
    exit(1);
}

echo "   ✅ Empresa: {$empresa->name}\n";
echo "   📧 Email: {$empresa->email}\n";
echo "   🔑 Código: {$empresa->codigo_convite}\n\n";

// 2. Ler arquivo de teste
echo "2️⃣  LENDO ARQUIVO DE TESTE...\n";
$testFile = '/tmp/teste_upload.jpg';

if (!file_exists($testFile)) {
    echo "❌ Arquivo de teste não encontrado: {$testFile}\n";
    exit(1);
}

$fileContent = file_get_contents($testFile);
$fileSize = strlen($fileContent);
echo "   ✅ Arquivo lido: " . round($fileSize / 1024, 2) . " KB\n\n";

// 3. Gerar nome criptografado MD5
echo "3️⃣  GERANDO NOME CRIPTOGRAFADO...\n";
$extension = 'jpg';
$uniqueString = uniqid('', true) . microtime(true) . rand(1000, 9999);
$hashedName = md5($uniqueString);
$filename = $hashedName . '.' . $extension;

echo "   🔐 String única: " . substr($uniqueString, 0, 40) . "...\n";
echo "   📏 Hash MD5: {$hashedName}\n";
echo "   📝 Filename: {$filename}\n\n";

// 4. Processar imagem (redimensionar)
echo "4️⃣  PROCESSANDO IMAGEM (640x1100)...\n";
$manager = new ImageManager(new Driver());
$image = $manager->read($fileContent);
$originalWidth = $image->width();
$originalHeight = $image->height();
echo "   📐 Dimensões originais: {$originalWidth}x{$originalHeight}\n";

$image->resize(640, 1100);
echo "   ✅ Redimensionada para: 640x1100\n\n";

// 5. Salvar no storage
echo "5️⃣  SALVANDO NO STORAGE...\n";
$path = 'midias/' . $filename;
Storage::disk('public')->put($path, (string) $image->encode());

$fullPath = storage_path('app/public/' . $path);
$savedSize = file_exists($fullPath) ? filesize($fullPath) : 0;

echo "   ✅ Path relativo: {$path}\n";
echo "   📂 Path absoluto: {$fullPath}\n";
echo "   💾 Tamanho salvo: " . round($savedSize / 1024, 2) . " KB\n";
echo "   🔗 URL pública: http://localhost:8082/storage/{$path}\n\n";

// 6. Criar registro no banco
echo "6️⃣  CRIANDO REGISTRO NO BANCO...\n";
$midia = Midia::create([
    'title' => 'Teste Upload MD5 - ' . date('Y-m-d H:i:s'),
    'tipo' => 'imagem',
    'formato' => 'jpg',
    'path' => $path,
    'empresa_legado_id' => $empresa->id,
    'position' => 1,
    'start_datetime' => now(),
    'end_datetime' => now()->addDays(30),
    'status_aprovacao' => 'aprovado',
]);

echo "   ✅ ID Criativo: {$midia->id}\n";
echo "   📝 Título: {$midia->title}\n";
echo "   📂 Path: {$midia->path}\n";
echo "   🏢 Empresa: {$midia->empresa->name}\n";
echo "   ✅ Status: {$midia->status_aprovacao}\n\n";

// 7. Validações finais
echo "7️⃣  VALIDAÇÕES FINAIS...\n";

$validations = [
    'Campo path preenchido no banco' => !empty($midia->path),
    'Arquivo existe em storage/app/public/midias/' => file_exists($fullPath),
    'Tamanho do arquivo > 0' => $savedSize > 0,
    'Formato correto (jpg)' => $midia->formato === 'jpg',
    'Tipo correto (imagem)' => $midia->tipo === 'imagem',
    'Relacionamento com empresa' => $midia->empresa_legado_id === $empresa->id,
    'Nome arquivo é MD5 (32 chars)' => strlen($hashedName) === 32,
];

$allPassed = true;
foreach ($validations as $test => $passed) {
    $status = $passed ? '✅' : '❌';
    echo "   {$status} {$test}\n";
    if (!$passed) $allPassed = false;
}

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║                                                                ║\n";
if ($allPassed) {
    echo "║   🎉 TODOS OS TESTES PASSARAM! UPLOAD FUNCIONANDO! 🎉         ║\n";
} else {
    echo "║   ❌ ALGUNS TESTES FALHARAM - VERIFICAR ACIMA                 ║\n";
}
echo "║                                                                ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

echo "📊 RESUMO:\n";
echo "   • Criativo ID: {$midia->id}\n";
echo "   • Arquivo: {$filename}\n";
echo "   • Path banco: {$midia->path}\n";
echo "   • URL: http://localhost:8082/storage/{$path}\n\n";

exit($allPassed ? 0 : 1);
