# 🚀 FASE 2 - CADASTRO E APROVAÇÃO DE USUÁRIOS

## 📋 Objetivo

Criar a interface completa para:
1. **Registro de novos usuários** usando código de convite
2. **Dashboard de aprovações** para SuperAdmin
3. **Sistema de notificações** de status
4. **Fluxo completo de onboarding**

---

## 🎯 User Stories

### **US 2.1 - Registro com Código de Convite**
```
Como um usuário novo
Quero me cadastrar usando um código de convite
Para me vincular a uma franquia específica
```

**Critérios de Aceitação:**
- ✅ Tela de registro solicita código de convite (10 caracteres)
- ✅ Valida se código existe e pertence a franquia ativa
- ✅ Mostra nome da franquia após validar código
- ✅ Coleta: nome, email, senha, telefone
- ✅ Cria usuário com `status_cadastro: 'pendente'`
- ✅ Envia email de confirmação
- ✅ Redireciona para tela "Aguardando Aprovação"

---

### **US 2.2 - Dashboard de Aprovações (SuperAdmin)**
```
Como SuperAdmin
Quero visualizar todos os cadastros pendentes
Para aprovar ou rejeitar novos usuários
```

**Critérios de Aceitação:**
- ✅ Lista todos usuários com status 'pendente'
- ✅ Mostra: nome, email, empresa, data de cadastro
- ✅ Badge visual com contagem de pendentes
- ✅ Ação: Aprovar (modal de confirmação)
- ✅ Ação: Rejeitar (modal com campo motivo obrigatório)
- ✅ Registra quem aprovou/rejeitou e quando
- ✅ Envia email notificando o resultado

---

### **US 2.3 - Meu Status de Cadastro**
```
Como um usuário pendente
Quero visualizar o status da minha aprovação
Para saber se já posso acessar o sistema
```

**Critérios de Aceitação:**
- ✅ Tela exibe status atual (pendente/aprovado/rejeitado)
- ✅ Se pendente: mensagem "Aguardando aprovação"
- ✅ Se aprovado: redireciona para dashboard
- ✅ Se rejeitado: exibe motivo da rejeição
- ✅ Opção de reenviar cadastro (se rejeitado)

---

### **US 2.4 - Notificações por Email**
```
Como um usuário
Quero receber emails sobre meu cadastro
Para ser notificado das mudanças de status
```

**Critérios de Aceitação:**
- ✅ Email 1: "Cadastro recebido - Aguardando aprovação"
- ✅ Email 2: "Cadastro aprovado - Bem-vindo!"
- ✅ Email 3: "Cadastro rejeitado - Motivo: ..."
- ✅ Templates Blade personalizados
- ✅ Queue para envio assíncrono

---

## 🏗️ Arquitetura Técnica

### **1. Routes (routes/web.php)**

```php
// Área Pública (sem auth)
Route::middleware('guest')->group(function () {
    // Registro
    Route::get('/register', [RegisterController::class, 'showForm'])
        ->name('register');
    Route::post('/register', [RegisterController::class, 'register'])
        ->name('register.store');
    
    // Validação de código (AJAX)
    Route::post('/register/validate-code', [RegisterController::class, 'validateCode'])
        ->name('register.validate-code');
});

// Área Autenticada
Route::middleware('auth')->group(function () {
    
    // Status de aprovação (qualquer usuário logado)
    Route::get('/meu-status', [ApprovalStatusController::class, 'show'])
        ->name('approval.status');
    
    // Dashboard de Aprovações (apenas SuperAdmin)
    Route::middleware('can:manage-approvals')->group(function () {
        Route::get('/admin/aprovacoes', [ApprovalController::class, 'index'])
            ->name('admin.approvals.index');
        Route::post('/admin/aprovacoes/{user}/aprovar', [ApprovalController::class, 'approve'])
            ->name('admin.approvals.approve');
        Route::post('/admin/aprovacoes/{user}/rejeitar', [ApprovalController::class, 'reject'])
            ->name('admin.approvals.reject');
    });
});
```

---

### **2. Controllers**

#### **RegisterController**
```php
class RegisterController extends Controller
{
    public function showForm()
    {
        return view('auth.register');
    }
    
    public function validateCode(Request $request)
    {
        $request->validate(['codigo' => 'required|string|size:10']);
        
        $empresa = Empresa::where('codigo_convite', $request->codigo)
                          ->where('tipo_empresa', 'franquia')
                          ->where('status', 0) // ativa
                          ->first();
        
        if (!$empresa) {
            return response()->json([
                'valid' => false,
                'message' => 'Código inválido ou franquia desativada'
            ], 422);
        }
        
        return response()->json([
            'valid' => true,
            'empresa' => [
                'id' => $empresa->id,
                'name' => $empresa->name,
            ]
        ]);
    }
    
    public function register(RegisterRequest $request)
    {
        DB::transaction(function () use ($request) {
            $empresa = Empresa::findOrFail($request->empresa_id);
            
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'empresa_id' => $empresa->id,
                'status_cadastro' => 'pendente',
            ]);
            
            // Enviar email
            Mail::to($user->email)
                ->queue(new CadastroPendenteAprovacao($user));
            
            // Notificar admins
            $admins = User::where('is_superadmin', true)->get();
            foreach ($admins as $admin) {
                $admin->notify(new NovoCadastroPendente($user));
            }
        });
        
        return redirect()->route('login')
            ->with('success', 'Cadastro realizado! Aguarde aprovação.');
    }
}
```

#### **ApprovalController**
```php
class ApprovalController extends Controller
{
    public function index()
    {
        $pendentes = User::with('empresa')
            ->where('status_cadastro', 'pendente')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('admin.approvals.index', compact('pendentes'));
    }
    
    public function approve(User $user)
    {
        DB::transaction(function () use ($user) {
            $user->update([
                'status_cadastro' => 'aprovado',
                'aprovado_por' => auth()->id(),
                'aprovado_em' => now(),
            ]);
            
            Mail::to($user->email)
                ->queue(new CadastroAprovado($user));
        });
        
        return back()->with('success', "Usuário {$user->name} aprovado!");
    }
    
    public function reject(RejectUserRequest $request, User $user)
    {
        DB::transaction(function () use ($request, $user) {
            $user->update([
                'status_cadastro' => 'rejeitado',
                'aprovado_por' => auth()->id(),
                'aprovado_em' => now(),
                'motivo_rejeicao' => $request->motivo,
            ]);
            
            Mail::to($user->email)
                ->queue(new CadastroRejeitado($user));
        });
        
        return back()->with('success', "Cadastro de {$user->name} rejeitado.");
    }
}
```

---

### **3. Requests (Validação)**

#### **RegisterRequest**
```php
class RegisterRequest extends FormRequest
{
    public function rules()
    {
        return [
            'codigo_convite' => [
                'required',
                'string',
                'size:10',
                'exists:empresas,codigo_convite',
            ],
            'empresa_id' => 'required|exists:empresas,id',
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
        ];
    }
    
    public function messages()
    {
        return [
            'codigo_convite.exists' => 'Código de convite inválido',
            'email.unique' => 'Este email já está cadastrado',
        ];
    }
}
```

#### **RejectUserRequest**
```php
class RejectUserRequest extends FormRequest
{
    public function rules()
    {
        return [
            'motivo' => 'required|string|min:10|max:500',
        ];
    }
}
```

---

### **4. Views (Blade)**

#### **resources/views/auth/register.blade.php**
```blade
<x-guest-layout>
    <h2>Cadastro de Novo Usuário</h2>
    
    <form method="POST" action="{{ route('register.store') }}" id="registerForm">
        @csrf
        
        {{-- Passo 1: Código de Convite --}}
        <div id="step1">
            <label>Código de Convite</label>
            <input 
                type="text" 
                name="codigo_convite" 
                id="codigo_convite"
                maxlength="10"
                placeholder="XXXX999999"
                required
            >
            <button type="button" id="validateCodeBtn">
                Validar Código
            </button>
            
            <div id="empresaInfo" class="hidden">
                <p>✅ Franquia: <strong id="empresaName"></strong></p>
                <input type="hidden" name="empresa_id" id="empresa_id">
            </div>
        </div>
        
        {{-- Passo 2: Dados Pessoais (hidden inicialmente) --}}
        <div id="step2" class="hidden">
            <h3>Seus Dados</h3>
            
            <label>Nome Completo</label>
            <input type="text" name="name" required>
            
            <label>Email</label>
            <input type="email" name="email" required>
            
            <label>Telefone</label>
            <input type="text" name="phone">
            
            <label>Senha</label>
            <input type="password" name="password" required>
            
            <label>Confirmar Senha</label>
            <input type="password" name="password_confirmation" required>
            
            <button type="submit">Finalizar Cadastro</button>
        </div>
    </form>
    
    @push('scripts')
    <script>
        // AJAX para validar código
        document.getElementById('validateCodeBtn').addEventListener('click', function() {
            const codigo = document.getElementById('codigo_convite').value;
            
            fetch('{{ route("register.validate-code") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ codigo })
            })
            .then(res => res.json())
            .then(data => {
                if (data.valid) {
                    document.getElementById('empresa_id').value = data.empresa.id;
                    document.getElementById('empresaName').textContent = data.empresa.name;
                    document.getElementById('empresaInfo').classList.remove('hidden');
                    document.getElementById('step2').classList.remove('hidden');
                } else {
                    alert(data.message);
                }
            });
        });
    </script>
    @endpush
</x-guest-layout>
```

#### **resources/views/admin/approvals/index.blade.php**
```blade
<x-app-layout>
    <h1>Aprovações Pendentes</h1>
    
    @if($pendentes->count() > 0)
        <div class="badge">
            {{ $pendentes->total() }} cadastros aguardando
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Data</th>
                    <th>Nome</th>
                    <th>Email</th>
                    <th>Franquia</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody>
                @foreach($pendentes as $user)
                <tr>
                    <td>{{ $user->created_at->format('d/m/Y H:i') }}</td>
                    <td>{{ $user->name }}</td>
                    <td>{{ $user->email }}</td>
                    <td>{{ $user->empresa->name }}</td>
                    <td>
                        <form method="POST" action="{{ route('admin.approvals.approve', $user) }}" style="display:inline">
                            @csrf
                            <button class="btn-success">✅ Aprovar</button>
                        </form>
                        
                        <button 
                            class="btn-danger" 
                            onclick="openRejectModal({{ $user->id }})"
                        >
                            ❌ Rejeitar
                        </button>
                    </td>
                </tr>
                @endforeach
            </tbody>
        </table>
        
        {{ $pendentes->links() }}
    @else
        <p>Nenhum cadastro pendente 🎉</p>
    @endif
    
    {{-- Modal de Rejeição --}}
    <div id="rejectModal" class="modal hidden">
        <form method="POST" id="rejectForm">
            @csrf
            <h3>Motivo da Rejeição</h3>
            <textarea name="motivo" required minlength="10"></textarea>
            <button type="submit">Confirmar Rejeição</button>
            <button type="button" onclick="closeRejectModal()">Cancelar</button>
        </form>
    </div>
</x-app-layout>
```

---

### **5. Emails (Mailable)**

#### **app/Mail/CadastroPendenteAprovacao.php**
```php
class CadastroPendenteAprovacao extends Mailable
{
    public function __construct(public User $user) {}
    
    public function build()
    {
        return $this->subject('Cadastro Recebido - Aguardando Aprovação')
                    ->markdown('emails.cadastro-pendente');
    }
}
```

#### **resources/views/emails/cadastro-pendente.blade.php**
```blade
@component('mail::message')
# Olá, {{ $user->name }}!

Seu cadastro foi recebido com sucesso e está **aguardando aprovação**.

**Franquia:** {{ $user->empresa->name }}

Você receberá um email assim que seu cadastro for aprovado.

Obrigado,<br>
{{ config('app.name') }}
@endcomponent
```

---

### **6. Notifications**

#### **app/Notifications/NovoCadastroPendente.php**
```php
class NovoCadastroPendente extends Notification
{
    public function __construct(public User $user) {}
    
    public function via($notifiable)
    {
        return ['database', 'mail'];
    }
    
    public function toArray($notifiable)
    {
        return [
            'user_id' => $this->user->id,
            'user_name' => $this->user->name,
            'user_email' => $this->user->email,
            'empresa_name' => $this->user->empresa->name,
        ];
    }
}
```

---

### **7. Middleware de Bloqueio**

#### **app/Http/Middleware/CheckApprovalStatus.php**
```php
class CheckApprovalStatus
{
    public function handle($request, Closure $next)
    {
        $user = auth()->user();
        
        // Se pendente, redireciona para tela de status
        if ($user->status_cadastro === 'pendente') {
            return redirect()->route('approval.status');
        }
        
        // Se rejeitado, desloga
        if ($user->status_cadastro === 'rejeitado') {
            auth()->logout();
            return redirect()->route('login')
                ->with('error', 'Seu cadastro foi rejeitado.');
        }
        
        return $next($request);
    }
}
```

**Registrar em `app/Http/Kernel.php`:**
```php
protected $routeMiddleware = [
    // ...
    'approved' => \App\Http\Middleware\CheckApprovalStatus::class,
];
```

**Aplicar nas rotas:**
```php
Route::middleware(['auth', 'approved'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index']);
    // ... outras rotas protegidas
});
```

---

## 🎨 Design (Wireframes)

### **Tela 1: Registro**
```
┌─────────────────────────────────────┐
│  FreeFi Admin - Cadastro            │
├─────────────────────────────────────┤
│                                     │
│  Passo 1: Código de Convite         │
│  ┌────────────────┐  ┌──────────┐  │
│  │ XXXX999999     │  │ Validar  │  │
│  └────────────────┘  └──────────┘  │
│                                     │
│  ✅ Franquia: FreeFi São Paulo      │
│                                     │
│  ─────────────────────────────────  │
│                                     │
│  Passo 2: Seus Dados                │
│  Nome:    ┌──────────────────────┐  │
│           │                      │  │
│           └──────────────────────┘  │
│                                     │
│  Email:   ┌──────────────────────┐  │
│           │                      │  │
│           └──────────────────────┘  │
│                                     │
│  Senha:   ┌──────────────────────┐  │
│           │ ••••••••             │  │
│           └──────────────────────┘  │
│                                     │
│           ┌──────────────────────┐  │
│           │  Finalizar Cadastro  │  │
│           └──────────────────────┘  │
└─────────────────────────────────────┘
```

### **Tela 2: Dashboard de Aprovações**
```
┌──────────────────────────────────────────────┐
│  Aprovações Pendentes            [3] 🔔      │
├──────────────────────────────────────────────┤
│  Data       Nome           Empresa    Ações  │
├──────────────────────────────────────────────┤
│  17/10 14h  João Silva     SP     ✅ ❌      │
│  17/10 15h  Maria Santos   RJ     ✅ ❌      │
│  17/10 16h  Pedro Alves    MG     ✅ ❌      │
└──────────────────────────────────────────────┘
```

---

## ✅ Checklist de Implementação

### **Backend**
- [ ] Criar RegisterController
- [ ] Criar ApprovalController
- [ ] Criar ApprovalStatusController
- [ ] Criar RegisterRequest
- [ ] Criar RejectUserRequest
- [ ] Criar CheckApprovalStatus Middleware
- [ ] Criar Mailables (3 emails)
- [ ] Criar Notifications
- [ ] Configurar Queue (Redis ou Database)

### **Frontend**
- [ ] Criar view auth/register.blade.php
- [ ] Criar view admin/approvals/index.blade.php
- [ ] Criar view approval/status.blade.php
- [ ] Criar templates de email
- [ ] Adicionar validação JavaScript
- [ ] Criar modal de rejeição
- [ ] Adicionar badges de notificação

### **Testes**
- [ ] Feature Test: Registro com código válido
- [ ] Feature Test: Registro com código inválido
- [ ] Feature Test: Aprovação de usuário
- [ ] Feature Test: Rejeição de usuário
- [ ] Feature Test: Middleware bloqueia pendentes
- [ ] Unit Test: Validação de código
- [ ] Unit Test: Envio de emails

### **Documentação**
- [ ] README atualizado
- [ ] Guia de uso para admins
- [ ] Fluxo de onboarding documentado

---

## 📅 Estimativa de Tempo

- **Backend:** 6-8 horas
- **Frontend:** 4-6 horas
- **Emails/Notifications:** 2-3 horas
- **Testes:** 3-4 horas
- **Documentação:** 1-2 horas

**Total:** 16-23 horas (~3-4 dias)

---

## 🚀 Próxima Fase (FASE 3)

Após concluir FASE 2, partimos para:

**FASE 3: Controle de Acesso e Permissions**
- Global Scopes (filtro automático)
- Roles e Permissions
- Middleware de autorização
- Testes de isolamento de dados

---

## 📌 Observações Importantes

1. **Segurança:**
   - Sempre validar código de convite no backend
   - Usar CSRF tokens em todos os forms
   - Hash de senhas com bcrypt
   - Limitar tentativas de registro (rate limiting)

2. **UX:**
   - Feedback visual em tempo real
   - Mensagens de erro claras
   - Loading states durante AJAX
   - Confirmações antes de aprovar/rejeitar

3. **Performance:**
   - Queue para envio de emails
   - Eager loading (with) nas queries
   - Cache de códigos de convite válidos
   - Pagination nas listagens

4. **Acessibilidade:**
   - Labels em todos os inputs
   - Mensagens de erro acessíveis
   - Navegação por teclado
   - Contraste adequado

---

**Pronto para começar a FASE 2?** 🚀
