# 🎮 GUIA PRÁTICO - TESTANDO O SISTEMA DE REGISTRO

Este documento fornece exemplos práticos e passo a passo para testar o sistema de registro com código de convite.

---

## 🌐 TESTES VIA NAVEGADOR

### 1. Acesso ao Formulário de Registro

**URL:** http://localhost:8082/register

**Comportamento esperado:**
- ✅ Página carrega com formulário de registro
- ✅ Campo "Código de Convite" visível e em destaque
- ✅ Botão "Validar Código" habilitado
- ✅ Formulário de dados pessoais oculto inicialmente

---

### 2. Teste de Validação - Código Válido

**Passo a passo:**
1. Digite no campo "Código de Convite": `zxhb099306` (minúsculo)
2. Observe a conversão automática para: `ZXHB099306`
3. Clique em "Validar Código"

**Comportamento esperado:**
- ✅ Loading spinner aparece no botão
- ✅ Requisição AJAX para `/register/validate-code`
- ✅ Mensagem verde: "Código válido! Preencha seus dados para continuar"
- ✅ Exibe: "Franquia: FreeFi São Paulo"
- ✅ Formulário de dados pessoais aparece
- ✅ Campo de código fica desabilitado

**Resposta AJAX esperada:**
```json
{
  "valid": true,
  "empresa": {
    "id": 1,
    "name": "FreeFi São Paulo",
    "email": "saopaulo@freefi.com.br"
  }
}
```

---

### 3. Teste de Validação - Código Inválido

**Códigos para testar:**

#### a) Código Inexistente
```
Código: XPTO999999
```
**Comportamento esperado:**
- ❌ Mensagem vermelha: "Código de convite inválido ou franquia desativada."
- ❌ Formulário de dados não aparece

#### b) Franquia Inativa
```
Código: EGU2160261
```
**Comportamento esperado:**
- ❌ Mensagem vermelha: "Código de convite inválido ou franquia desativada."
- ❌ Formulário de dados não aparece

#### c) Código com Tamanho Inválido
```
Código: ABC123
```
**Comportamento esperado:**
- ❌ Erro de validação: "O campo codigo deve ter 10 caracteres."

---

### 4. Teste de Registro Completo - Sucesso

**Pré-requisito:** Código válido já validado (ex: `ZXHB099306`)

**Dados para preencher:**
```
Nome:               João Silva Teste
Email:              joao.teste.2025@example.com
Telefone:           (11) 99999-9999
Senha:              senha12345
Confirmar Senha:    senha12345
```

**Passo a passo:**
1. Preencha todos os campos conforme acima
2. Clique em "Registrar"

**Comportamento esperado:**
- ✅ Formulário é submetido via POST para `/register`
- ✅ Redirecionamento para `/login`
- ✅ Mensagem de sucesso: "Cadastro realizado com sucesso! Aguarde a aprovação."
- ✅ Usuário criado no banco com:
  - `status_cadastro` = 'pendente'
  - `empresa_id` = 1 (FreeFi São Paulo)
  - `password` = hash bcrypt

**Verificação no banco:**
```sql
SELECT id, name, email, status_cadastro, empresa_id, created_at
FROM users
WHERE email = 'joao.teste.2025@example.com';
```

---

### 5. Teste de Registro - Erros de Validação

#### a) Email Duplicado
```
Email: joao.teste.2025@example.com (já cadastrado acima)
```
**Comportamento esperado:**
- ❌ Erro: "Este email já está cadastrado."

#### b) Nome Muito Curto
```
Nome: Jo
```
**Comportamento esperado:**
- ❌ Erro: "O campo nome deve ter pelo menos 3 caracteres."

#### c) Senha Fraca
```
Senha: 123
```
**Comportamento esperado:**
- ❌ Erro: "A senha deve ter no mínimo 8 caracteres."

#### d) Senhas Não Coincidem
```
Senha:           senha12345
Confirmar Senha: senha54321
```
**Comportamento esperado:**
- ❌ Erro: "As senhas não coincidem."

---

## 🧪 TESTES VIA SCRIPTS PHP

### 1. Executar Testes de Validação Estrutural

```bash
docker exec freefi-admin-app php run-registration-tests.php
```

**O que é testado:**
- ✅ Existência de rotas
- ✅ Estrutura de banco de dados
- ✅ Códigos de convite cadastrados
- ✅ Hashing de senhas
- ✅ Factory de usuários

**Saída esperada:**
```
🧪 TESTES DE VALIDAÇÃO DE ROTAS
════════════════════════════════════════════════════════════
✓ Rota GET /register existe
✓ Rota POST /register/validate-code existe
✓ Rota POST /register existe

📊 TESTES DE ESTRUTURA DE BANCO DE DADOS
════════════════════════════════════════════════════════════
✓ Tabela users tem todos os campos necessários
✓ Tabela empresas tem campo codigo_convite

... (mais 8 testes)

Taxa de sucesso: 100%
```

---

### 2. Executar Testes de Integração End-to-End

```bash
docker exec freefi-admin-app php test-registration-flow.php
```

**O que é testado:**
- ✅ Validação AJAX de códigos
- ✅ Registro completo de usuário
- ✅ Segurança de senhas
- ✅ Transações de banco
- ✅ Casos extremos

**Saída esperada:**
```
╔═══════════════════════════════════════════════════════════════╗
║  TESTE DE INTEGRAÇÃO - REGISTRO COM CÓDIGO DE CONVITE        ║
╚═══════════════════════════════════════════════════════════════╝

═══ FASE 1: PREPARAÇÃO DE DADOS ═══
🧪 Teste 1: Buscar franquia ativa para teste ✓ PASSOU

═══ FASE 2: TESTES DE VALIDAÇÃO DE CÓDIGO (AJAX) ═══
🧪 Teste 3: Validar código existente e ativo ✓ PASSOU
🧪 Teste 4: Rejeitar código inexistente ✓ PASSOU

... (mais 9 testes)

Total de testes: 13
✓ Passou: 13
Taxa de sucesso: 100%
```

---

## 🔧 TESTES VIA TINKER

### 1. Abrir Tinker

```bash
docker exec -it freefi-admin-app php artisan tinker
```

### 2. Listar Franquias com Código de Convite

```php
$franquias = App\Models\Empresa::where('tipo_empresa', 'franquia')
    ->whereNotNull('codigo_convite')
    ->get(['name', 'codigo_convite', 'status']);

foreach ($franquias as $f) {
    echo "{$f->name}: {$f->codigo_convite} (Status: " . 
         ($f->status == 0 ? 'Ativa' : 'Inativa') . ")\n";
}
```

**Saída esperada:**
```
FreeFi São Paulo: ZXHB099306 (Status: Ativa)
FreeFi Rio de Janeiro: ZBU0842472 (Status: Ativa)
FreeFi Minas Gerais: HANK512710 (Status: Ativa)
FreeFi Bahia: EGU2160261 (Status: Inativa)
```

### 3. Testar Validação de Código

```php
use Illuminate\Http\Request;
use App\Http\Controllers\Auth\RegisterController;

$request = Request::create('/register/validate-code', 'POST', [
    'codigo' => 'ZXHB099306'
]);

$controller = new RegisterController();
$response = $controller->validateCode($request);

echo $response->getContent();
```

**Saída esperada:**
```json
{
  "valid": true,
  "empresa": {
    "id": 1,
    "name": "FreeFi São Paulo",
    "email": "saopaulo@freefi.com.br"
  }
}
```

### 4. Criar Usuário de Teste

```php
use App\Models\User;

$user = User::create([
    'name' => 'Teste Tinker',
    'email' => 'teste.tinker.' . time() . '@example.com',
    'password' => bcrypt('senha12345'),
    'empresa_id' => 1,
    'status_cadastro' => 'pendente',
]);

echo "Usuário criado: ID {$user->id}, Status: {$user->status_cadastro}\n";
echo "Senha hasheada: {$user->password}\n";
```

**Saída esperada:**
```
Usuário criado: ID 15, Status: pendente
Senha hasheada: $2y$12$kJ8dF3nH2mL9pQ4rT6vW8eYz...
```

### 5. Verificar Usuários Pendentes

```php
$pendentes = User::where('status_cadastro', 'pendente')
    ->with('empresa')
    ->get();

echo "Total de usuários pendentes: {$pendentes->count()}\n\n";

foreach ($pendentes as $user) {
    echo "ID: {$user->id}\n";
    echo "Nome: {$user->name}\n";
    echo "Email: {$user->email}\n";
    echo "Franquia: {$user->empresa->name}\n";
    echo "Cadastrado em: {$user->created_at->format('d/m/Y H:i')}\n";
    echo "---\n";
}
```

---

## 📮 TESTES VIA POSTMAN/CURL

### 1. Validar Código de Convite (AJAX)

#### Request
```bash
curl -X POST http://localhost:8082/register/validate-code \
  -H "Content-Type: application/json" \
  -H "X-CSRF-TOKEN: seu-token-csrf" \
  -d '{"codigo": "ZXHB099306"}'
```

#### Response (Sucesso)
```json
{
  "valid": true,
  "empresa": {
    "id": 1,
    "name": "FreeFi São Paulo",
    "email": "saopaulo@freefi.com.br"
  }
}
```

#### Response (Falha)
```json
{
  "valid": false,
  "message": "Código de convite inválido ou franquia desativada."
}
```

---

### 2. Registro Completo

#### Request
```bash
curl -X POST http://localhost:8082/register \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -H "X-CSRF-TOKEN: seu-token-csrf" \
  -d "codigo_convite=ZXHB099306" \
  -d "name=João da Silva" \
  -d "email=joao.curl@example.com" \
  -d "phone=11999999999" \
  -d "password=senha12345" \
  -d "password_confirmation=senha12345"
```

#### Response (Sucesso)
```
HTTP/1.1 302 Found
Location: /login
Set-Cookie: laravel_session=...

Flash message: "Cadastro realizado com sucesso! Aguarde a aprovação."
```

#### Response (Erro de Validação)
```json
{
  "message": "Os dados fornecidos são inválidos.",
  "errors": {
    "email": [
      "Este email já está cadastrado."
    ]
  }
}
```

---

## 🗄️ VERIFICAÇÕES NO BANCO DE DADOS

### 1. Verificar Usuário Criado

```sql
-- Usuário mais recente
SELECT 
    u.id,
    u.name,
    u.email,
    u.status_cadastro,
    e.name as empresa_nome,
    e.codigo_convite,
    u.created_at
FROM users u
INNER JOIN empresas e ON u.empresa_id = e.id
ORDER BY u.created_at DESC
LIMIT 1;
```

### 2. Listar Usuários Pendentes

```sql
SELECT 
    u.id,
    u.name,
    u.email,
    e.name as franquia,
    u.created_at
FROM users u
INNER JOIN empresas e ON u.empresa_id = e.id
WHERE u.status_cadastro = 'pendente'
ORDER BY u.created_at DESC;
```

### 3. Verificar Hash de Senha

```sql
SELECT 
    id,
    name,
    LEFT(password, 20) as senha_hash_inicio,
    LENGTH(password) as tamanho_hash
FROM users
WHERE id = (SELECT MAX(id) FROM users);
```

**Resultado esperado:**
```
senha_hash_inicio: $2y$12$kJ8dF3nH2mL9
tamanho_hash: 60
```

---

## 🎯 CHECKLIST DE TESTES MANUAIS

### ✅ Testes Básicos
- [ ] Página de registro carrega corretamente
- [ ] Campo de código aceita input e converte para uppercase
- [ ] Botão "Validar Código" funciona
- [ ] Código válido exibe formulário de dados
- [ ] Código inválido exibe mensagem de erro
- [ ] Formulário completo pode ser submetido

### ✅ Validações Frontend
- [ ] Conversão automática para maiúsculo
- [ ] AJAX validation funciona
- [ ] Loading spinner aparece durante validação
- [ ] Mensagens de sucesso/erro são exibidas
- [ ] Step 2 só aparece após validação bem-sucedida

### ✅ Validações Backend
- [ ] Código com tamanho diferente de 10 é rejeitado
- [ ] Código inexistente é rejeitado
- [ ] Franquia inativa é rejeitada
- [ ] Email duplicado é rejeitado
- [ ] Nome com menos de 3 caracteres é rejeitado
- [ ] Senha com menos de 8 caracteres é rejeitada
- [ ] Senhas diferentes são rejeitadas
- [ ] Telefone é opcional

### ✅ Segurança
- [ ] Senha é hasheada com bcrypt
- [ ] Hash tem 60+ caracteres
- [ ] Senha nunca armazenada em texto plano
- [ ] Transação protege cadastro incompleto

### ✅ Integração
- [ ] Usuário criado com status='pendente'
- [ ] Usuário associado à empresa correta
- [ ] Redirect para login após cadastro
- [ ] Flash message exibida

---

## 📊 DADOS DE TESTE DISPONÍVEIS

### Códigos Válidos (Use livremente)
```
ZXHB099306 - FreeFi São Paulo
ZBU0842472 - FreeFi Rio de Janeiro
HANK512710 - FreeFi Minas Gerais
```

### Códigos para Testes Negativos
```
EGU2160261 - Franquia inativa (deve ser rejeitado)
XPTO999999 - Inexistente (deve ser rejeitado)
ABC123     - Tamanho inválido (deve ser rejeitado)
```

### Emails de Teste
```
teste1@example.com
teste2@example.com
usuario.teste@example.com
novo.cadastro@example.com
```

**Dica:** Use timestamp para emails únicos:
```
teste.{timestamp}@example.com
```

---

**Última atualização:** <?php echo date('d/m/Y H:i:s'); ?>  
**Próximo teste:** Acesse http://localhost:8082/register e experimente!
