<?php

require __DIR__ . '/../vendor/autoload.php';

$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\{Empresa, Hotspot, Criativo, Publicacao, WalledGarden, PublicacaoImpressao, PublicacaoClique};

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "5️⃣ TESTE E2E - CRIANDO DADOS DE TESTE\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "🔧 Criando dados de teste...\n";

// 1. Criar Franquia
$franquia = Empresa::firstOrCreate(
    ['tipo_empresa' => 'franquia'],
    [
        'name' => 'Franquia Teste E2E',
        'razao_social' => 'Franquia Teste LTDA',
        'cnpj' => '12345678000199',
        'status' => true
    ]
);
echo "✅ Franquia criada: ID={$franquia->id}\n";

// 2. Criar Cliente
$cliente = Empresa::firstOrCreate(
    ['tipo_empresa' => 'cliente', 'name' => 'Cliente Teste E2E'],
    [
        'razao_social' => 'Cliente Teste LTDA',
        'cnpj' => '98765432000188',
        'status' => true
    ]
);
echo "✅ Cliente criado: ID={$cliente->id}\n";

// 3. Criar Hotspot
$hotspot = Hotspot::firstOrCreate(
    ['serial' => 'E2E-TEST-001'],
    [
        'marca' => 'MikroTik',
        'modelo' => 'RB750',
        'franquia_id' => $franquia->id,
        'cliente_id' => $cliente->id,
        'status' => true,
        'vpn_ip' => '10.0.0.1'
    ]
);
echo "✅ Hotspot criado: ID={$hotspot->id}, Serial={$hotspot->serial}\n";

// 4. Criar Criativo
$criativo = Criativo::firstOrCreate(
    ['title' => 'Banner Teste E2E'],
    [
        'franquia_id' => $franquia->id,
        'tipo' => 'imagem',
        'formato' => 'jpg',
        'path' => 'https://via.placeholder.com/728x90.jpg',
        'status_aprovacao' => 'aprovado'
    ]
);
echo "✅ Criativo criado: ID={$criativo->id}\n";

// 5. Criar Publicação
// Buscar um usuário admin para publicada_por
$admin = \App\Models\User::first();
if (!$admin) {
    echo "⚠️ Nenhum usuário encontrado, criando usuário admin...\n";
    $admin = \App\Models\User::create([
        'name' => 'Admin Teste',
        'email' => 'admin@teste.com',
        'password' => bcrypt('password'),
        'tipo' => 'admin'
    ]);
}

$publicacao = Publicacao::firstOrCreate(
    ['titulo' => 'Publicação Teste E2E'],
    [
        'criativo_id' => $criativo->id,
        'franquia_id' => $franquia->id,
        'cliente_id' => $cliente->id,
        'posicao' => 1,
        'tipo_alcance' => 'todos',
        'data_inicio' => now()->subDays(10),
        'data_fim' => now()->addDays(20),
        'status' => 'ativa',
        'link_destino' => 'https://exemplo.com/promo',
        'habilitar_clique' => true,
        'track_impressoes' => true,
        'track_cliques' => true,
        'publicada_por' => $admin->id,
        'publicada_em' => now()
    ]
);
echo "✅ Publicação criada: ID={$publicacao->id}\n";

// 6. Criar Walled Garden
$wg = WalledGarden::firstOrCreate(
    [
        'publicacao_id' => $publicacao->id,
        'tipo' => 'domain',
        'endereco' => 'cdn.exemplo.com'
    ],
    [
        'descricao' => 'CDN de imagens',
        'ativo' => true
    ]
);
echo "✅ Walled Garden criado: ID={$wg->id}\n";

echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "6️⃣ TESTE E2E - SIMULANDO API DO HOTSPOT\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

// Teste 1: Heartbeat
echo "📡 Teste 1: Heartbeat\n";
$hotspot->update([
    'last_activity_at' => now(),
    'vpn_ip' => '10.0.0.100'
]);
echo "✅ Heartbeat atualizado: last_activity_at={$hotspot->last_activity_at}\n\n";

// Teste 2: Buscar Publicações
echo "📡 Teste 2: Buscar Publicações\n";
$publicacoesVigentes = Publicacao::where('status', 'ativa')
    ->where('data_inicio', '<=', now())
    ->where('data_fim', '>=', now())
    ->with(['criativo', 'walledGardens'])
    ->get();
echo "✅ Publicações encontradas: {$publicacoesVigentes->count()}\n";
foreach($publicacoesVigentes as $pub) {
    echo "   - {$pub->titulo} (ID={$pub->id})\n";
    echo "     Criativo: {$pub->criativo->title}\n";
    echo "     Walled Gardens: {$pub->walledGardens->count()}\n";
}
echo "\n";

// Teste 3: Registrar Impressão
echo "📡 Teste 3: Registrar Impressão\n";
$impressao = PublicacaoImpressao::firstOrNew([
    'hotspot_id' => $hotspot->id,
    'publicacao_id' => $publicacao->id,
    'criativo_id' => $criativo->id,
    'data' => now()->format('Y-m-d')
]);
$impressao->impressoes = ($impressao->impressoes ?? 0) + 1;
$impressao->usuarios_unicos = max($impressao->usuarios_unicos ?? 0, 5);
$impressao->save();
echo "✅ Impressão registrada: total={$impressao->impressoes}, unicos={$impressao->usuarios_unicos}\n\n";

// Teste 4: Registrar Clique
echo "📡 Teste 4: Registrar Clique\n";
$clique = PublicacaoClique::create([
    'hotspot_id' => $hotspot->id,
    'publicacao_id' => $publicacao->id,
    'criativo_id' => $criativo->id,
    'ip_usuario' => '192.168.1.100',
    'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
    'clicked_at' => now()
]);
echo "✅ Clique registrado: ID={$clique->id}\n\n";

// Teste 5: Calcular CTR
echo "📡 Teste 5: Calcular CTR\n";
$totalImpressoes = PublicacaoImpressao::where('publicacao_id', $publicacao->id)->sum('impressoes');
$totalCliques = PublicacaoClique::where('publicacao_id', $publicacao->id)->count();
$ctr = $totalImpressoes > 0 ? ($totalCliques / $totalImpressoes) * 100 : 0;
echo "✅ CTR calculado:\n";
echo "   - Impressões: {$totalImpressoes}\n";
echo "   - Cliques: {$totalCliques}\n";
echo "   - CTR: " . number_format($ctr, 2) . "%\n\n";

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "7️⃣ TESTE DE RELACIONAMENTOS\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

// Testar relacionamentos
echo "🔗 Testando Relacionamentos:\n";
echo "✅ Publicação -> Walled Gardens: " . $publicacao->walledGardens()->count() . " registros\n";
echo "✅ Publicação -> Impressões: " . $publicacao->impressoes()->count() . " registros\n";
echo "✅ Publicação -> Cliques: " . $publicacao->cliques()->count() . " registros\n";
echo "✅ Hotspot -> Impressões: " . $hotspot->impressoes()->count() . " registros\n";
echo "✅ Hotspot -> Cliques: " . $hotspot->cliques()->count() . " registros\n";
echo "✅ Criativo -> Impressões: " . $criativo->impressoes()->count() . " registros\n";
echo "✅ Criativo -> Cliques: " . $criativo->cliques()->count() . " registros\n";

echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "✅ TODOS OS TESTES E2E PASSARAM!\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "📊 RESUMO DOS DADOS CRIADOS:\n";
echo "   - Franquia: {$franquia->name}\n";
echo "   - Cliente: {$cliente->name}\n";
echo "   - Hotspot: {$hotspot->serial}\n";
echo "   - Criativo: {$criativo->title}\n";
echo "   - Publicação: {$publicacao->titulo}\n";
echo "   - Walled Gardens: 1\n";
echo "   - Impressões: {$totalImpressoes}\n";
echo "   - Cliques: {$totalCliques}\n";
echo "   - CTR: " . number_format($ctr, 2) . "%\n";
