<?php

namespace App\Models;

use App\Traits\Userstamps;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Empresa extends Model
{
    use HasFactory, SoftDeletes, Userstamps;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'logomarca',
        'status',
        'tipo_empresa',
        'empresa_pai_id',
        'codigo_convite',
        'token_convite',
        'address',
        'status_aprovacao',
        'aprovado_por',
        'aprovado_em',
        'motivo_rejeicao',
        // Mercado Pago
        'mp_access_token',
        'mp_public_key',
        'mp_ativo',
        'mp_split_habilitado',
        'mp_percentual_split',
        'mp_empresa_split_id',
        'mp_ambiente',
    ];

    protected $casts = [
        'tipo_empresa' => 'string',
        'aprovado_em' => 'datetime',
        // Mercado Pago
        'mp_access_token' => 'encrypted',
        'mp_ativo' => 'boolean',
        'mp_split_habilitado' => 'boolean',
        'mp_percentual_split' => 'decimal:2',
    ];

    // ==========================================
    // RELACIONAMENTOS HIERARQUIA
    // ==========================================
    
    /**
     * Empresa pai (matriz ou franquia)
     */
    public function empresaPai()
    {
        return $this->belongsTo(Empresa::class, 'empresa_pai_id');
    }
    
    /**
     * Franquias (se for matriz)
     */
    public function franquias()
    {
        return $this->hasMany(Empresa::class, 'empresa_pai_id')
                    ->where('tipo_empresa', 'franquia');
    }
    
    /**
     * Clientes (se for matriz ou franquia)
     */
    public function clientes()
    {
        return $this->hasMany(Empresa::class, 'empresa_pai_id')
                    ->where('tipo_empresa', 'cliente');
    }

    // ==========================================
    // RELACIONAMENTOS LEGADOS
    // ==========================================
    
    public function midias()
    {
        return $this->hasMany(Midia::class);
    }

    /**
     * Hotspots como CLIENTE
     */
    public function hotspots()
    {
        return $this->hasMany(Hotspot::class, 'cliente_id');
    }
    
    /**
     * Hotspots gerenciados (se for franquia)
     */
    public function hotspotsGerenciados()
    {
        return $this->hasMany(Hotspot::class, 'franquia_id');
    }
    
    /**
     * Usuários da empresa (legado - hasMany via empresa_id)
     */
    public function usuarios()
    {
        return $this->hasMany(User::class, 'empresa_id');
    }

    /**
     * Usuário que aprovou/rejeitou o cadastro
     */
    public function aprovador()
    {
        return $this->belongsTo(User::class, 'aprovado_por');
    }

    /**
     * Empresa que recebe o split de pagamento (normalmente a matriz ou franquia pai)
     */
    public function empresaSplit()
    {
        return $this->belongsTo(Empresa::class, 'mp_empresa_split_id');
    }

    /**
     * Usuários com acesso a esta empresa (novo - many-to-many)
     * Permite que múltiplos usuários tenham acesso a esta franquia
     */
    public function usuariosComAcesso()
    {
        return $this->belongsToMany(User::class, 'empresa_user')
            ->withPivot('role')
            ->withTimestamps();
    }

    /**
     * Alias em inglês para usuarios()
     * Mantém consistência com o Model User
     */
    public function users()
    {
        return $this->usuarios();
    }

    /**
     * Retorna todos os clientes acessíveis por esta franquia
     * (clientes diretos dela)
     */
    public function getClientesAcessiveisAttribute()
    {
        if ($this->tipo_empresa !== 'franquia') {
            return collect([]);
        }

        return $this->clientes;
    }
    
    /**
     * Publicações da empresa (se for cliente)
     */
    public function publicacoes()
    {
        return $this->hasMany(Publicacao::class, 'cliente_id');
    }
    
    /**
     * Formatos permitidos (se for franquia)
     */
    public function formatosPermitidos()
    {
        return $this->hasMany(FranquiaFormatoPermitido::class, 'franquia_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function destroyer()
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }
}
