<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class PedidoPublicidade extends Model
{
    use SoftDeletes;

    protected $table = 'pedidos_publicidade';

    protected $fillable = [
        'codigo_pedido',
        'empresa_id',
        'codigo_empresa',
        'arquivo_criativo',
        'nome_criativo',
        'tipo_arquivo',
        'tipo_selecao',
        'hotspots_selecionados',
        'franquias_selecionadas',
        'data_inicio',
        'data_fim',
        'posicao',
        'periodicidade',
        'valor_unitario',
        'quantidade_hotspots',
        'valor_total',
        'status_pagamento',
        'forma_pagamento',
        'transaction_id',
        'dados_pagamento',
        'pago_em',
        'status',
        'observacoes',
        'criado_por',
        'aprovado_por',
        'aprovado_em',
    ];

    protected $casts = [
        'hotspots_selecionados' => 'array',
        'franquias_selecionadas' => 'array',
        'data_inicio' => 'date',
        'data_fim' => 'date',
        'valor_unitario' => 'decimal:2',
        'valor_total' => 'decimal:2',
        'pago_em' => 'datetime',
        'aprovado_em' => 'datetime',
    ];

    /**
     * Gerar código único do pedido
     */
    public static function gerarCodigoPedido()
    {
        do {
            $codigo = 'PUB-' . strtoupper(Str::random(8));
        } while (self::where('codigo_pedido', $codigo)->exists());

        return $codigo;
    }

    /**
     * Relacionamento com Empresa
     */
    public function empresa()
    {
        return $this->belongsTo(Empresa::class);
    }

    /**
     * Relacionamento com criador
     */
    public function criador()
    {
        return $this->belongsTo(User::class, 'criado_por');
    }

    /**
     * Relacionamento com aprovador
     */
    public function aprovador()
    {
        return $this->belongsTo(User::class, 'aprovado_por');
    }

    /**
     * Relacionamento com mídia do pedido
     */
    public function midiaPedido()
    {
        return $this->hasOne(MidiaPedido::class, 'pedido_id');
    }

    /**
     * Relacionamento com transações financeiras
     */
    public function transacoes()
    {
        return $this->hasMany(TransacaoFinanceira::class, 'pedido_id');
    }

    /**
     * Relacionamento com publicações criadas a partir deste pedido
     */
    public function publicacoes()
    {
        return $this->hasMany(Publicacao::class, 'pedido_id');
    }

    /**
     * Obter hotspots selecionados (models)
     */
    public function getHotspotsAttribute()
    {
        if ($this->tipo_selecao === 'individual' && $this->hotspots_selecionados) {
            return Hotspot::whereIn('id', $this->hotspots_selecionados)->get();
        }

        if ($this->tipo_selecao === 'franquia_completa' && $this->franquias_selecionadas) {
            $franquiaId = $this->franquias_selecionadas[0] ?? null;
            if ($franquiaId) {
                return Hotspot::where('franquia_id', $franquiaId)->get();
            }
        }

        if ($this->tipo_selecao === 'multiplas_franquias' && $this->franquias_selecionadas) {
            return Hotspot::whereIn('franquia_id', $this->franquias_selecionadas)->get();
        }

        return collect();
    }

    /**
     * Calcular valor total
     */
    public function calcularValorTotal()
    {
        return $this->valor_unitario * $this->quantidade_hotspots;
    }

    /**
     * Aprovar pedido
     */
    public function aprovar($userId)
    {
        $this->update([
            'status' => 'aprovado',
            'aprovado_por' => $userId,
            'aprovado_em' => now(),
        ]);
    }

    /**
     * Marcar como pago
     */
    public function marcarComoPago($transactionId = null, $formaPagamento = null)
    {
        $this->update([
            'status_pagamento' => 'aprovado',
            'transaction_id' => $transactionId,
            'forma_pagamento' => $formaPagamento,
            'pago_em' => now(),
            'status' => 'aguardando_aprovacao',
        ]);
    }
}
