<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\ConfiguracaoPreco;
use App\Models\PedidoPublicidade;
use App\Models\TransacaoFinanceira;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class PublicidadeApiController extends Controller
{
    /**
     * Validar código da empresa
     *
     * POST /api/publicidade/validar-empresa
     * Body: { "codigo": "EMP000001" }
     */
    public function validarEmpresa(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'codigo' => 'required|string|max:50',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Código da empresa é obrigatório',
                'errors' => $validator->errors(),
            ], 422);
        }

        $empresa = Empresa::where('codigo', $request->codigo)
            ->where('tipo_empresa', 'cliente')
            ->where('status_cadastro', 'aprovado')
            ->first();

        if (!$empresa) {
            return response()->json([
                'success' => false,
                'message' => 'Empresa não encontrada ou não autorizada',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Empresa encontrada',
            'data' => [
                'id' => $empresa->id,
                'codigo' => $empresa->codigo,
                'name' => $empresa->name,
                'tipo_empresa' => $empresa->tipo_empresa,
                'franquia_id' => $empresa->empresa_pai_id,
            ],
        ]);
    }

    /**
     * Obter preços de publicidade
     *
     * GET /api/publicidade/precos
     */
    public function obterPrecos(): JsonResponse
    {
        $precoSemanal = ConfiguracaoPreco::getPrecoSemanal();
        $precoMensal = ConfiguracaoPreco::getPrecoMensal();
        $multiplicador = ConfiguracaoPreco::getMultiplicadorMensal();

        return response()->json([
            'success' => true,
            'data' => [
                'semanal' => [
                    'valor' => $precoSemanal,
                    'valor_formatado' => 'R$ ' . number_format($precoSemanal, 2, ',', '.'),
                    'descricao' => 'Por semana',
                ],
                'mensal' => [
                    'valor' => $precoMensal,
                    'valor_formatado' => 'R$ ' . number_format($precoMensal, 2, ',', '.'),
                    'descricao' => 'Por mês (4 semanas)',
                    'economia' => number_format((($precoSemanal * 4) - $precoMensal), 2, ',', '.'),
                ],
                'multiplicador' => $multiplicador,
            ],
        ]);
    }

    /**
     * Webhook do Mercado Pago
     *
     * POST /api/publicidade/webhook/mercadopago
     */
    public function webhookMercadoPago(Request $request): JsonResponse
    {
        try {
            Log::info('Webhook Mercado Pago recebido', [
                'headers' => $request->headers->all(),
                'body' => $request->all(),
            ]);

            // Validar tipo de notificação
            $tipo = $request->input('type') ?? $request->input('topic');

            if (!$tipo || !in_array($tipo, ['payment', 'merchant_order'])) {
                Log::warning('Webhook Mercado Pago: tipo de notificação não suportado', [
                    'tipo' => $tipo,
                ]);
                return response()->json(['success' => true, 'message' => 'Tipo não processado'], 200);
            }

            // Obter ID do pagamento
            $paymentId = $request->input('data.id') ?? $request->input('id');

            if (!$paymentId) {
                Log::error('Webhook Mercado Pago: ID do pagamento não encontrado');
                return response()->json(['success' => false, 'message' => 'ID não fornecido'], 400);
            }

            // Buscar informações do pagamento no Mercado Pago usando o SDK
            try {
                $payment = \MercadoPago\Payment::find_by_id($paymentId);

                if (!$payment) {
                    Log::error('Webhook Mercado Pago: Pagamento não encontrado na API do MP', [
                        'payment_id' => $paymentId,
                    ]);
                    return response()->json(['success' => false, 'message' => 'Pagamento não encontrado'], 404);
                }
            } catch (\Exception $e) {
                Log::error('Erro ao buscar pagamento no Mercado Pago', [
                    'payment_id' => $paymentId,
                    'error' => $e->getMessage(),
                ]);
                return response()->json(['success' => false, 'message' => 'Erro ao buscar pagamento'], 500);
            }

            // Buscar pedido pela external_reference
            $codigoPedido = $payment->external_reference;
            $pedido = PedidoPublicidade::where('codigo_pedido', $codigoPedido)->first();

            if (!$pedido) {
                Log::warning('Webhook Mercado Pago: pedido não encontrado', [
                    'codigo_pedido' => $codigoPedido,
                    'payment_id' => $paymentId,
                ]);
                return response()->json(['success' => true, 'message' => 'Pedido não encontrado'], 200);
            }

            // Processar status do pagamento
            $statusPagamento = $payment->status;
            $statusPedido = $pedido->status;

            switch ($statusPagamento) {
                case 'approved':
                    $statusPedido = 'em_analise'; // Pago, aguardando análise do criativo
                    $statusPagamentoModel = 'aprovado';
                    break;

                case 'pending':
                case 'in_process':
                case 'in_mediation':
                    $statusPedido = 'aguardando_pagamento';
                    $statusPagamentoModel = 'pendente';
                    break;

                case 'rejected':
                case 'cancelled':
                    $statusPedido = 'cancelado';
                    $statusPagamentoModel = 'recusado';
                    break;

                case 'refunded':
                case 'charged_back':
                    $statusPedido = 'cancelado';
                    $statusPagamentoModel = 'reembolsado';
                    break;

                default:
                    $statusPagamentoModel = 'pendente';
                    break;
            }

            // Atualizar pedido
            $pedido->update([
                'status' => $statusPedido,
                'status_pagamento' => $statusPagamentoModel,
                'mp_payment_id' => $payment->id,
                'mp_payment_status' => $payment->status,
                'mp_payment_status_detail' => $payment->status_detail,
                'mp_payment_type' => $payment->payment_type_id,
                'mp_transaction_amount' => $payment->transaction_amount,
                'mp_paid_at' => $payment->date_approved ? date('Y-m-d H:i:s', strtotime($payment->date_approved)) : null,
            ]);

            Log::info('Webhook Mercado Pago processado com sucesso', [
                'pedido_id' => $pedido->id,
                'codigo_pedido' => $pedido->codigo_pedido,
                'payment_id' => $payment->id,
                'status_pagamento' => $statusPagamento,
                'status_pedido' => $statusPedido,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Webhook processado com sucesso',
                'pedido_id' => $pedido->id,
                'status' => $statusPedido,
            ], 200);

        } catch (\Exception $e) {
            Log::error('Erro ao processar webhook Mercado Pago', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao processar webhook',
            ], 500);
        }
    }
}
