<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Hotspot;
use App\Models\TawktoConfig;
use Illuminate\Http\Request;

class TawktoApiController extends Controller
{
    /**
     * Buscar configuração do Tawk.to por serial do hotspot
     *
     * GET /api/tawkto/config/{serial}
     *
     * @param string $serial Serial do hotspot (equipment serial Mikrotik)
     * @return \Illuminate\Http\JsonResponse
     */
    public function getConfigBySerial($serial)
    {
        try {
            // Buscar hotspot pelo serial
            $hotspot = Hotspot::where('serial', $serial)->first();

            if (!$hotspot) {
                return response()->json([
                    'success' => false,
                    'message' => 'Hotspot não encontrado',
                    'data' => null
                ], 404);
            }

            // Buscar config do tawk.to da franquia deste hotspot
            $config = TawktoConfig::where('empresa_id', $hotspot->franquia_id)
                ->where('ativo', true)
                ->with('empresa')
                ->first();

            if (!$config) {
                return response()->json([
                    'success' => false,
                    'message' => 'Configuração Tawk.to não encontrada ou inativa',
                    'data' => null
                ], 404);
            }

            // Preparar metadados para enviar ao Tawk.to
            $metadata = [
                'franquia_nome' => $config->empresa->name ?? 'Não identificado',
                'franquia_codigo' => $config->empresa->codigo ?? '',
                'cliente_nome' => $hotspot->cliente->name ?? 'Não identificado',
                'cliente_codigo' => $hotspot->cliente->codigo ?? '',
                'hotspot_serial' => $serial,
                'hotspot_nome' => $hotspot->nome ?? '',
            ];

            // Retornar configuração
            return response()->json([
                'success' => true,
                'message' => 'Configuração encontrada',
                'data' => [
                    'ativo' => $config->ativo,
                    'property_id' => $config->property_id,
                    'widget_id' => $config->widget_id,
                    'paginas_habilitadas' => $config->paginas_habilitadas ?? ['index', 'cadastro', 'login', 'welcome'],
                    'customizacao' => $config->customizacao ?? [
                        'primaryColor' => '#3B87C6',
                        'position' => 'br'
                    ],
                    'metadata' => array_merge($metadata, $config->metadata_personalizado ?? []),
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar configuração',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Buscar configuração por código de empresa (fallback)
     *
     * GET /api/tawkto/config-empresa/{codigo}
     *
     * @param string $codigo Código da empresa
     * @return \Illuminate\Http\JsonResponse
     */
    public function getConfigByCodigo($codigo)
    {
        try {
            // Buscar empresa pelo código
            $empresa = \App\Models\Empresa::where('codigo', $codigo)->first();

            if (!$empresa) {
                return response()->json([
                    'success' => false,
                    'message' => 'Empresa não encontrada',
                    'data' => null
                ], 404);
            }

            // Buscar config do tawk.to
            $config = TawktoConfig::where('empresa_id', $empresa->id)
                ->where('ativo', true)
                ->first();

            if (!$config) {
                return response()->json([
                    'success' => false,
                    'message' => 'Configuração Tawk.to não encontrada ou inativa',
                    'data' => null
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Configuração encontrada',
                'data' => [
                    'ativo' => $config->ativo,
                    'property_id' => $config->property_id,
                    'widget_id' => $config->widget_id,
                    'paginas_habilitadas' => $config->paginas_habilitadas ?? ['index', 'cadastro', 'login', 'welcome'],
                    'customizacao' => $config->customizacao ?? [
                        'primaryColor' => '#3B87C6',
                        'position' => 'br'
                    ],
                    'metadata' => [
                        'empresa_nome' => $empresa->name,
                        'empresa_codigo' => $empresa->codigo,
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao buscar configuração',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
