<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

/**
 * Controller para gerenciamento de clientes por usuários de franquia
 *
 * Este controller é usado APENAS por usuários de franquia.
 * SuperAdmins usam o EmpresaController.
 *
 * Funcionalidades:
 * - Listar clientes da franquia
 * - Criar novo cliente (sempre vinculado à franquia do usuário)
 * - Editar clientes da franquia
 * - Visualizar detalhes do cliente
 *
 * Criado em: 08/11/2025
 * Documentação: tests/documentation/ACESSO-USUARIO-FRANQUIA-08-11-2025.md
 */
class ClienteController extends Controller
{
    /**
     * Lista todos os clientes das franquias do usuário logado
     *
     * Suporta usuários com múltiplas franquias (many-to-many)
     * e também usuários com franquia única (legado via empresa_id)
     *
     * Query: SELECT * FROM empresas
     *        WHERE tipo_empresa = 'cliente'
     *        AND empresa_pai_id IN ({franquia_ids})
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        // 1. Buscar todas as franquias do usuário (novo: many-to-many)
        $franquiasIds = $user->empresas()
            ->where('tipo_empresa', 'franquia')
            ->pluck('empresas.id')
            ->toArray();

        // 2. Fallback: usar empresa_id legado se não tiver franquias via pivot
        if (empty($franquiasIds) && $user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            $franquiasIds = [$user->empresa_id];
        }

        // 3. Se ainda não tem franquias, retornar vazio
        if (empty($franquiasIds)) {
            return view('clientes.index', [
                'clientes' => collect([]),
                'franquias' => collect([]),
                'totalFranquias' => 0
            ]);
        }

        // 4. Buscar todas as franquias para exibir informações
        $franquias = Empresa::whereIn('id', $franquiasIds)->get();

        // 5. Query para clientes de TODAS as franquias do usuário
        $query = Empresa::where('tipo_empresa', 'cliente')
                        ->whereIn('empresa_pai_id', $franquiasIds);

        // 6. Filtro por busca textual
        if ($request->filled('q')) {
            $query->where('name', 'like', '%' . $request->input('q') . '%');
        }

        // 7. Filtro por status
        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        // 8. Filtro por franquia específica (se usuário tiver múltiplas)
        if ($request->filled('franquia_filter') && in_array($request->input('franquia_filter'), $franquiasIds)) {
            $query->where('empresa_pai_id', $request->input('franquia_filter'));
        }

        $clientes = $query->with('empresaPai')->orderBy('name', 'asc')->paginate(15);

        return view('clientes.index', compact('clientes', 'franquias'))
            ->with('totalFranquias', count($franquiasIds));
    }

    /**
     * Mostra formulário para criar novo cliente
     *
     * Se usuário tiver múltiplas franquias, mostra dropdown para escolher
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $user = Auth::user();

        // 1. Buscar todas as franquias do usuário (novo: many-to-many)
        $franquiasIds = $user->empresas()
            ->where('tipo_empresa', 'franquia')
            ->pluck('empresas.id')
            ->toArray();

        // 2. Fallback: usar empresa_id legado
        if (empty($franquiasIds) && $user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            $franquiasIds = [$user->empresa_id];
        }

        // 3. Buscar dados completos das franquias
        $franquias = Empresa::whereIn('id', $franquiasIds)->get();

        return view('clientes.create', compact('franquias'));
    }

    /**
     * Armazena novo cliente no banco de dados
     *
     * Sempre vincula à franquia do usuário logado:
     * - tipo_empresa: 'cliente'
     * - empresa_pai_id: {franquia_id}
     * - status: 0 (ativo)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        // OWASP A03:2021 – Injection Prevention
        // OWASP A04:2021 – Insecure Design Prevention
        $validated = $request->validate([
            'name' => 'required|string|min:3|max:255|regex:/^[\pL\s\-]+$/u', // Apenas letras, espaços e hífens
            'email' => 'nullable|email:rfc,dns|max:255|unique:empresas',
            'phone' => 'nullable|string|max:20|regex:/^[\d\s\-\(\)\+]+$/', // Apenas números e caracteres de telefone
            'address' => 'nullable|string|max:500',
            'logomarca' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048|dimensions:max_width=4000,max_height=4000',
            'empresa_pai_id' => 'required|exists:empresas,id', // Nova validação: franquia escolhida
        ], [
            'name.regex' => 'O nome deve conter apenas letras, espaços e hífens.',
            'name.min' => 'O nome deve ter no mínimo 3 caracteres.',
            'phone.regex' => 'Formato de telefone inválido.',
            'logomarca.dimensions' => 'A imagem não pode exceder 4000x4000 pixels.',
            'empresa_pai_id.required' => 'Você deve selecionar uma franquia.',
            'empresa_pai_id.exists' => 'Franquia inválida.',
        ]);

        // OWASP A01:2021 – Broken Access Control
        // Buscar franquias do usuário para validação
        $franquiasIds = $user->empresas()
            ->where('tipo_empresa', 'franquia')
            ->pluck('empresas.id')
            ->toArray();

        // Fallback: usar empresa_id legado
        if (empty($franquiasIds) && $user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            $franquiasIds = [$user->empresa_id];
        }

        // Verificar se a franquia escolhida pertence ao usuário
        if (!in_array($validated['empresa_pai_id'], $franquiasIds)) {
            abort(403, 'Você não tem permissão para criar clientes para esta franquia.');
        }

        // OWASP A01:2021 – Broken Access Control Prevention
        // Garantir que apenas dados validados sejam usados
        $data = [
            'name' => strip_tags($validated['name']), // Remove tags HTML
            'email' => isset($validated['email']) ? filter_var($validated['email'], FILTER_SANITIZE_EMAIL) : null,
            'phone' => isset($validated['phone']) ? preg_replace('/[^0-9\s\-\(\)\+]/', '', $validated['phone']) : null,
            'address' => isset($validated['address']) ? strip_tags($validated['address']) : null,
        ];

        // Sempre cria como cliente vinculado à franquia escolhida
        $data['tipo_empresa'] = 'cliente';
        $data['empresa_pai_id'] = $validated['empresa_pai_id']; // Franquia escolhida pelo usuário
        $data['status'] = 0; // Ativo

        // Upload e redimensionamento da logomarca
        if ($request->hasFile('logomarca')) {
            $file = $request->file('logomarca');

            // OWASP A03:2021 – Injection Prevention (Path Traversal)
            // Gerar nome de arquivo seguro (sem usar nome original)
            $extension = $file->getClientOriginalExtension();
            $filename = uniqid('logo_', true) . '_' . time() . '.' . $extension;

            // Redimensionar a imagem para máximo 350px de largura
            $manager = new ImageManager(new Driver());
            $image = $manager->read($file);
            $image->scaleDown(350);

            // Salvar a imagem redimensionada
            $path = 'logos/' . $filename;
            Storage::disk('public')->put($path, $image->encode());

            $data['logomarca'] = $path;
        }

        try {
            $cliente = Empresa::create($data);

            // OWASP A09:2021 – Security Logging and Monitoring
            \Log::info('Cliente criado', [
                'cliente_id' => $cliente->id,
                'cliente_name' => $cliente->name,
                'franquia_id' => $validated['empresa_pai_id'],
                'user_id' => Auth::id(),
                'ip' => $request->ip(),
            ]);

            return redirect()->route('clientes.index')
                            ->with('success', 'Cliente criado com sucesso!');
        } catch (\Exception $e) {
            // OWASP A09:2021 – Security Logging
            \Log::error('Erro ao criar cliente', [
                'error' => $e->getMessage(),
                'user_id' => Auth::id(),
                'ip' => $request->ip(),
            ]);

            return back()->withInput()
                        ->with('error', 'Erro ao criar cliente. Tente novamente.');
        }
    }

    /**
     * Mostra detalhes de um cliente específico
     *
     * @param  \App\Models\Empresa  $cliente
     * @return \Illuminate\View\View
     */
    public function show(Empresa $cliente)
    {
        // Verifica se o cliente pertence à franquia do usuário
        Gate::authorize('view', $cliente);

        // Obter a franquia pai do cliente
        $franquia = $cliente->empresaPai;

        // Buscar todos os usuários do cliente
        $usuarios = $cliente->users()
            ->withTrashed()
            ->orderBy('status_cadastro', 'asc')
            ->orderBy('name', 'asc')
            ->get();

        // Estatísticas de usuários
        $totalUsuarios = $usuarios->count();
        $usuariosAtivos = $usuarios->where('status_cadastro', 'aprovado')->where('deleted_at', null)->count();
        $usuariosPendentes = $usuarios->where('status_cadastro', 'pendente')->count();

        // Estatísticas do cliente
        $totalHotspots = $cliente->hotspots()->count();
        $hotspotsAtivos = $cliente->hotspots()->where('status', 1)->count();

        return view('clientes.show', compact(
            'cliente',
            'franquia',
            'usuarios',
            'totalUsuarios',
            'usuariosAtivos',
            'usuariosPendentes',
            'totalHotspots',
            'hotspotsAtivos'
        ));
    }

    /**
     * Mostra formulário para editar cliente
     *
     * @param  \App\Models\Empresa  $cliente
     * @return \Illuminate\View\View
     */
    public function edit(Empresa $cliente)
    {
        // Verifica se o cliente pertence à franquia do usuário
        Gate::authorize('update', $cliente);

        // Obter a franquia pai do cliente
        $franquia = $cliente->empresaPai;

        return view('clientes.edit', compact('cliente', 'franquia'));
    }

    /**
     * Atualiza dados do cliente
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Empresa  $cliente
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Empresa $cliente)
    {
        // OWASP A01:2021 – Broken Access Control
        // Verifica se o cliente pertence à franquia do usuário
        Gate::authorize('update', $cliente);

        // OWASP A03:2021 – Injection Prevention
        $validated = $request->validate([
            'name' => 'required|string|min:3|max:255|regex:/^[\pL\s\-]+$/u',
            'email' => 'nullable|email:rfc,dns|max:255|unique:empresas,email,' . $cliente->id,
            'phone' => 'nullable|string|max:20|regex:/^[\d\s\-\(\)\+]+$/',
            'address' => 'nullable|string|max:500',
            'status' => 'required|in:0,1',
            'logomarca' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048|dimensions:max_width=4000,max_height=4000',
        ], [
            'name.regex' => 'O nome deve conter apenas letras, espaços e hífens.',
            'phone.regex' => 'Formato de telefone inválido.',
        ]);

        // Sanitizar dados
        $data = [
            'name' => strip_tags($validated['name']),
            'email' => isset($validated['email']) ? filter_var($validated['email'], FILTER_SANITIZE_EMAIL) : null,
            'phone' => isset($validated['phone']) ? preg_replace('/[^0-9\s\-\(\)\+]/', '', $validated['phone']) : null,
            'address' => isset($validated['address']) ? strip_tags($validated['address']) : null,
            'status' => (int) $validated['status'],
        ];

        // Remover logomarca se solicitado
        if ($request->has('remove_logo') && $request->input('remove_logo') == 1) {
            if ($cliente->logomarca && Storage::disk('public')->exists($cliente->logomarca)) {
                Storage::disk('public')->delete($cliente->logomarca);
            }
            $data['logomarca'] = null;
        }

        // Upload e redimensionamento da logomarca
        if ($request->hasFile('logomarca')) {
            // Deletar logomarca anterior se existir
            if ($cliente->logomarca && Storage::disk('public')->exists($cliente->logomarca)) {
                Storage::disk('public')->delete($cliente->logomarca);
            }

            $file = $request->file('logomarca');

            // OWASP A03:2021 – Injection Prevention (Path Traversal)
            // Gerar nome de arquivo seguro (sem usar nome original)
            $extension = $file->getClientOriginalExtension();
            $filename = uniqid('logo_', true) . '_' . time() . '.' . $extension;

            $manager = new ImageManager(new Driver());
            $image = $manager->read($file);
            $image->scaleDown(350);

            $path = 'logos/' . $filename;
            Storage::disk('public')->put($path, $image->encode());

            $data['logomarca'] = $path;
        }

        try {
            $cliente->update($data);

            // OWASP A09:2021 – Security Logging
            \Log::info('Cliente atualizado', [
                'cliente_id' => $cliente->id,
                'cliente_name' => $cliente->name,
                'franquia_id' => Auth::user()->empresa_id,
                'user_id' => Auth::id(),
                'ip' => $request->ip(),
                'changes' => $cliente->getChanges(),
            ]);

            return redirect()->route('clientes.index')
                            ->with('success', 'Cliente atualizado com sucesso!');
        } catch (\Exception $e) {
            \Log::error('Erro ao atualizar cliente', [
                'cliente_id' => $cliente->id,
                'error' => $e->getMessage(),
                'user_id' => Auth::id(),
                'ip' => $request->ip(),
            ]);

            return back()->withInput()
                        ->with('error', 'Erro ao atualizar cliente. Tente novamente.');
        }
    }

    /**
     * Remove cliente (soft delete ou inativa)
     *
     * Obs: Pode ser implementado posteriormente
     *
     * @param  \App\Models\Empresa  $cliente
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Empresa $cliente)
    {
        // Verifica se o cliente pertence à franquia do usuário
        Gate::authorize('delete', $cliente);

        // Verifica se o cliente possui hotspots ativos
        if ($cliente->hotspots()->where('status', 1)->count() > 0) {
            return redirect()->route('clientes.index')
                           ->with('error', 'Não é possível deletar um cliente com hotspots ativos.');
        }

        // Inativa ao invés de deletar
        $cliente->update(['status' => 1]); // 1 = Inativo

        return redirect()->route('clientes.index')
                        ->with('success', 'Cliente inativado com sucesso!');
    }

    /**
     * Aprovar cadastro de cliente (Franqueado OU SuperAdmin)
     */
    public function approve(Empresa $cliente)
    {
        $user = Auth::user();

        // Verificar permissão: SuperAdmin OU Franqueado responsável
        if (!$user->is_superadmin) {
            // Verificar se é franqueado e se o cliente pertence a uma das suas franquias
            $franquiasIds = $user->empresas()->where('tipo_empresa', 'franquia')
                ->pluck('empresas.id')->toArray();

            // Fallback empresa_id legado
            if (empty($franquiasIds) && $user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
                $franquiasIds = [$user->empresa_id];
            }

            if (!in_array($cliente->empresa_pai_id, $franquiasIds)) {
                abort(403, 'Você não tem permissão para aprovar este cliente');
            }
        }

        $cliente->update([
            'status_aprovacao' => 'aprovado',
            'aprovado_por' => $user->id,
            'aprovado_em' => now(),
            'motivo_rejeicao' => null
        ]);

        return redirect()->back()->with('success', 'Cliente aprovado com sucesso!');
    }

    /**
     * Rejeitar cadastro de cliente (Franqueado OU SuperAdmin)
     */
    public function reject(Request $request, Empresa $cliente)
    {
        $user = Auth::user();

        // Verificar permissão: SuperAdmin OU Franqueado responsável
        if (!$user->is_superadmin) {
            // Verificar se é franqueado e se o cliente pertence a uma das suas franquias
            $franquiasIds = $user->empresas()->where('tipo_empresa', 'franquia')
                ->pluck('empresas.id')->toArray();

            // Fallback empresa_id legado
            if (empty($franquiasIds) && $user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
                $franquiasIds = [$user->empresa_id];
            }

            if (!in_array($cliente->empresa_pai_id, $franquiasIds)) {
                abort(403, 'Você não tem permissão para rejeitar este cliente');
            }
        }

        $request->validate([
            'motivo_rejeicao' => 'required|string|max:500'
        ]);

        $cliente->update([
            'status_aprovacao' => 'rejeitado',
            'aprovado_por' => $user->id,
            'aprovado_em' => now(),
            'motivo_rejeicao' => $request->motivo_rejeicao
        ]);

        return redirect()->back()->with('success', 'Cliente rejeitado!');
    }
}
