<?php

namespace App\Http\Controllers;

use App\Models\Midia;
use App\Models\Publicacao;
use App\Models\PublicacaoImpressao;
use App\Models\PublicacaoClique;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Dashboard principal - redireciona baseado no tipo de usuário
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        if ($user->is_superadmin) {
            return $this->dashboardSuperAdmin($user);
        } elseif ($user->empresa && $user->empresa->tipo_empresa === 'cliente') {
            return $this->dashboardCliente($user);
        } elseif ($user->isFranchiseUser()) {
            // Suporta tanto empresa_id (legado) quanto empresa_user (múltiplas franquias)
            return $this->dashboardFranquia($user);
        }

        // Fallback
        return view('dashboard');
    }

    /**
     * Dashboard do Cliente
     * Mostra métricas e performance das publicações
     */
    private function dashboardCliente($user)
    {
        $empresaId = $user->empresa_id;

        // KPI 1: Total de Criativos (próprios + via publicações)
        $totalCriativos = Midia::where(function($q) use ($empresaId) {
            $q->where('empresa_legado_id', $empresaId)
              ->orWhereHas('publicacoes', function($pubQuery) use ($empresaId) {
                  $pubQuery->where('cliente_id', $empresaId);
              });
        })->count();

        // KPI 2: Publicações Ativas
        $publicacoesAtivas = Publicacao::where('cliente_id', $empresaId)
            ->where('status', 'ativa')
            ->count();

        // Investimento Total (soma dos valores das publicações ativas)
        $investimentoTotal = Publicacao::where('cliente_id', $empresaId)
            ->where('status', 'ativa')
            ->sum('valor_contrato');

        // KPI 3: Impressões (últimos 30 dias)
        $impressoes = PublicacaoImpressao::whereHas('publicacao', function($q) use ($empresaId) {
                $q->where('cliente_id', $empresaId);
            })
            ->where('data', '>=', now()->subDays(30))
            ->sum('impressoes');

        // KPI 4: Cliques (últimos 30 dias)
        $cliques = PublicacaoClique::whereHas('publicacao', function($q) use ($empresaId) {
                $q->where('cliente_id', $empresaId);
            })
            ->where('clicked_at', '>=', now()->subDays(30))
            ->count();

        // CTR Geral
        $ctrGeral = $impressoes > 0 ? round(($cliques / $impressoes) * 100, 2) : 0;

        // Top 3 Criativos (baseado em impressões)
        $topCriativos = Midia::select('criativos.*')
            ->selectRaw('COALESCE(SUM(publicacao_impressoes.impressoes), 0) as total_impressoes')
            ->selectRaw('COUNT(DISTINCT publicacao_cliques.id) as total_cliques')
            ->leftJoin('publicacoes', 'criativos.id', '=', 'publicacoes.criativo_id')
            ->leftJoin('publicacao_impressoes', function($join) use ($empresaId) {
                $join->on('publicacoes.id', '=', 'publicacao_impressoes.publicacao_id')
                     ->where('publicacao_impressoes.data', '>=', now()->subDays(30));
            })
            ->leftJoin('publicacao_cliques', function($join) use ($empresaId) {
                $join->on('publicacoes.id', '=', 'publicacao_cliques.publicacao_id')
                     ->where('publicacao_cliques.clicked_at', '>=', now()->subDays(30));
            })
            ->where('publicacoes.cliente_id', $empresaId)
            ->groupBy('criativos.id', 'criativos.title', 'criativos.tipo', 'criativos.path',
                     'criativos.empresa_legado_id', 'criativos.status', 'criativos.created_at',
                     'criativos.updated_at', 'criativos.deleted_at', 'criativos.equipment_serial',
                     'criativos.views_count', 'criativos.position', 'criativos.start_datetime',
                     'criativos.end_datetime', 'criativos.franquia_id', 'criativos.formato',
                     'criativos.conteudo_html', 'criativos.status_aprovacao', 'criativos.aprovado_por',
                     'criativos.aprovado_em', 'criativos.motivo_rejeicao', 'criativos.bloqueado_permanente',
                     'criativos.created_by', 'criativos.updated_by', 'criativos.deleted_by')
            ->orderByDesc('total_impressoes')
            ->limit(3)
            ->get()
            ->map(function($criativo) {
                $criativo->ctr = $criativo->total_impressoes > 0
                    ? round(($criativo->total_cliques / $criativo->total_impressoes) * 100, 2)
                    : 0;
                return $criativo;
            });

        // Status das Publicações
        $statusPublicacoes = Publicacao::select('status', DB::raw('COUNT(*) as total'))
            ->where('cliente_id', $empresaId)
            ->groupBy('status')
            ->pluck('total', 'status')
            ->toArray();

        return view('dashboard-cliente', compact(
            'totalCriativos',
            'publicacoesAtivas',
            'investimentoTotal',
            'impressoes',
            'cliques',
            'ctrGeral',
            'topCriativos',
            'statusPublicacoes'
        ));
    }

    /**
     * Dashboard da Franquia
     * Visão consolidada dos clientes (suporta múltiplas franquias)
     */
    private function dashboardFranquia($user)
    {
        // Buscar IDs de todas as franquias do usuário
        $franquiasIds = $user->empresas()
            ->where('tipo_empresa', 'franquia')
            ->pluck('empresas.id')
            ->toArray();

        // Fallback: usar empresa_id legado se não tiver franquias via pivot
        if (empty($franquiasIds) && $user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            $franquiasIds = [$user->empresa_id];
        }

        // Buscar todas as franquias para passar para a view
        $franquias = Empresa::whereIn('id', $franquiasIds)->get();

        // KPI 1: Total de Clientes (de todas as franquias)
        $totalClientes = Empresa::whereIn('empresa_pai_id', $franquiasIds)
            ->where('tipo_empresa', 'cliente')
            ->count();

        // KPI 2: Total de Publicações (de todos os clientes)
        $totalPublicacoes = Publicacao::whereHas('cliente', function($q) use ($franquiasIds) {
                $q->whereIn('empresa_pai_id', $franquiasIds);
            })
            ->count();

        // KPI 3: Total de Criativos
        $totalCriativos = Midia::where(function($query) use ($franquiasIds) {
                $query->whereIn('empresa_legado_id', $franquiasIds)
                      ->orWhereIn('empresa_legado_id', function($subQuery) use ($franquiasIds) {
                          $subQuery->select('id')
                                ->from('empresas')
                                ->whereIn('empresa_pai_id', $franquiasIds)
                                ->where('tipo_empresa', 'cliente');
                      });
            })
            ->count();

        // KPI 4: Alcance Mensal (impressões)
        $alcanceMensal = PublicacaoImpressao::whereHas('publicacao.cliente', function($q) use ($franquiasIds) {
                $q->whereIn('empresa_pai_id', $franquiasIds);
            })
            ->where('data', '>=', now()->startOfMonth())
            ->sum('usuarios_unicos');

        // ============================================
        // KPIs DE ACESSO WIFI
        // ============================================

        // KPI 5: Conexões Únicas (últimos 7 dias)
        $conexoesUnicas = DB::table('logs_acesso_wifi')
            ->whereIn('franquia_id', $franquiasIds)
            ->where('created_at', '>=', now()->subDays(7))
            ->distinct('mac_address_hash')
            ->count('mac_address_hash');

        // KPI 6: Taxa de Retorno (usuários que voltam)
        $taxaRetorno = DB::table('logs_acesso_wifi')
            ->select(DB::raw('
                (COUNT(DISTINCT CASE WHEN conexoes > 1 THEN mac_address_hash END) /
                 NULLIF(COUNT(DISTINCT mac_address_hash), 0)) * 100 as taxa
            '))
            ->fromSub(function ($query) use ($franquiasIds) {
                $query->select('mac_address_hash', DB::raw('COUNT(*) as conexoes'))
                    ->from('logs_acesso_wifi')
                    ->whereIn('franquia_id', $franquiasIds)
                    ->where('created_at', '>=', now()->subDays(30))
                    ->groupBy('mac_address_hash');
            }, 'subquery')
            ->value('taxa') ?? 0;

        // KPI 7: Tempo Médio de Sessão (em minutos)
        $tempoMedioSessao = DB::table('logs_acesso_wifi')
            ->whereIn('franquia_id', $franquiasIds)
            ->where('created_at', '>=', now()->subDays(7))
            ->where('tempo_sessao', '>', 0)
            ->avg('tempo_sessao');
        $tempoMedioSessao = $tempoMedioSessao ? round($tempoMedioSessao / 60, 1) : 0;

        // KPI 8: % Dispositivos Mobile
        $totalAcessos = DB::table('logs_acesso_wifi')
            ->whereIn('franquia_id', $franquiasIds)
            ->where('created_at', '>=', now()->subDays(7))
            ->count();

        $acessosMobile = DB::table('logs_acesso_wifi')
            ->whereIn('franquia_id', $franquiasIds)
            ->where('created_at', '>=', now()->subDays(7))
            ->where('dispositivo_tipo', 'mobile')
            ->count();

        $percentualMobile = $totalAcessos > 0 ? round(($acessosMobile / $totalAcessos) * 100, 1) : 0;

        // Top 5 Hotspots Mais Ativos
        $topHotspots = DB::table('hotspots as h')
            ->select([
                'h.id',
                'h.serial',
                'e.name as cliente_nome',
                DB::raw('COUNT(DISTINCT l.mac_address_hash) as total_conexoes'),
                DB::raw('ROUND(AVG(l.tempo_sessao) / 60, 1) as tempo_medio'),
                DB::raw('(COUNT(DISTINCT CASE WHEN conexoes.total > 1 THEN l.mac_address_hash END) /
                         NULLIF(COUNT(DISTINCT l.mac_address_hash), 0)) * 100 as taxa_retorno')
            ])
            ->leftJoin('empresas as e', 'h.cliente_id', '=', 'e.id')
            ->leftJoin('logs_acesso_wifi as l', function($join) {
                $join->on('h.id', '=', 'l.hotspot_id')
                     ->where('l.created_at', '>=', now()->subDays(7));
            })
            ->leftJoin(DB::raw('(
                SELECT mac_address_hash, COUNT(*) as total
                FROM logs_acesso_wifi
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
                GROUP BY mac_address_hash
            ) as conexoes'), 'l.mac_address_hash', '=', 'conexoes.mac_address_hash')
            ->whereIn('h.franquia_id', $franquiasIds)
            ->where('h.situacao', 'instalado')
            ->groupBy('h.id', 'h.serial', 'e.name')
            ->orderByDesc('total_conexoes')
            ->limit(5)
            ->get();

        // Status dos Hotspots
        $statusHotspots = DB::table('hotspots')
            ->select([
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(CASE WHEN last_activity_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 1 ELSE 0 END) as online'),
                DB::raw('SUM(CASE WHEN last_activity_at < DATE_SUB(NOW(), INTERVAL 24 HOUR) OR last_activity_at IS NULL THEN 1 ELSE 0 END) as alerta')
            ])
            ->whereIn('franquia_id', $franquiasIds)
            ->where('situacao', 'instalado')
            ->where('status', 0)
            ->first();

        // Conexões por Horário (últimos 7 dias) - Para gráfico
        $conexoesPorHorario = DB::table('logs_acesso_wifi')
            ->select([
                DB::raw('HOUR(created_at) as hora'),
                DB::raw('COUNT(DISTINCT mac_address_hash) as total')
            ])
            ->whereIn('franquia_id', $franquiasIds)
            ->where('created_at', '>=', now()->subDays(7))
            ->groupBy('hora')
            ->orderBy('hora')
            ->get()
            ->pluck('total', 'hora')
            ->toArray();

        // Preencher horas vazias com 0
        $conexoesPorHorarioCompleto = [];
        for ($i = 0; $i < 24; $i++) {
            $conexoesPorHorarioCompleto[$i] = $conexoesPorHorario[$i] ?? 0;
        }

        // ============================================
        // KPIs DE RECEITA
        // ============================================

        // Buscar IDs dos clientes de todas as franquias
        $clientesIds = Empresa::whereIn('empresa_pai_id', $franquiasIds)
            ->where('tipo_empresa', 'cliente')
            ->pluck('id')
            ->toArray();

        // Receita Total (soma de todas as publicações ativas)
        $receitaTotal = Publicacao::whereIn('cliente_id', $clientesIds)
            ->where('status', 'ativa')
            ->sum('valor_contrato');

        // Receita Mensal (publicações do mês atual)
        $receitaMensal = Publicacao::whereIn('cliente_id', $clientesIds)
            ->whereYear('created_at', now()->year)
            ->whereMonth('created_at', now()->month)
            ->sum('valor_contrato');

        // Top 10 Clientes por Receita
        $receitaPorCliente = DB::table('empresas as e')
            ->select([
                'e.id',
                'e.name',
                DB::raw('COUNT(p.id) as total_publicacoes'),
                DB::raw('SUM(CASE WHEN p.status = "ativa" THEN p.valor_contrato ELSE 0 END) as receita_ativa'),
                DB::raw('SUM(p.valor_contrato) as receita_total')
            ])
            ->leftJoin('publicacoes as p', 'e.id', '=', 'p.cliente_id')
            ->whereIn('e.id', $clientesIds)
            ->groupBy('e.id', 'e.name')
            ->orderByDesc('receita_ativa')
            ->limit(10)
            ->get();

        // Ticket Médio
        $totalClientesComPublicacoes = $receitaPorCliente->where('total_publicacoes', '>', 0)->count();
        $ticketMedio = $totalClientesComPublicacoes > 0
            ? $receitaTotal / $totalClientesComPublicacoes
            : 0;

        // Cliente mais rentável
        $clienteMaisRentavel = $receitaPorCliente->first();
        $percentualMaisRentavel = $receitaTotal > 0 && $clienteMaisRentavel
            ? round(($clienteMaisRentavel->receita_ativa / $receitaTotal) * 100, 1)
            : 0;

        return view('dashboard-franquia', compact(
            'totalClientes',
            'totalPublicacoes',
            'totalCriativos',
            'alcanceMensal',
            'franquias',
            // KPIs de Acesso WiFi
            'conexoesUnicas',
            'taxaRetorno',
            'tempoMedioSessao',
            'percentualMobile',
            'topHotspots',
            'statusHotspots',
            'conexoesPorHorarioCompleto',
            // KPIs de Receita
            'receitaTotal',
            'receitaMensal',
            'receitaPorCliente',
            'ticketMedio',
            'clienteMaisRentavel',
            'percentualMaisRentavel'
        ));
    }

    /**
     * Dashboard do SuperAdmin
     * Visão geral do sistema
     */
    private function dashboardSuperAdmin($user)
    {
        // KPIs para SuperAdmin
        $totalEmpresas = Empresa::count();
        $totalPublicacoes = Publicacao::count();
        $totalCriativos = Midia::count();
        $totalHotspots = \App\Models\Hotspot::count();

        return view('dashboard-superadmin', compact(
            'totalEmpresas',
            'totalPublicacoes',
            'totalCriativos',
            'totalHotspots'
        ));
    }
}
