<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class HierarchyController extends Controller
{
    /**
     * Mostra a hierarquia completa da franquia
     * Apenas para franqueados e SuperAdmin
     */
    public function index()
    {
        $user = Auth::user();

        // SuperAdmin vê todas as franquias
        if ($user->is_superadmin) {
            return $this->hierarchySuperAdmin();
        }

        // Franqueado vê sua hierarquia
        if ($user->isFranchiseUser()) {
            return $this->hierarchyFranquia($user);
        }

        // Cliente não tem acesso a hierarquia
        abort(403, 'Você não tem permissão para acessar esta página.');
    }

    /**
     * Hierarquia completa do SuperAdmin
     */
    private function hierarchySuperAdmin()
    {
        // Buscar todas as franquias (exceto matriz)
        $franquias = Empresa::where('tipo_empresa', 'franquia')
            ->where('empresa_pai_id', '!=', null) // Excluir matriz se houver
            ->orWhere(function($q) {
                $q->where('tipo_empresa', 'franquia')
                  ->whereNull('empresa_pai_id');
            })
            ->with(['clientes' => function($query) {
                $query->where('tipo_empresa', 'cliente');
            }])
            ->withCount([
                'clientes' => function($query) {
                    $query->where('tipo_empresa', 'cliente');
                },
                'usuarios'
            ])
            ->orderBy('name')
            ->get();

        // Para cada franquia, buscar usuários dos clientes
        foreach ($franquias as $franquia) {
            foreach ($franquia->clientes as $cliente) {
                $cliente->usuarios = User::where('empresa_id', $cliente->id)->get();
            }
            // Usuários da própria franquia
            $franquia->usuarios_franquia = User::where('empresa_id', $franquia->id)->get();
        }

        return view('hierarchy.index', [
            'tipo' => 'superadmin',
            'franquias' => $franquias,
            'totalFranquias' => $franquias->count(),
            'totalClientes' => $franquias->sum('clientes_count'),
            'totalUsuarios' => User::count(),
        ]);
    }

    /**
     * Hierarquia da franquia específica
     */
    private function hierarchyFranquia($user)
    {
        // Buscar a franquia do usuário
        $franquia = $user->empresa;

        // Se o usuário for multi-franquia, pegar a primeira
        if (!$franquia || $franquia->tipo_empresa !== 'franquia') {
            $franquia = $user->empresas()
                ->where('tipo_empresa', 'franquia')
                ->first();
        }

        if (!$franquia) {
            abort(403, 'Você não está vinculado a nenhuma franquia.');
        }

        // Buscar clientes da franquia
        $clientes = Empresa::where('empresa_pai_id', $franquia->id)
            ->where('tipo_empresa', 'cliente')
            ->withCount('usuarios')
            ->orderBy('name')
            ->get();

        // Para cada cliente, buscar seus usuários
        foreach ($clientes as $cliente) {
            $cliente->usuarios = User::where('empresa_id', $cliente->id)->get();
        }

        // Usuários da própria franquia
        $usuariosFranquia = User::where('empresa_id', $franquia->id)->get();

        // Estatísticas
        $totalClientes = $clientes->count();
        $totalUsuarios = $usuariosFranquia->count() + $clientes->sum('usuarios_count');
        $clientesAtivos = $clientes->where('status_aprovacao', 'aprovado')->count();
        $clientesPendentes = $clientes->where('status_aprovacao', 'pendente')->count();

        return view('hierarchy.index', [
            'tipo' => 'franquia',
            'franquia' => $franquia,
            'clientes' => $clientes,
            'usuariosFranquia' => $usuariosFranquia,
            'totalClientes' => $totalClientes,
            'totalUsuarios' => $totalUsuarios,
            'clientesAtivos' => $clientesAtivos,
            'clientesPendentes' => $clientesPendentes,
        ]);
    }
}
