<?php

namespace App\Http\Controllers;

use App\Models\Hotspot;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;

class HotspotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Hotspot::with(['franquia', 'cliente']);

        // ========================================
        // FILTRO DE HIERARQUIA
        // ========================================
        if (!auth()->user()->is_superadmin) {
            $empresaUsuario = auth()->user()->empresa;

            if ($empresaUsuario) {
                if ($empresaUsuario->tipo_empresa === 'franquia') {
                    // FRANQUEADO: vê hotspots da sua franquia + hotspots dos seus clientes
                    // IDs: franquia + seus clientes
                    $empresasIds = \App\Models\Empresa::where('id', $empresaUsuario->id)
                        ->orWhere('empresa_pai_id', $empresaUsuario->id)
                        ->pluck('id')
                        ->toArray();

                    // Filtra: franquia_id OU cliente_id devem estar na lista
                    $query->where(function($q) use ($empresasIds) {
                        $q->whereIn('franquia_id', $empresasIds)
                          ->orWhereIn('cliente_id', $empresasIds);
                    });
                } else {
                    // CLIENTE: vê apenas hotspots onde cliente_id = sua empresa
                    $query->where('cliente_id', $empresaUsuario->id);
                }
            } else {
                // Se não tem empresa, verificar empresas via pivot (multi-franquia)
                $franquiasIds = auth()->user()->empresas()
                    ->where('tipo_empresa', 'franquia')
                    ->pluck('empresas.id')
                    ->toArray();

                if (!empty($franquiasIds)) {
                    // IDs das franquias + clientes dessas franquias
                    $empresasIds = \App\Models\Empresa::whereIn('id', $franquiasIds)
                        ->orWhereIn('empresa_pai_id', $franquiasIds)
                        ->pluck('id')
                        ->toArray();

                    $query->where(function($q) use ($empresasIds) {
                        $q->whereIn('franquia_id', $empresasIds)
                          ->orWhereIn('cliente_id', $empresasIds);
                    });
                } else {
                    // Sem empresa: não vê nenhum hotspot
                    $query->whereRaw('1 = 0');
                }
            }
        }
        // SuperAdmin vê todos os hotspots (sem filtro)

        // Filtro avançado por tipo
        if ($request->filled('filter_type') && $request->filled('search')) {
            $filterType = $request->input('filter_type');
            $searchTerm = $request->input('search');

            switch ($filterType) {
                case 'franquia':
                    $query->whereHas('franquia', function($q) use ($searchTerm) {
                        $q->where('name', 'like', '%' . $searchTerm . '%');
                    });
                    break;
                case 'cliente':
                    $query->whereHas('cliente', function($q) use ($searchTerm) {
                        $q->where('name', 'like', '%' . $searchTerm . '%');
                    });
                    break;
                case 'marca':
                    $query->where('marca', 'like', '%' . $searchTerm . '%');
                    break;
                case 'modelo':
                    $query->where('modelo', 'like', '%' . $searchTerm . '%');
                    break;
                case 'serial':
                    $query->where('serial', 'like', '%' . $searchTerm . '%');
                    break;
            }
        }

        $hotspots = $query->latest()->paginate(15)->withQueryString();

        return view('hotspots.index', compact('hotspots'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = auth()->user();

        // Filtrar empresas baseado na hierarquia
        if ($user->is_superadmin) {
            // SuperAdmin vê todas as empresas
            $empresas = Empresa::orderBy('name')->get();
        } else {
            $empresaUsuario = $user->empresa;

            if ($empresaUsuario && $empresaUsuario->tipo_empresa === 'franquia') {
                // Franqueado vê: sua franquia + seus clientes
                $empresas = Empresa::where('id', $empresaUsuario->id)
                    ->orWhere('empresa_pai_id', $empresaUsuario->id)
                    ->orderBy('name')
                    ->get();
            } elseif (!$empresaUsuario) {
                // Multi-franquia via pivot
                $franquiasIds = $user->empresas()
                    ->where('tipo_empresa', 'franquia')
                    ->pluck('empresas.id')
                    ->toArray();

                if (!empty($franquiasIds)) {
                    $empresas = Empresa::whereIn('id', $franquiasIds)
                        ->orWhereIn('empresa_pai_id', $franquiasIds)
                        ->orderBy('name')
                        ->get();
                } else {
                    // Sem permissão
                    abort(403, 'Você não tem permissão para criar hotspots.');
                }
            } else {
                // Cliente não pode criar hotspots
                abort(403, 'Apenas franqueados podem criar hotspots.');
            }
        }

        return view('hotspots.create', compact('empresas'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'marca' => 'required|string|max:255',
            'modelo' => 'required|string|max:255',
            'serial' => 'required|string|max:255|unique:hotspots',
            'franquia_id' => 'nullable|exists:empresas,id',
            'cliente_id' => 'nullable|exists:empresas,id',
            'situacao' => 'nullable|in:estoque,em_transito,aguardando_instalacao,instalado,manutencao,desativado',
            'vpn_ip' => 'nullable|ip',
            'vpn_login' => 'nullable|string|max:100',
            'vpn_password' => 'nullable|string|max:255',
        ]);

        $data = $request->all();

        // Preencher automaticamente cod_franquia e cod_cliente baseado nos IDs
        if (!empty($data['franquia_id'])) {
            $franquia = Empresa::find($data['franquia_id']);
            $data['cod_franquia'] = $franquia ? $franquia->codigo : null;
        }

        if (!empty($data['cliente_id'])) {
            $cliente = Empresa::find($data['cliente_id']);
            $data['cod_cliente'] = $cliente ? $cliente->codigo : null;
        }

        // REGRA: Franqueados só podem criar hotspots com situação até "aguardando_instalacao"
        if ($user->isFranchiseUser() && !$user->is_superadmin) {
            $situacoesPermitidas = ['estoque', 'em_transito', 'aguardando_instalacao'];

            if (isset($data['situacao']) && !in_array($data['situacao'], $situacoesPermitidas)) {
                return redirect()->back()
                    ->withErrors(['situacao' => 'Franqueados só podem criar hotspots com situação: Estoque, Em Trânsito ou Aguardando Instalação'])
                    ->withInput();
            }

            // Força status de aprovação como pendente para franqueados
            $data['status_aprovacao'] = 'pendente';
        } else if ($user->is_superadmin) {
            // SuperAdmin pode criar já aprovado
            $data['status_aprovacao'] = $data['status_aprovacao'] ?? 'aprovado';
            $data['aprovado_por'] = $user->id;
            $data['aprovado_em'] = now();
        }

        $data['created_by'] = $user->id;

        Hotspot::create($data);

        $message = $user->isFranchiseUser() && !$user->is_superadmin
            ? 'Hotspot cadastrado com sucesso! Aguardando aprovação do administrador.'
            : 'Hotspot cadastrado com sucesso!';

        return redirect()->route('hotspots.index')->with('success', $message);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Hotspot $hotspot)
    {
        // Verificar autorização
        Gate::authorize('update', $hotspot);

        $user = auth()->user();

        // Filtrar empresas baseado na hierarquia
        if ($user->is_superadmin) {
            // SuperAdmin vê todas as empresas
            $empresas = Empresa::orderBy('name')->get();
        } else {
            $empresaUsuario = $user->empresa;

            if ($empresaUsuario && $empresaUsuario->tipo_empresa === 'franquia') {
                // Franqueado vê: sua franquia + seus clientes
                $empresas = Empresa::where('id', $empresaUsuario->id)
                    ->orWhere('empresa_pai_id', $empresaUsuario->id)
                    ->orderBy('name')
                    ->get();
            } elseif (!$empresaUsuario) {
                // Multi-franquia via pivot
                $franquiasIds = $user->empresas()
                    ->where('tipo_empresa', 'franquia')
                    ->pluck('empresas.id')
                    ->toArray();

                if (!empty($franquiasIds)) {
                    $empresas = Empresa::whereIn('id', $franquiasIds)
                        ->orWhereIn('empresa_pai_id', $franquiasIds)
                        ->orderBy('name')
                        ->get();
                } else {
                    // Sem permissão
                    abort(403, 'Você não tem permissão para editar hotspots.');
                }
            } else {
                // Cliente não pode editar hotspots
                abort(403, 'Apenas franqueados podem editar hotspots.');
            }
        }

        return view('hotspots.edit', compact('hotspot', 'empresas'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Hotspot $hotspot)
    {
        // Verificar autorização
        Gate::authorize('update', $hotspot);

        $user = Auth::user();

        $request->validate([
            'marca' => 'required|string|max:255',
            'modelo' => 'required|string|max:255',
            'serial' => 'required|string|max:255|unique:hotspots,serial,' . $hotspot->id,
            'franquia_id' => 'nullable|exists:empresas,id',
            'cliente_id' => 'nullable|exists:empresas,id',
            'situacao' => 'nullable|in:estoque,em_transito,aguardando_instalacao,instalado,manutencao,desativado',
            'vpn_ip' => 'nullable|ip',
            'vpn_login' => 'nullable|string|max:100',
            'vpn_password' => 'nullable|string|max:255',
        ]);

        $data = $request->all();

        // Preencher automaticamente cod_franquia e cod_cliente baseado nos IDs
        if (!empty($data['franquia_id'])) {
            $franquia = Empresa::find($data['franquia_id']);
            $data['cod_franquia'] = $franquia ? $franquia->codigo : null;
        } else {
            // Se franquia_id foi removido, limpar cod_franquia
            $data['cod_franquia'] = null;
        }

        if (!empty($data['cliente_id'])) {
            $cliente = Empresa::find($data['cliente_id']);
            $data['cod_cliente'] = $cliente ? $cliente->codigo : null;
        } else {
            // Se cliente_id foi removido, limpar cod_cliente
            $data['cod_cliente'] = null;
        }

        // REGRA: Franqueados só podem atualizar hotspots com situação até "aguardando_instalacao"
        if ($user->isFranchiseUser() && !$user->is_superadmin) {
            $situacoesPermitidas = ['estoque', 'em_transito', 'aguardando_instalacao'];

            if (isset($data['situacao']) && !in_array($data['situacao'], $situacoesPermitidas)) {
                return redirect()->back()
                    ->withErrors(['situacao' => 'Franqueados só podem definir situação: Estoque, Em Trânsito ou Aguardando Instalação'])
                    ->withInput();
            }

            // Se alterar algo, volta para pendente
            if ($hotspot->status_aprovacao === 'aprovado') {
                $data['status_aprovacao'] = 'pendente';
                $data['aprovado_por'] = null;
                $data['aprovado_em'] = null;
            }
        }

        $data['updated_by'] = $user->id;

        $hotspot->update($data);

        return redirect()->route('hotspots.index')->with('success', 'Hotspot atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Hotspot $hotspot)
    {
        // Verificar autorização
        Gate::authorize('delete', $hotspot);

        $hotspot->delete();

        return redirect()->route('hotspots.index')->with('success', 'Hotspot excluído com sucesso!');
    }

    /**
     * Toggle status of the hotspot
     */
    public function toggleStatus(Hotspot $hotspot)
    {
        $hotspot->status = !$hotspot->status;
        $hotspot->save();

        return redirect()->route('hotspots.index')->with('success', 'Status alterado com sucesso!');
    }

    /**
     * Update situacao of the hotspot
     */
    public function updateSituacao(Request $request, Hotspot $hotspot)
    {
        $request->validate([
            'situacao' => 'required|in:estoque,em_transito,aguardando_instalacao,instalado,manutencao,desativado',
        ]);

        $hotspot->situacao = $request->situacao;
        $hotspot->save();

        return response()->json([
            'success' => true,
            'message' => 'Situação atualizada com sucesso!',
            'situacao' => $hotspot->situacao
        ]);
    }

    /**
     * Get trashed hotspots
     */
    public function trash()
    {
        $query = Hotspot::onlyTrashed()
                        ->with('destroyer:id,name', 'franquia:id,name', 'cliente:id,name');

        // ========================================
        // FILTRO DE HIERARQUIA (mesmo do index)
        // ========================================
        if (!auth()->user()->is_superadmin) {
            $empresaUsuario = auth()->user()->empresa;

            if ($empresaUsuario) {
                if ($empresaUsuario->tipo_empresa === 'franquia') {
                    // FRANQUEADO: vê hotspots da sua franquia + hotspots dos seus clientes
                    $empresasIds = \App\Models\Empresa::where('id', $empresaUsuario->id)
                        ->orWhere('empresa_pai_id', $empresaUsuario->id)
                        ->pluck('id')
                        ->toArray();

                    $query->where(function($q) use ($empresasIds) {
                        $q->whereIn('franquia_id', $empresasIds)
                          ->orWhereIn('cliente_id', $empresasIds);
                    });
                } else {
                    // CLIENTE: vê apenas hotspots onde cliente_id = sua empresa
                    $query->where('cliente_id', $empresaUsuario->id);
                }
            } else {
                // Multi-franquia via pivot
                $franquiasIds = auth()->user()->empresas()
                    ->where('tipo_empresa', 'franquia')
                    ->pluck('empresas.id')
                    ->toArray();

                if (!empty($franquiasIds)) {
                    $empresasIds = \App\Models\Empresa::whereIn('id', $franquiasIds)
                        ->orWhereIn('empresa_pai_id', $franquiasIds)
                        ->pluck('id')
                        ->toArray();

                    $query->where(function($q) use ($empresasIds) {
                        $q->whereIn('franquia_id', $empresasIds)
                          ->orWhereIn('cliente_id', $empresasIds);
                    });
                } else {
                    // Sem empresa: não vê nenhum hotspot
                    $query->whereRaw('1 = 0');
                }
            }
        }

        $hotspots = $query->latest('deleted_at')->get();
        return response()->json($hotspots);
    }

    /**
     * Restore trashed hotspot
     */
    public function restore($id)
    {
        $hotspot = Hotspot::onlyTrashed()->findOrFail($id);

        // Verificar autorização
        Gate::authorize('restore', $hotspot);

        $hotspot->restore();

        return redirect()->route('hotspots.index')->with('success', 'Hotspot restaurado com sucesso!');
    }

    /**
     * Approve hotspot (SuperAdmin only)
     */
    public function approve(Hotspot $hotspot)
    {
        // Verificar autorização via Policy
        Gate::authorize('approve', Hotspot::class);

        $user = Auth::user();

        $hotspot->update([
            'status_aprovacao' => 'aprovado',
            'aprovado_por' => $user->id,
            'aprovado_em' => now(),
            'motivo_rejeicao' => null
        ]);

        return redirect()->back()->with('success', 'Hotspot aprovado com sucesso!');
    }

    /**
     * Reject hotspot (SuperAdmin only)
     */
    public function reject(Request $request, Hotspot $hotspot)
    {
        // Verificar autorização via Policy
        Gate::authorize('reject', Hotspot::class);

        $user = Auth::user();

        $request->validate([
            'motivo_rejeicao' => 'required|string|max:500'
        ]);

        $hotspot->update([
            'status_aprovacao' => 'rejeitado',
            'aprovado_por' => $user->id,
            'aprovado_em' => now(),
            'motivo_rejeicao' => $request->motivo_rejeicao
        ]);

        return redirect()->back()->with('success', 'Hotspot rejeitado!');
    }
}
