<?php

namespace App\Http\Controllers;

use App\Models\HotspotCustomization;
use App\Models\Hotspot;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class HotspotCustomizationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = HotspotCustomization::with('hotspot');

        // Filtro por hotspot serial
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->whereHas('hotspot', function($q) use ($searchTerm) {
                $q->where('serial', 'like', '%' . $searchTerm . '%')
                  ->orWhere('nome', 'like', '%' . $searchTerm . '%');
            })->orWhere('brand_name', 'like', '%' . $searchTerm . '%');
        }

        // Filtro por status
        if ($request->filled('status')) {
            $status = $request->input('status');
            if ($status === 'active') {
                $query->where('is_active', true);
            } elseif ($status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        $customizations = $query->latest()->paginate(15)->withQueryString();

        return view('customizations.index', compact('customizations'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Buscar hotspots que ainda não têm customização
        $hotspots = Hotspot::whereDoesntHave('customization')
            ->orderBy('serial')
            ->get();

        return view('customizations.create', compact('hotspots'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'hotspot_id' => 'required|exists:hotspots,id|unique:hotspot_customizations',
            'brand_name' => 'nullable|string|max:255',
            'logo_url' => 'nullable|string|max:500',
            'welcome_title' => 'nullable|string|max:500',
            'welcome_subtitle' => 'nullable|string|max:500',
            'primary_color' => 'nullable|string|max:50',
            'primary_hover_color' => 'nullable|string|max:50',
            'bg_color_1' => 'nullable|string|max:100',
            'bg_color_2' => 'nullable|string|max:100',
            'bg_color_3' => 'nullable|string|max:100',
            'bg_color_4' => 'nullable|string|max:100',
            'modal_countdown_seconds' => 'nullable|integer|min:1|max:60',
            'banner_rotation_seconds' => 'nullable|integer|min:1|max:60',
            'status_refresh_seconds' => 'nullable|integer|min:10|max:300',
            'enable_advertisement' => 'boolean',
            'require_name' => 'boolean',
            'require_cpf' => 'boolean',
            'enable_cpf_mask' => 'boolean',
            'show_mac_address' => 'boolean',
            'enable_auto_login' => 'boolean',
            'terms_url' => 'nullable|url|max:500',
            'privacy_url' => 'nullable|url|max:500',
            'custom_css' => 'nullable|string',
            'custom_js' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['created_by'] = Auth::id();
        $validated['updated_by'] = Auth::id();

        HotspotCustomization::create($validated);

        return redirect()
            ->route('customizations.index')
            ->with('success', 'Customização criada com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show(HotspotCustomization $customization)
    {
        $customization->load('hotspot', 'creator', 'updater');

        return view('customizations.show', compact('customization'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(HotspotCustomization $customization)
    {
        $customization->load('hotspot');

        return view('customizations.edit', compact('customization'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, HotspotCustomization $customization)
    {
        $validated = $request->validate([
            'brand_name' => 'nullable|string|max:255',
            'logo_url' => 'nullable|string|max:500',
            'welcome_title' => 'nullable|string|max:500',
            'welcome_subtitle' => 'nullable|string|max:500',
            'primary_color' => 'nullable|string|max:50',
            'primary_hover_color' => 'nullable|string|max:50',
            'bg_color_1' => 'nullable|string|max:100',
            'bg_color_2' => 'nullable|string|max:100',
            'bg_color_3' => 'nullable|string|max:100',
            'bg_color_4' => 'nullable|string|max:100',
            'modal_countdown_seconds' => 'nullable|integer|min:1|max:60',
            'banner_rotation_seconds' => 'nullable|integer|min:1|max:60',
            'status_refresh_seconds' => 'nullable|integer|min:10|max:300',
            'enable_advertisement' => 'boolean',
            'require_name' => 'boolean',
            'require_cpf' => 'boolean',
            'enable_cpf_mask' => 'boolean',
            'show_mac_address' => 'boolean',
            'enable_auto_login' => 'boolean',
            'terms_url' => 'nullable|url|max:500',
            'privacy_url' => 'nullable|url|max:500',
            'custom_css' => 'nullable|string',
            'custom_js' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['updated_by'] = Auth::id();

        $customization->update($validated);

        return redirect()
            ->route('customizations.index')
            ->with('success', 'Customização atualizada com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(HotspotCustomization $customization)
    {
        $customization->delete();

        return redirect()
            ->route('customizations.index')
            ->with('success', 'Customização removida com sucesso!');
    }

    /**
     * Preview customization script
     */
    public function preview(HotspotCustomization $customization)
    {
        $script = $customization->generateLoaderScript();

        return response($script)
            ->header('Content-Type', 'application/javascript; charset=UTF-8');
    }

    /**
     * Toggle active status
     */
    public function toggle(HotspotCustomization $customization)
    {
        $customization->update([
            'is_active' => !$customization->is_active,
            'updated_by' => Auth::id(),
        ]);

        $status = $customization->is_active ? 'ativada' : 'desativada';

        return redirect()
            ->back()
            ->with('success', "Customização {$status} com sucesso!");
    }
}
