<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class MercadoPagoConfigController extends Controller
{
    use AuthorizesRequests;

    /**
     * Exibe o formulário de configuração do Mercado Pago para a empresa
     */
    public function edit(Empresa $empresa)
    {
        // Verificar se usuário tem permissão para editar esta empresa
        $this->authorize('update', $empresa);

        // Buscar possíveis empresas para split (matriz ou franquia pai)
        $empresasSplit = [];

        if ($empresa->tipo_empresa === 'franquia') {
            // Franquia pode fazer split com a matriz
            $empresasSplit = Empresa::where('tipo_empresa', 'matriz')
                ->where('mp_ativo', true)
                ->get();
        } elseif ($empresa->tipo_empresa === 'cliente') {
            // Cliente pode fazer split com sua franquia pai
            if ($empresa->empresa_pai_id) {
                $empresasSplit = Empresa::where('id', $empresa->empresa_pai_id)
                    ->where('mp_ativo', true)
                    ->get();
            }
        }

        return view('empresas.mercadopago-config', compact('empresa', 'empresasSplit'));
    }

    /**
     * Atualiza as configurações do Mercado Pago
     */
    public function update(Request $request, Empresa $empresa)
    {
        $this->authorize('update', $empresa);

        $validated = $request->validate([
            'mp_public_key' => 'nullable|string|max:255',
            'mp_access_token' => 'nullable|string',
            'mp_ativo' => 'boolean',
            'mp_split_habilitado' => 'boolean',
            'mp_percentual_split' => 'nullable|numeric|min:0|max:100',
            'mp_empresa_split_id' => [
                'nullable',
                'exists:empresas,id',
                Rule::requiredIf($request->mp_split_habilitado == true),
            ],
            'mp_ambiente' => ['required', Rule::in(['sandbox', 'production'])],
        ], [
            'mp_public_key.required' => 'A chave pública é obrigatória',
            'mp_access_token.required' => 'O access token é obrigatório',
            'mp_percentual_split.min' => 'O percentual deve ser no mínimo 0',
            'mp_percentual_split.max' => 'O percentual deve ser no máximo 100',
            'mp_empresa_split_id.required' => 'Selecione a empresa que receberá o split',
            'mp_empresa_split_id.exists' => 'Empresa selecionada inválida',
        ]);

        // Validações adicionais
        if ($request->mp_split_habilitado) {
            if (!$request->mp_percentual_split) {
                return back()->withErrors([
                    'mp_percentual_split' => 'O percentual de split é obrigatório quando o split está habilitado',
                ])->withInput();
            }

            // Verificar se a empresa de split tem MP ativo
            $empresaSplit = Empresa::find($request->mp_empresa_split_id);
            if (!$empresaSplit || !$empresaSplit->mp_ativo) {
                return back()->withErrors([
                    'mp_empresa_split_id' => 'A empresa selecionada não possui Mercado Pago ativo',
                ])->withInput();
            }
        }

        // Se desativou o MP, zera o split também
        if (!$request->mp_ativo) {
            $validated['mp_split_habilitado'] = false;
            $validated['mp_percentual_split'] = null;
            $validated['mp_empresa_split_id'] = null;
        }

        // Atualizar empresa
        $empresa->update($validated);

        return redirect()
            ->route('empresas.edit', $empresa)
            ->with('success', 'Configurações do Mercado Pago atualizadas com sucesso!');
    }

    /**
     * Testa a conexão com o Mercado Pago
     */
    public function testarConexao(Empresa $empresa)
    {
        $this->authorize('update', $empresa);

        if (!$empresa->mp_ativo || !$empresa->mp_access_token) {
            return response()->json([
                'success' => false,
                'message' => 'Mercado Pago não está configurado para esta empresa',
            ], 400);
        }

        try {
            \MercadoPago\SDK::setAccessToken($empresa->mp_access_token);

            // Tenta buscar informações da conta
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://api.mercadopago.com/v1/account/credentials');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $empresa->mp_access_token,
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode === 200) {
                return response()->json([
                    'success' => true,
                    'message' => 'Conexão com Mercado Pago estabelecida com sucesso!',
                    'ambiente' => $empresa->mp_ambiente,
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao conectar com Mercado Pago. Verifique as credenciais.',
                ], 400);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao testar conexão: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Desativa o Mercado Pago para a empresa
     */
    public function desativar(Empresa $empresa)
    {
        $this->authorize('update', $empresa);

        $empresa->update([
            'mp_ativo' => false,
            'mp_split_habilitado' => false,
        ]);

        return redirect()
            ->route('empresas.edit', $empresa)
            ->with('success', 'Mercado Pago desativado com sucesso!');
    }
}
