<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PublicClienteRegisterController extends Controller
{
    /**
     * Exibir formulário de cadastro público de cliente
     * Link compartilhado pela franquia
     */
    public function showRegistrationForm($franquiaToken)
    {
        // Buscar franquia pelo token
        $franquia = Empresa::where('token_convite', $franquiaToken)
            ->where('tipo_empresa', 'franquia')
            ->firstOrFail();

        return view('public.register-cliente', compact('franquia'));
    }

    /**
     * Processar cadastro público de cliente
     */
    public function register(Request $request, $franquiaToken)
    {
        // Validar franquia
        $franquia = Empresa::where('token_convite', $franquiaToken)
            ->where('tipo_empresa', 'franquia')
            ->firstOrFail();

        // Validação dos dados
        $request->validate([
            'empresa_nome' => 'required|string|min:3|max:255|regex:/^[a-zA-ZÀ-ÿ0-9\s\-]+$/',
            'empresa_email' => 'nullable|email:rfc,dns|max:255',
            'empresa_phone' => 'nullable|string|max:20|regex:/^[0-9\s\(\)\-\+]+$/',
            'empresa_address' => 'nullable|string|max:500',
            'usuario_nome' => 'required|string|min:3|max:255',
            'usuario_email' => 'required|email:rfc,dns|max:255|unique:users,email',
            'usuario_password' => 'required|string|min:8|confirmed',
        ], [
            'empresa_nome.required' => 'O nome da empresa é obrigatório',
            'usuario_email.unique' => 'Este email já está cadastrado no sistema',
            'usuario_password.min' => 'A senha deve ter no mínimo 8 caracteres',
            'usuario_password.confirmed' => 'As senhas não conferem',
        ]);

        DB::beginTransaction();

        try {
            // Criar empresa cliente (PENDENTE de aprovação)
            $empresa = Empresa::create([
                'name' => $request->empresa_nome,
                'email' => $request->empresa_email,
                'phone' => $request->empresa_phone,
                'address' => $request->empresa_address,
                'tipo_empresa' => 'cliente',
                'empresa_pai_id' => $franquia->id,
                'status' => 0, // Ativo
                'status_aprovacao' => 'pendente', // Aguardando aprovação
            ]);

            // Criar usuário
            $user = User::create([
                'name' => $request->usuario_nome,
                'email' => $request->usuario_email,
                'password' => Hash::make($request->usuario_password),
                'franquia_id;' => $empresa->id,
                'is_superadmin' => false,
            ]);

            DB::commit();

            return redirect()
                ->route('login')
                ->with('success', 'Cadastro realizado com sucesso! Aguarde a aprovação da franquia para acessar o sistema.');

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->withErrors(['error' => 'Erro ao processar cadastro: ' . $e->getMessage()])
                ->withInput();
        }
    }
}
