<?php

namespace App\Http\Controllers;

use App\Models\Publicacao;
use App\Models\PublicacaoImpressao;
use App\Models\PublicacaoClique;
use App\Models\Criativo;
use App\Models\Empresa;
use App\Models\WalledGarden;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Carbon\CarbonPeriod;

/**
 * Controller para gerenciamento de Publicações
 *
 * OWASP A01:2021 - Broken Access Control Prevention
 * Filtros aplicados baseados no tipo de usuário:
 * - SuperAdmin: Vê todas as publicações
 * - Franquia: Vê publicações dos seus clientes
 * - Cliente: Vê apenas publicações atribuídas a ele
 *
 * Atualizado em: 08/11/2025
 */
class PublicacaoController extends Controller
{
    /**
     * Lista publicações com filtros baseados no tipo de usuário
     */
    public function index(Request $request)
    {
        $query = Publicacao::with(['criativo', 'franquia', 'cliente', 'publicadoPor']);

        // OWASP A01:2021 - Filtrar baseado no tipo de usuário
        $user = Auth::user();

        if (!$user->is_superadmin) {
            if ($user->empresa && $user->empresa->tipo_empresa === 'cliente') {
                // CLIENTE: Vê apenas publicações atribuídas a ele
                $query->where('cliente_id', $user->empresa_id);
            } elseif ($user->empresa && $user->empresa->tipo_empresa === 'franquia') {
                // FRANQUIA: Vê publicações dos seus clientes
                $clienteIds = Empresa::where('empresa_pai_id', $user->empresa_id)
                                    ->where('tipo_empresa', 'cliente')
                                    ->pluck('id');
                $query->whereIn('cliente_id', $clienteIds);
            }
        }
        // SuperAdmin vê todas (sem filtro adicional)

        // Filtro por status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filtro por franquia (somente para SuperAdmin)
        if ($request->filled('franquia_id') && $user->is_superadmin) {
            $query->where('franquia_id', $request->franquia_id);
        }

        // Filtro por cliente
        if ($request->filled('cliente_id')) {
            // Validar se usuário pode filtrar por este cliente
            if (!$user->is_superadmin) {
                if ($user->empresa && $user->empresa->tipo_empresa === 'cliente') {
                    // Cliente só pode filtrar por si mesmo
                    if ($request->cliente_id != $user->empresa_id) {
                        $query->where('cliente_id', $user->empresa_id);
                    } else {
                        $query->where('cliente_id', $request->cliente_id);
                    }
                } elseif ($user->empresa && $user->empresa->tipo_empresa === 'franquia') {
                    // Franquia só pode filtrar por seus clientes
                    $isClienteValido = Empresa::where('id', $request->cliente_id)
                                             ->where('empresa_pai_id', $user->empresa_id)
                                             ->where('tipo_empresa', 'cliente')
                                             ->exists();
                    if ($isClienteValido) {
                        $query->where('cliente_id', $request->cliente_id);
                    }
                }
            } else {
                $query->where('cliente_id', $request->cliente_id);
            }
        }

        // Filtro por período
        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->porPeriodo($request->data_inicio, $request->data_fim);
        }

        // Filtro por busca de título
        if ($request->filled('q')) {
            $query->where('titulo', 'like', '%' . $request->q . '%');
        }

        // Publicações vigentes (checkbox)
        if ($request->filled('vigentes')) {
            $query->vigentes();
        }

        $publicacoes = $query->latest()->paginate(15);

        $criativoPaths = $publicacoes->pluck('criativo.path')->filter()->unique();

        if ($criativoPaths->isNotEmpty()) {
            $visualizacoesPorImagem = DB::table('auditoria_splash')
                ->select('imagem', DB::raw('COUNT(*) as total'))
                ->whereIn('imagem', $criativoPaths)
                ->groupBy('imagem')
                ->pluck('total', 'imagem');

            $publicacoes->getCollection()->transform(function ($publicacao) use ($visualizacoesPorImagem) {
                $path = $publicacao->criativo->path ?? null;
                $total = $path ? ($visualizacoesPorImagem[$path] ?? 0) : 0;
                $publicacao->setAttribute('total_visualizacoes_auditoria', $total);
                return $publicacao;
            });
        } else {
            $publicacoes->getCollection()->transform(function ($publicacao) {
                $publicacao->setAttribute('total_visualizacoes_auditoria', 0);
                return $publicacao;
            });
        }

        // Cards de estatísticas
        $stats = [
            'ativas' => Publicacao::ativas()->count(),
            'pausadas' => Publicacao::pausadas()->count(),
            'expiradas' => Publicacao::expiradas()->count(),
            'vigentes' => Publicacao::vigentes()->count(),
            'valor_total_ativas' => Publicacao::ativas()->sum('valor_contrato'),
        ];

        // Listas para filtros
        $franquias = Empresa::where('tipo_empresa', 'franquia')->orderBy('name')->get();
        $clientes = Empresa::where('tipo_empresa', 'cliente')->orderBy('name')->get();

        return view('publicacoes.index', compact('publicacoes', 'stats', 'franquias', 'clientes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Filtrar criativos baseado no tipo de usuário
        $criativos = $this->filtrarCriativosPorUsuario();

        $franquias = Empresa::where('tipo_empresa', 'franquia')->orderBy('name')->get();
        $clientes = Empresa::where('tipo_empresa', 'cliente')->orderBy('name')->get();
        $empresas = Empresa::orderBy('name')->get();

        return view('publicacoes.create', compact('criativos', 'franquias', 'clientes', 'empresas'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'criativo_id' => 'required|exists:criativos,id',
            'franquia_id' => 'required|exists:empresas,id',
            'cliente_id' => 'required|exists:empresas,id',
            'titulo' => 'required|string|max:255',
            'data_inicio' => 'required|date',
            'data_fim' => 'required|date|after:data_inicio',
            'link_destino' => ['nullable', 'max:500', function ($attribute, $value, $fail) {
                if ($value && $value !== '#' && !filter_var($value, FILTER_VALIDATE_URL)) {
                    $fail('O campo Link de Destino deve ser uma URL válida ou #.');
                }
            }],
            'habilitar_clique' => 'nullable|boolean',
            'track_impressoes' => 'nullable|boolean',
            'track_cliques' => 'nullable|boolean',
            'valor_contrato' => 'nullable|numeric|min:0',
            'impressoes_contratadas' => 'nullable|integer|min:0',
            'posicao' => 'required|integer|min:1|max:6',
            'status' => 'required|in:ativa,pausada,expirada,cancelada',
            'tipo_alcance' => 'required|in:todos,serial_especifico,empresa',
            'equipment_serial' => 'nullable|string|max:255|required_if:tipo_alcance,serial_especifico',
            'empresa_alcance_id' => 'nullable|integer|exists:empresas,id|required_if:tipo_alcance,empresa',
            // Walled Gardens
            'walled_gardens' => 'nullable|array',
            'walled_gardens.*.tipo' => 'required|in:domain,ip,subnet',
            'walled_gardens.*.endereco' => 'required|string|max:255',
            'walled_gardens.*.descricao' => 'nullable|string|max:255',
            'walled_gardens.*.ativo' => 'nullable|boolean',
        ]);

        // Buscar o criativo para pegar o tipo
        $criativo = Criativo::findOrFail($validated['criativo_id']);
        $validated['tipo_midia'] = $criativo->tipo ?? $criativo->type ?? 'imagem';
        $validated['total_visualizacoes'] = 0;

        // Validações adicionais
        $this->validateBusinessRules($request);

        // Adicionar dados do usuário logado
        $validated['publicada_por'] = auth()->id();
        $validated['publicada_em'] = now();

        // Usar transação para garantir atomicidade
        DB::beginTransaction();
        try {
            // Criar publicação
            $publicacao = Publicacao::create($validated);

            // Salvar Walled Gardens se houver
            if ($request->has('walled_gardens') && is_array($request->walled_gardens)) {
                foreach ($request->walled_gardens as $wg) {
                    $publicacao->walledGardens()->create([
                        'tipo' => $wg['tipo'],
                        'endereco' => $wg['endereco'],
                        'descricao' => $wg['descricao'] ?? null,
                        'ativo' => $wg['ativo'] ?? true,
                    ]);
                }
            }

            DB::commit();

            return redirect()
                ->route('publicacoes.index')
                ->with('success', 'Publicação criada com sucesso!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Erro ao criar publicação: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Publicacao $publicacao)
    {
        $publicacao->load([
            'criativo',
            'franquia',
            'cliente',
            'publicadoPor',
            'walledGardens',
            'impressoes',
            'cliques'
        ]);

        $impressoesDetalhadas = $publicacao->impressoes()->sum('impressoes');

        $auditoriaQuery = $this->auditoriaSplashQuery($publicacao);
        $visualizacoesLogs = (clone $auditoriaQuery)->count();
        $totalUsuariosLogs = (clone $auditoriaQuery)
            ->distinct('cl_mac_address')
            ->count('cl_mac_address');

        $totalVisualizacoes = $impressoesDetalhadas > 0 ? $impressoesDetalhadas : $visualizacoesLogs;
        $totalCliques = $publicacao->cliques()->count();
        $ctr = $totalVisualizacoes > 0 ? round(($totalCliques / $totalVisualizacoes) * 100, 2) : 0;

        $stats = [
            'total_visualizacoes' => $totalVisualizacoes,
            'total_impressoes' => $totalVisualizacoes,
            'total_cliques' => $totalCliques,
            'ctr' => $ctr,
            'usuarios_unicos' => $impressoesDetalhadas > 0
                ? $publicacao->impressoes()->sum('usuarios_unicos')
                : $totalUsuariosLogs,
            'walled_gardens_ativos' => $publicacao->walledGardens()->where('ativo', true)->count(),
        ];

        return view('publicacoes.show', compact('publicacao', 'stats'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Publicacao $publicacao)
    {
        $publicacao->load('walledGardens', 'criativo');

        // Buscar criativos da tabela correta (criativos, não midias)
        $criativos = \App\Models\Criativo::where('status_aprovacao', 'aprovado')
                          ->where('status', 0)
                          ->orderBy('title')
                          ->get();

        $franquias = Empresa::where('tipo_empresa', 'franquia')->orderBy('name')->get();
        $clientes = Empresa::where('tipo_empresa', 'cliente')->orderBy('name')->get();

        return view('publicacoes.edit', compact('publicacao', 'criativos', 'franquias', 'clientes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Publicacao $publicacao)
    {
        $validated = $request->validate([
            'criativo_id' => 'required|exists:criativos,id',
            'franquia_id' => 'required|exists:empresas,id',
            'cliente_id' => 'required|exists:empresas,id',
            'titulo' => 'required|string|max:255',
            'data_inicio' => 'required|date',
            'data_fim' => 'required|date|after:data_inicio',
            'link_destino' => ['nullable', 'max:500', function ($attribute, $value, $fail) {
                if ($value && $value !== '#' && !filter_var($value, FILTER_VALIDATE_URL)) {
                    $fail('O campo Link de Destino deve ser uma URL válida ou #.');
                }
            }],
            'habilitar_clique' => 'nullable|boolean',
            'track_impressoes' => 'nullable|boolean',
            'track_cliques' => 'nullable|boolean',
            'valor_contrato' => 'nullable|numeric|min:0',
            'impressoes_contratadas' => 'nullable|integer|min:0',
            'posicao' => 'required|integer|min:1|max:6',
            'status' => 'required|in:ativa,pausada,expirada,cancelada',
            'tipo_alcance' => 'required|in:todos,serial_especifico,empresa',
            'equipment_serial' => 'nullable|string|max:255|required_if:tipo_alcance,serial_especifico',
            'empresa_alcance_id' => 'nullable|integer|exists:empresas,id|required_if:tipo_alcance,empresa',
            // Walled Gardens
            'walled_gardens' => 'nullable|array',
            'walled_gardens.*.id' => 'nullable|exists:walled_gardens,id',
            'walled_gardens.*.tipo' => 'required|in:domain,ip,subnet',
            'walled_gardens.*.endereco' => 'required|string|max:255',
            'walled_gardens.*.descricao' => 'nullable|string|max:255',
            'walled_gardens.*.ativo' => 'nullable|boolean',
        ]);

        // Validações adicionais
        $this->validateBusinessRules($request, $publicacao->id);

        // Usar transação para garantir atomicidade
        DB::beginTransaction();
        try {
            // Atualizar publicação
            $publicacao->update($validated);

            // Sincronizar Walled Gardens
            if ($request->has('walled_gardens')) {
                $walledGardenIds = [];

                foreach ($request->walled_gardens as $wgData) {
                    if (isset($wgData['id']) && $wgData['id']) {
                        // Atualizar existente
                        $wg = WalledGarden::find($wgData['id']);
                        if ($wg && $wg->publicacao_id == $publicacao->id) {
                            $wg->update([
                                'tipo' => $wgData['tipo'],
                                'endereco' => $wgData['endereco'],
                                'descricao' => $wgData['descricao'] ?? null,
                                'ativo' => $wgData['ativo'] ?? true,
                            ]);
                            $walledGardenIds[] = $wg->id;
                        }
                    } else {
                        // Criar novo
                        $wg = $publicacao->walledGardens()->create([
                            'tipo' => $wgData['tipo'],
                            'endereco' => $wgData['endereco'],
                            'descricao' => $wgData['descricao'] ?? null,
                            'ativo' => $wgData['ativo'] ?? true,
                        ]);
                        $walledGardenIds[] = $wg->id;
                    }
                }

                // Deletar walled gardens que foram removidos
                $publicacao->walledGardens()
                    ->whereNotIn('id', $walledGardenIds)
                    ->delete();
            } else {
                // Se não enviou nenhum walled garden, deletar todos
                $publicacao->walledGardens()->delete();
            }

            DB::commit();

            return redirect()
                ->route('publicacoes.index')
                ->with('success', 'Publicação atualizada com sucesso!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Erro ao atualizar publicação: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Publicacao $publicacao)
    {
        $publicacao->delete();

        return redirect()
            ->route('publicacoes.index')
            ->with('success', 'Publicação excluída com sucesso!');
    }

    /**
     * Toggle status da publicação (ativa <-> pausada)
     */
    public function toggleStatus(Publicacao $publicacao)
    {
        // Só permite alternar entre ativa e pausada
        if ($publicacao->status === 'ativa') {
            $publicacao->status = 'pausada';
            $message = 'Publicação pausada com sucesso!';
        } elseif ($publicacao->status === 'pausada') {
            $publicacao->status = 'ativa';
            $message = 'Publicação ativada com sucesso!';
        } else {
            return redirect()
                ->back()
                ->with('error', 'Não é possível alterar o status desta publicação.');
        }

        $publicacao->save();

        return redirect()
            ->route('publicacoes.index')
            ->with('success', $message);
    }

    /**
     * Validações de regras de negócio
     */
    private function validateBusinessRules(Request $request, $ignoreId = null)
    {
        // Verificar se o criativo está aprovado
        $criativo = Criativo::find($request->criativo_id);
        if (!$criativo || $criativo->status_aprovacao !== 'aprovado') {
            throw new \Exception('Apenas criativos aprovados podem ser usados em publicações.');
        }

        // Verificar se franquia é realmente franquia
        $franquia = Empresa::find($request->franquia_id);
        if (!$franquia || $franquia->tipo_empresa !== 'franquia') {
            throw new \Exception('A empresa selecionada como franquia deve ser do tipo "franquia".');
        }

        // Verificar se cliente é realmente cliente
        $cliente = Empresa::find($request->cliente_id);
        if (!$cliente || $cliente->tipo_empresa !== 'cliente') {
            throw new \Exception('A empresa selecionada como cliente deve ser do tipo "cliente".');
        }

        // Verificar se as datas não estão no passado (apenas para novas publicações)
        if (!config('features.publicacoes_permitir_data_passada')) { // Feature flag para controlar esta validação
            if (!$ignoreId) {
                $dataInicio = Carbon::parse($request->data_inicio);
                // if ($dataInicio->isPast() && !$dataInicio->isToday()) {
                //     throw new \Exception('A data de início não pode ser no passado.');
                // }
            }
        }
    }

    /**
     * Filtrar criativos baseado no tipo de usuário
     * Super Admin: vê todos
     * Franquia: vê todos (podem vender para qualquer cliente)
     * Cliente: vê apenas seus próprios criativos
     */
    private function filtrarCriativosPorUsuario()
    {
        $user = auth()->user();
        
        $query = Criativo::where('status_aprovacao', 'aprovado')
                      ->where('status', 0) // 0 = ativo, 1 = inativo
                      ->orderBy('title');

        // Se o usuário tem uma empresa associada
        if ($user->empresa_id) {
            $empresa = Empresa::find($user->empresa_id);
            
            // Se for cliente, filtrar apenas criativos da própria empresa
            if ($empresa && $empresa->tipo_empresa === 'cliente') {
                $query->where('empresa_id', $user->empresa_id);
            }
            // Se for franquia, pode ver todos os criativos (vende para qualquer cliente)
            // Se for super admin (sem empresa), pode ver todos
        }

        return $query->get();
    }

    /**
     * API: Incrementar visualizações (usado por equipamentos externos)
     */
    public function incrementarVisualizacoes(Request $request, $id)
    {
        $validated = $request->validate([
            'quantidade' => 'nullable|integer|min:1|max:100',
        ]);

        $publicacao = Publicacao::findOrFail($id);
        $quantidade = $validated['quantidade'] ?? 1;
        
        $publicacao->incrementarVisualizacoes($quantidade);

        return response()->json([
            'success' => true,
            'total_visualizacoes' => $publicacao->fresh()->total_visualizacoes,
            'message' => 'Visualizações incrementadas com sucesso.'
        ]);
    }

    /**
     * API: Buscar dados do criativo (usado no formulário para preview)
     */
    public function buscarCriativo($id)
    {
        $criativo = Criativo::with('franquia')->findOrFail($id);
        
        return response()->json([
            'id' => $criativo->id,
            'title' => $criativo->title,
            'tipo' => $criativo->tipo ?? 'imagem',
            'path' => $criativo->path ? \Storage::url($criativo->path) : null,
            'thumbnail' => $criativo->path ? \Storage::url($criativo->path) : null,
            'empresa' => $criativo->franquia ? $criativo->franquia->name : null,
            'empresa_id' => $criativo->franquia_id,
            'width' => $criativo->largura ?? 640,
            'height' => $criativo->altura ?? 1100,
            'size' => $criativo->tamanho ?? null,
        ]);
    }

    /**
     * Exibir estatísticas detalhadas da publicação
     */
        public function estatisticas(Publicacao $publicacao)
        {
            $publicacao->load(['criativo', 'franquia', 'cliente']);

            $impressoesQuery = PublicacaoImpressao::where('publicacao_id', $publicacao->id);
            $impressoesProcessadas = $impressoesQuery->get();
            $temDadosDetalhados = $impressoesProcessadas->isNotEmpty();

            $auditoriaBaseQuery = $this->auditoriaSplashQuery($publicacao);
            $totalVisualizacoesAggregadas = (clone $auditoriaBaseQuery)->count();
            $totalUsuariosLogs = (clone $auditoriaBaseQuery)
                ->distinct('cl_mac_address')
                ->count('cl_mac_address');

            $totalImpressionsDetalhadas = $impressoesProcessadas->sum('impressoes');
            $totalUsuariosDetalhados = $impressoesProcessadas->sum('usuarios_unicos');

            $cliquesBaseQuery = PublicacaoClique::where('publicacao_id', $publicacao->id);
            $totalCliques = (clone $cliquesBaseQuery)->count();

            $totalImpressions = $temDadosDetalhados ? $totalImpressionsDetalhadas : $totalVisualizacoesAggregadas;
            $totalUsuariosUnicos = $temDadosDetalhados ? $totalUsuariosDetalhados : $totalUsuariosLogs;

            $diasDecorridos = max(1, $publicacao->data_inicio->diffInDays(now()) + 1);

            $stats = [
                'total_impressoes' => $totalImpressions,
                'total_usuarios_unicos' => $totalUsuariosUnicos,
                'total_cliques' => $totalCliques,
                'ctr' => $totalImpressions > 0 ? round(($totalCliques / $totalImpressions) * 100, 2) : 0,
                'media_diaria_impressoes' => $totalImpressions > 0 ? round($totalImpressions / $diasDecorridos, 2) : 0,
                'media_diaria_cliques' => $totalCliques > 0 ? round($totalCliques / $diasDecorridos, 2) : 0,
                'usando_dados_detalhados' => $temDadosDetalhados,
            ];

            $hotspotsAtivosDetalhados = $impressoesProcessadas->pluck('hotspot_id')->filter()->unique()->count();
            $hotspotsAtivos = $hotspotsAtivosDetalhados;
            if ($hotspotsAtivos === 0) {
                $hotspotsAtivos = (clone $cliquesBaseQuery)
                    ->distinct('hotspot_id')
                    ->count('hotspot_id');
            }
            if ($hotspotsAtivos === 0) {
                $hotspotsAtivos = (clone $auditoriaBaseQuery)
                    ->whereNotNull('hs_serial')
                    ->distinct('hs_serial')
                    ->count('hs_serial');
            }
            $stats['hotspots_ativos'] = $hotspotsAtivos;

            $logsPaginados = (clone $auditoriaBaseQuery)
                ->orderBy('created_at', 'desc')
                ->paginate(10, ['*'], 'page_logs');

            $logsAcesso = $logsPaginados->getCollection()->map(function ($log) {
                return [
                    'id' => $log->id,
                    'mac_parcial' => $this->formatarMacParcial($log->cl_mac_address),
                    'mac_address' => $log->cl_mac_address ?? '---',
                    'ip_cliente' => $log->ip_cliente ?? '---',
                    'ip_origem' => $log->ip_origem ?? '---',
                    'dispositivo' => 'Splash Page',
                    'hotspot' => $log->hs_serial ?? 'N/A',
                    'cliente' => $log->dsc_identity ?? 'N/A',
                    'pag_click' => $log->pag_click ?? null,
                    'imagem' => $log->imagem ?? '---',
                    'tempo_sessao' => 0,
                    'created_at' => Carbon::parse($log->created_at)->format('d/m/Y H:i'),
                ];
            })->all();

            $stats['logs_acesso'] = $logsAcesso;
            $stats['total_usuarios_logs'] = $totalUsuariosLogs;

            $inicioJanela = now()->copy()->subDays(29)->startOfDay();
            $fimJanela = now()->copy()->endOfDay();

            $visualizacoesPorDiaCollection = (clone $auditoriaBaseQuery)
                ->whereBetween('created_at', [$inicioJanela, $fimJanela])
                ->selectRaw('DATE(created_at) as data, COUNT(*) as total_visualizacoes')
                ->groupBy('data')
                ->pluck('total_visualizacoes', 'data');

            $impressoesUltimos30 = PublicacaoImpressao::where('publicacao_id', $publicacao->id)
                ->whereBetween('data', [$inicioJanela->toDateString(), $fimJanela->toDateString()])
                ->select('data', DB::raw('SUM(impressoes) as total_impressoes'))
                ->groupBy('data')
                ->pluck('total_impressoes', 'data');

            $cliquesPorDiaMap = (clone $cliquesBaseQuery)
                ->whereBetween('clicked_at', [$inicioJanela, $fimJanela])
                ->selectRaw('DATE(clicked_at) as data, COUNT(*) as total_cliques')
                ->groupBy('data')
                ->pluck('total_cliques', 'data');

            $dadosPorDia = [];
            foreach (CarbonPeriod::create($inicioJanela, '1 day', $fimJanela) as $dia) {
                $dataStr = $dia->toDateString();
                $dadosPorDia[] = [
                    'data' => $dataStr,
                    'visualizacoes' => (int) ($visualizacoesPorDiaCollection[$dataStr] ?? 0),
                    'impressoes' => (int) ($impressoesUltimos30[$dataStr] ?? 0),
                    'cliques' => (int) ($cliquesPorDiaMap[$dataStr] ?? 0),
                ];
            }
            $stats['impressoes_por_dia'] = $dadosPorDia;

            $visualizacoesPorHoraCollection = (clone $auditoriaBaseQuery)
                ->whereBetween('created_at', [$inicioJanela, $fimJanela])
                ->selectRaw('HOUR(created_at) as hora, COUNT(*) as total')
                ->groupBy('hora')
                ->pluck('total', 'hora');

            $stats['visualizacoes_por_hora'] = collect(range(0, 23))
                ->map(fn ($hora) => [
                    'hora' => $hora,
                    'total' => (int) ($visualizacoesPorHoraCollection[$hora] ?? 0),
                ])
                ->values()
                ->toArray();

            $hotspotCliquesPorId = (clone $cliquesBaseQuery)
                ->selectRaw('hotspot_id, COUNT(*) as total_cliques')
                ->groupBy('hotspot_id')
                ->pluck('total_cliques', 'hotspot_id');

            $hotspotCliquesPorSerial = (clone $cliquesBaseQuery)
                ->join('hotspots', 'publicacao_cliques.hotspot_id', '=', 'hotspots.id')
                ->selectRaw('hotspots.serial as serial, COUNT(*) as total_cliques')
                ->groupBy('hotspots.serial')
                ->pluck('total_cliques', 'serial');

            if ($temDadosDetalhados) {
                $topHotspots = PublicacaoImpressao::where('publicacao_id', $publicacao->id)
                    ->join('hotspots', 'publicacao_impressoes.hotspot_id', '=', 'hotspots.id')
                    ->selectRaw('publicacao_impressoes.hotspot_id, hotspots.serial, SUM(publicacao_impressoes.impressoes) as total_visualizacoes')
                    ->groupBy('publicacao_impressoes.hotspot_id', 'hotspots.serial')
                    ->orderByDesc('total_visualizacoes')
                    ->take(5)
                    ->get()
                    ->map(function ($item) use ($hotspotCliquesPorId) {
                        $cliques = $hotspotCliquesPorId[$item->hotspot_id] ?? 0;
                        return (object) [
                            'serial' => $item->serial,
                            'total_visualizacoes' => (int) $item->total_visualizacoes,
                            'total_cliques' => (int) $cliques,
                            'ctr' => $item->total_visualizacoes > 0
                                ? round(($cliques / $item->total_visualizacoes) * 100, 2)
                                : 0,
                        ];
                    });
            } else {
                $topHotspots = (clone $auditoriaBaseQuery)
                    ->selectRaw('hs_serial as serial, COUNT(*) as total_visualizacoes')
                    ->whereNotNull('hs_serial')
                    ->groupBy('serial')
                    ->orderByDesc('total_visualizacoes')
                    ->take(5)
                    ->get()
                    ->map(function ($item) use ($hotspotCliquesPorSerial) {
                        $cliques = $hotspotCliquesPorSerial[$item->serial] ?? 0;
                        return (object) [
                            'serial' => $item->serial,
                            'total_visualizacoes' => (int) $item->total_visualizacoes,
                            'total_cliques' => (int) $cliques,
                            'ctr' => $item->total_visualizacoes > 0
                                ? round(($cliques / $item->total_visualizacoes) * 100, 2)
                                : 0,
                        ];
                    });
            }

            $stats['top_hotspots'] = $topHotspots;

            return view('publicacoes.estatisticas', compact('publicacao', 'stats', 'logsPaginados'));
        }
    /**
     * Formatar MAC address de forma parcial para privacidade
     */
    private function formatarMacParcial($mac)
    {
        if (!$mac) return '***';
        return substr($mac, 0, 8) . '***' . substr($mac, -2);
    }

    /**
     * Base query para auditoria_splash filtrada por publicacao e período
     */
    private function auditoriaSplashQuery(Publicacao $publicacao)
    {
        $query = DB::table('auditoria_splash');

        $criativoPath = $publicacao->criativo->path ?? null;
        if (!$criativoPath) {
            return $query->whereRaw('0 = 1');
        }

        $query->where('imagem', $criativoPath);

        $inicio = $publicacao->data_inicio ? $publicacao->data_inicio->copy()->startOfDay() : null;
        $fim = ($publicacao->data_fim ?? now())->copy()->endOfDay();

        if ($inicio) {
            $query->where('created_at', '>=', $inicio);
        }

        if ($fim) {
            $query->where('created_at', '<=', $fim);
        }

        return $query;
    }

    /**
     * Retorna analytics em JSON para o modal
     */
    public function analyticsJson(Publicacao $publicacao)
    {
        // Estatísticas gerais
        // Priorizar dados detalhados da tabela publicacao_impressoes,
        // mas usar total_visualizacoes como fallback
        $impressoesDetalhadas = $publicacao->impressoes()->sum('impressoes');
        $usarDadosDetalhados = $impressoesDetalhadas > 0;

        $stats = [
            'total_visualizacoes' => $publicacao->total_visualizacoes ?? 0,
            'total_impressoes' => $usarDadosDetalhados ? $impressoesDetalhadas : ($publicacao->total_visualizacoes ?? 0),
            'total_usuarios_unicos' => $publicacao->impressoes()->sum('usuarios_unicos'),
            'total_cliques' => $publicacao->cliques()->count(),
            'ctr' => 0,
            'media_diaria_impressoes' => 0,
            'media_diaria_cliques' => 0,
            'usando_dados_detalhados' => $usarDadosDetalhados,
        ];

        // Calcular CTR
        if ($stats['total_impressoes'] > 0) {
            $stats['ctr'] = round(($stats['total_cliques'] / $stats['total_impressoes']) * 100, 2);
        }

        // Calcular médias diárias
        $diasDecorridos = max(1, $publicacao->data_inicio->diffInDays(now()));
        $stats['media_diaria_impressoes'] = round($stats['total_impressoes'] / $diasDecorridos, 2);
        $stats['media_diaria_cliques'] = round($stats['total_cliques'] / $diasDecorridos, 2);

        // Últimas impressões por dia (últimos 7 dias)
        $impressoesPorDia = $publicacao->impressoes()
            ->selectRaw('data, SUM(impressoes) as total_impressoes, SUM(usuarios_unicos) as total_usuarios')
            ->where('data', '>=', now()->subDays(7))
            ->groupBy('data')
            ->orderBy('data', 'desc')
            ->get();

        // Logs de acesso recentes (últimos 20 usuários da auditoria)
        $logsAcesso = $this->auditoriaSplashQuery($publicacao)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function($log) {
                // Simular a formatação do dispositivo que estaria no model LogAcessoWifi
                // Esta lógica pode ser mais elaborada se o user_agent estiver disponível
                $dispositivo = 'Computador'; // fallback
                if (isset($log->user_agent)) {
                    if (preg_match('/(mobile|android|iphone)/i', $log->user_agent)) {
                        $dispositivo = 'Celular';
                    }
                }

                return [
                    'id' => $log->id,
                    'mac_parcial' => $this->formatarMacParcial($log->cl_mac_address),
                    'ip_address' => $log->cl_ip_address ?? 'N/A',
                    'dispositivo' => $dispositivo,
                    'hotspot' => $log->hs_serial ?? 'N/A',
                    'pag_click' => $log->pag_click ?? null,
                    'tempo_sessao' => 0, // Não disponível na auditoria_splash
                    'created_at' => Carbon::parse($log->created_at)->format('d/m/Y H:i'),
                ];
            });

        $totalLogs = $this->auditoriaSplashQuery($publicacao)->count();

        return response()->json([
            'publicacao' => [
                'id' => $publicacao->id,
                'titulo' => $publicacao->titulo,
                'status' => $publicacao->status,
                'data_inicio' => $publicacao->data_inicio->format('d/m/Y'),
                'data_fim' => $publicacao->data_fim->format('d/m/Y'),
            ],
            'stats' => $stats,
            'impressoes_recentes' => $impressoesPorDia,
            'logs_acesso' => $logsAcesso,
            'total_logs' => $totalLogs,
        ]);
    }

    /**
     * API: Registrar impressão (usado por equipamentos externos)
     */
    public function registrarImpressao(Request $request)
    {
        $validated = $request->validate([
            'hotspot_id' => 'required|exists:hotspots,id',
            'publicacao_id' => 'required|exists:publicacoes,id',
            'criativo_id' => 'required|exists:criativos,id',
            'usuarios_unicos' => 'nullable|integer|min:1',
        ]);

        $publicacao = Publicacao::find($validated['publicacao_id']);

        // Verificar se tracking de impressões está habilitado
        if (!$publicacao->track_impressoes) {
            return response()->json([
                'success' => false,
                'message' => 'Tracking de impressões desabilitado para esta publicação.'
            ], 400);
        }

        $data = now()->format('Y-m-d');

        // Buscar ou criar registro de impressão do dia
        $impressao = \App\Models\PublicacaoImpressao::firstOrNew([
            'hotspot_id' => $validated['hotspot_id'],
            'publicacao_id' => $validated['publicacao_id'],
            'criativo_id' => $validated['criativo_id'],
            'data' => $data,
        ]);

        $impressao->impressoes = ($impressao->impressoes ?? 0) + 1;
        $impressao->usuarios_unicos = max(
            $impressao->usuarios_unicos ?? 0,
            $validated['usuarios_unicos'] ?? 1
        );
        $impressao->save();

        return response()->json([
            'success' => true,
            'message' => 'Impressão registrada com sucesso.',
            'data' => [
                'total_hoje' => $impressao->impressoes,
                'usuarios_unicos' => $impressao->usuarios_unicos,
            ]
        ]);
    }

    /**
     * API: Registrar clique (usado por equipamentos externos)
     */
    public function registrarClique(Request $request)
    {
        $validated = $request->validate([
            'hotspot_id' => 'required|exists:hotspots,id',
            'publicacao_id' => 'required|exists:publicacoes,id',
            'criativo_id' => 'required|exists:criativos,id',
            'ip_usuario' => 'required|ip',
            'user_agent' => 'nullable|string',
        ]);

        $publicacao = Publicacao::find($validated['publicacao_id']);

        // Verificar se tracking de cliques está habilitado
        if (!$publicacao->track_cliques) {
            return response()->json([
                'success' => false,
                'message' => 'Tracking de cliques desabilitado para esta publicação.'
            ], 400);
        }

        // Verificar se link de destino está habilitado
        if (!$publicacao->habilitar_clique || !$publicacao->link_destino) {
            return response()->json([
                'success' => false,
                'message' => 'Clique não habilitado para esta publicação.'
            ], 400);
        }

        // Registrar clique
        $clique = \App\Models\PublicacaoClique::create([
            'hotspot_id' => $validated['hotspot_id'],
            'publicacao_id' => $validated['publicacao_id'],
            'criativo_id' => $validated['criativo_id'],
            'ip_usuario' => $validated['ip_usuario'],
            'user_agent' => $validated['user_agent'] ?? null,
            'clicked_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Clique registrado com sucesso.',
            'redirect_url' => $publicacao->link_destino,
            'data' => [
                'clique_id' => $clique->id,
                'timestamp' => $clique->clicked_at->toIso8601String(),
            ]
        ]);
    }
}
