<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use App\Models\TawktoConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TawktoConfigController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        // Buscar configurações baseado no tipo de usuário
        $query = TawktoConfig::with('empresa');

        if ($user->tipo === 'franquia') {
            // Franquia vê apenas suas configs e de seus clientes
            $empresasIds = Empresa::where('id', $user->empresa_id)
                ->orWhere('empresa_pai_id', $user->empresa_id)
                ->pluck('id');
            $query->whereIn('empresa_id', $empresasIds);
        } elseif ($user->tipo === 'cliente') {
            // Cliente vê apenas sua config
            $query->where('empresa_id', $user->empresa_id);
        }
        // Super admin vê tudo

        $configs = $query->latest()->paginate(15);

        return view('tawkto.index', compact('configs'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();

        // Buscar empresas disponíveis baseado no tipo de usuário
        $empresas = $this->getEmpresasDisponiveis();

        // Páginas disponíveis para habilitar
        $paginasDisponiveis = [
            'index' => 'Página Inicial',
            'cadastro' => 'Cadastro',
            'login' => 'Login',
            'welcome' => 'Boas-vindas'
        ];

        return view('tawkto.create', compact('empresas', 'paginasDisponiveis'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        \Log::info('=== INÍCIO STORE TAWKTO ===');
        \Log::info('Request data: ' . json_encode($request->all()));

        try {
            \Log::info('Validando request...');

            $request->validate([
                'empresa_id' => 'required|exists:empresas,id',
                'property_id' => 'required|string|max:100',
                'widget_id' => 'required|string|max:100',
                'paginas_habilitadas' => 'nullable|array',
                'paginas_habilitadas.*' => 'in:index,cadastro,login,welcome',
            ]);

            \Log::info('Validação OK! Verificando duplicata...');

            // Verificar se já existe config para esta empresa
            $exists = TawktoConfig::where('empresa_id', $request->empresa_id)->exists();
            if ($exists) {
                \Log::warning('Já existe config para empresa_id: ' . $request->empresa_id);
                return back()->withErrors(['empresa_id' => 'Já existe uma configuração Tawk.to para esta empresa.'])->withInput();
            }

            \Log::info('Não há duplicata. Criando novo registro...');

            $config = new TawktoConfig();
            $config->empresa_id = $request->empresa_id;
            $config->property_id = $request->property_id;
            $config->widget_id = $request->widget_id;
            $config->ativo = $request->has('ativo');
            $config->paginas_habilitadas = $request->paginas_habilitadas ?? [];
            $config->customizacao = [
                'primaryColor' => $request->input('customizacao.primaryColor', '#3B87C6'),
                'position' => $request->input('customizacao.position', 'br'),
            ];
            $config->metadata_personalizado = $request->metadata_personalizado ?? [];

            \Log::info('Objeto criado. Salvando no banco...');
            \Log::info('Config antes do save: ' . json_encode($config->toArray()));

            $saved = $config->save();

            \Log::info('Resultado do save: ' . ($saved ? 'TRUE' : 'FALSE'));
            \Log::info('ID gerado: ' . $config->id);
            \Log::info('Total de configs no banco: ' . TawktoConfig::count());

            \Log::info('Redirecionando...');

            return redirect()
                ->route('tawkto.index')
                ->with('success', 'Configuração Tawk.to criada com sucesso!');

        } catch (\Exception $e) {
            \Log::error('ERRO AO CRIAR TAWKTO: ' . $e->getMessage());
            \Log::error('Trace: ' . $e->getTraceAsString());

            return back()
                ->withErrors(['error' => 'Erro ao salvar: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(TawktoConfig $tawkto)
    {
        $this->authorize('view', $tawkto);

        return view('tawkto.show', compact('tawkto'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(TawktoConfig $tawkto)
    {
        $this->authorize('update', $tawkto);

        $empresas = $this->getEmpresasDisponiveis();

        $paginasDisponiveis = [
            'index' => 'Página Inicial',
            'cadastro' => 'Cadastro',
            'login' => 'Login',
            'welcome' => 'Boas-vindas'
        ];

        return view('tawkto.edit', compact('tawkto', 'empresas', 'paginasDisponiveis'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, TawktoConfig $tawkto)
    {
        $this->authorize('update', $tawkto);

        $request->validate([
            'property_id' => 'required|string|max:100',
            'widget_id' => 'required|string|max:100',
            'ativo' => 'boolean',
            'paginas_habilitadas' => 'nullable|array',
            'paginas_habilitadas.*' => 'in:index,cadastro,login,welcome',
        ]);

        $tawkto->update([
            'property_id' => $request->property_id,
            'widget_id' => $request->widget_id,
            'ativo' => $request->has('ativo'),
            'paginas_habilitadas' => $request->paginas_habilitadas ?? [],
            'customizacao' => [
                'primaryColor' => $request->input('customizacao.primaryColor', '#3B87C6'),
                'position' => $request->input('customizacao.position', 'br'),
            ],
            'metadata_personalizado' => $request->metadata_personalizado ?? [],
        ]);

        return redirect()
            ->route('tawkto.index')
            ->with('success', 'Configuração Tawk.to atualizada com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(TawktoConfig $tawkto)
    {
        $this->authorize('delete', $tawkto);

        $tawkto->delete();

        return redirect()
            ->route('tawkto.index')
            ->with('success', 'Configuração Tawk.to removida com sucesso!');
    }

    /**
     * Buscar empresas disponíveis baseado no tipo de usuário
     * Organizado por tipo (franquias primeiro, depois clientes)
     */
    private function getEmpresasDisponiveis()
    {
        $user = Auth::user();

        if ($user->is_superadmin || $user->tipo === 'super_admin') {
            // Super Admin vê todas as empresas, organizadas por tipo
            $empresas = Empresa::orderBy('tipo_empresa')
                ->orderBy('name')
                ->get();

            // Agrupar por tipo para facilitar a renderização com optgroups
            return $empresas->groupBy('tipo_empresa');
        } elseif ($user->tipo === 'franquia') {
            // Franquia e seus clientes
            $empresas = Empresa::where('id', $user->empresa_id)
                ->orWhere('empresa_pai_id', $user->empresa_id)
                ->orderBy('tipo_empresa')
                ->orderBy('name')
                ->get();

            return $empresas->groupBy('tipo_empresa');
        } else {
            // Cliente vê apenas sua empresa
            $empresas = Empresa::where('id', $user->empresa_id)->get();
            return $empresas->groupBy('tipo_empresa');
        }
    }
}
