<?php

namespace App\Http\Controllers;

use App\Mail\CadastroAprovado;
use App\Mail\CadastroRejeitado;
use App\Models\Group;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::with('empresa');
        
        // Busca por nome ou email
        if ($request->has('q') && $request->q != '') {
            $search = $request->q;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }
        
        $users = $query->paginate(10);
        return view('users.index', compact('users'));
    }
    
    /**
     * Retorna usuários na lixeira (soft deleted)
     */
    public function trash()
    {
        $users = User::onlyTrashed()
            ->with(['destroyer'])
            ->orderBy('deleted_at', 'desc')
            ->get();
        
        return response()->json($users);
    }
    
    /**
     * Restaura um usuário da lixeira
     */
    public function restore($id)
    {
        $user = User::onlyTrashed()->findOrFail($id);
        $user->restore();
        
        return redirect()->route('users.index')
            ->with('success', 'Usuário restaurado com sucesso!');
    }
    
    /**
     * Aprova um cadastro de usuário
     */
    public function approve(User $user)
    {
        if ($user->status_cadastro === 'aprovado') {
            return redirect()->route('users.index')
                ->with('warning', 'Este usuário já está aprovado!');
        }
        
        $user->update(['status_cadastro' => 'aprovado']);
        
        // Verificar feature flags antes de enviar email
        if ($this->shouldSendEmail('cadastro_aprovado')) {
            $recipient = $this->getEmailRecipient($user->email);
            
            if (Config::get('features.dev_mode')) {
                // Modo desenvolvimento - apenas loga
                Log::info('Email de aprovação (dev mode)', [
                    'tipo' => 'CadastroAprovado',
                    'destinatario' => $user->email,
                    'usuario' => $user->name,
                ]);
            } else {
                // Enviar email real (com queue para não bloquear)
                Mail::to($recipient)->queue(new CadastroAprovado($user));
            }
            
            $emailMsg = Config::get('features.dev_mode') 
                ? ' (email logado em modo desenvolvimento)' 
                : ' Um e-mail de confirmação foi enviado.';
        } else {
            $emailMsg = ' (envio de email desabilitado)';
        }
        
        return redirect()->route('users.index')
            ->with('success', "Cadastro de {$user->name} aprovado com sucesso!{$emailMsg}");
    }
    
    /**
     * Rejeita um cadastro de usuário
     */
    public function reject(User $user, Request $request)
    {
        if ($user->status_cadastro === 'rejeitado') {
            return redirect()->route('users.index')
                ->with('warning', 'Este usuário já está rejeitado!');
        }
        
        $user->update(['status_cadastro' => 'rejeitado']);
        
        // Verificar feature flags antes de enviar email
        $motivo = $request->input('motivo', null);
        
        if ($this->shouldSendEmail('cadastro_rejeitado')) {
            $recipient = $this->getEmailRecipient($user->email);
            
            if (Config::get('features.dev_mode')) {
                // Modo desenvolvimento - apenas loga
                Log::info('Email de rejeição (dev mode)', [
                    'tipo' => 'CadastroRejeitado',
                    'destinatario' => $user->email,
                    'usuario' => $user->name,
                    'motivo' => $motivo,
                ]);
            } else {
                // Enviar email real com motivo (se fornecido)
                Mail::to($recipient)->queue(new CadastroRejeitado($user, $motivo));
            }
            
            $emailMsg = Config::get('features.dev_mode') 
                ? ' (email logado em modo desenvolvimento)' 
                : ' Um e-mail foi enviado ao usuário.';
        } else {
            $emailMsg = ' (envio de email desabilitado)';
        }
        
        return redirect()->route('users.index')
            ->with('info', "Cadastro de {$user->name} foi rejeitado.{$emailMsg}");
    }
    
    /**
     * Verifica se o envio de email está habilitado para um tipo específico
     */
    protected function shouldSendEmail(string $tipo): bool
    {
        // Verificar flag mestre primeiro
        if (!Config::get('features.email_enabled', true)) {
            return false;
        }
        
        // Verificar flag específica do tipo
        return Config::get("features.emails.{$tipo}", true);
    }
    
    /**
     * Retorna o destinatário do email (real ou de teste)
     */
    protected function getEmailRecipient(string $defaultEmail): string
    {
        // Se há email de teste configurado, redirecionar para ele
        $testRecipient = Config::get('features.test_recipient');
        
        if (!empty($testRecipient)) {
            Log::info('Email redirecionado para teste', [
                'original' => $defaultEmail,
                'teste' => $testRecipient,
            ]);
            
            return $testRecipient;
        }
        
        return $defaultEmail;
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('users.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('users.index');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $groups = Group::all();
        return view('users.edit', compact('user', 'groups'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,'.$user->id],
            'password' => ['nullable', 'string', 'min:8', 'confirmed'],
        ]);

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'password' => $request->password ? Hash::make($request->password) : $user->password,
        ]);

        $user->groups()->sync($request->groups);

        return redirect()->route('users.index');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $user->delete();

        return redirect()->route('users.index');
    }
}
