<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckApprovalStatus
{
    /**
     * Handle an incoming request.
     *
     * Bloqueia usuários com status_cadastro diferente de 'aprovado'
     * 
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Verificar se o usuário está autenticado
        if (Auth::check()) {
            $user = Auth::user();
            
            // Se for super admin, permitir acesso irrestrito
            if ($user->is_superadmin) {
                return $next($request);
            }

            // Verificar status de aprovação
            if ($user->status_cadastro !== 'aprovado') {
                // Fazer logout imediatamente
                Auth::logout();
                
                // Invalidar sessão
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                
                // Mensagens personalizadas por status
                if ($user->status_cadastro === 'pendente') {
                    $message = 'Seu cadastro está aguardando aprovação. Você receberá um email quando for aprovado.';
                } elseif ($user->status_cadastro === 'rejeitado') {
                    $message = 'Seu cadastro foi rejeitado. Entre em contato com o administrador para mais informações.';
                } else {
                    $message = 'Você não tem permissão para acessar o sistema.';
                }
                
                // Redirecionar para login com mensagem de erro
                return redirect()->route('login')
                    ->withErrors(['email' => $message])
                    ->withInput(['email' => $user->email]);
            }
        }
        
        return $next($request);
    }
}
