<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckEmpresaAprovada
{
    /**
     * Verificar se a empresa do usuário está aprovada
     *
     * Regras:
     * - SuperAdmin: sempre pode acessar
     * - Franqueados: sempre podem acessar (empresa tipo 'franquia')
     * - Clientes: só podem acessar se empresa estiver aprovada
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = Auth::user();

        // SuperAdmin sempre passa
        if ($user->is_superadmin) {
            return $next($request);
        }

        // Verificar se tem empresa vinculada
        if (!$user->empresa_id || !$user->empresa) {
            // Se for franqueado multi-franquia (sem empresa_id mas com pivot)
            if ($user->isFranchiseUser()) {
                return $next($request);
            }

            // Usuário sem empresa e não é franqueado
            Auth::logout();
            return redirect()->route('login')
                ->withErrors(['email' => 'Usuário sem empresa vinculada. Entre em contato com o suporte.']);
        }

        $empresa = $user->empresa;

        // Franquias e Matriz sempre podem acessar
        if (in_array($empresa->tipo_empresa, ['franquia', 'matriz'])) {
            return $next($request);
        }

        // CLIENTES: verificar status de aprovação
        if ($empresa->tipo_empresa === 'cliente') {

            // Verificar status
            if ($empresa->status_aprovacao === 'pendente') {
                Auth::logout();
                return redirect()->route('login')
                    ->withErrors([
                        'email' => 'Seu cadastro está aguardando aprovação. Você receberá um e-mail quando for aprovado.'
                    ]);
            }

            if ($empresa->status_aprovacao === 'rejeitado') {
                $motivo = $empresa->motivo_rejeicao
                    ? 'Motivo: ' . $empresa->motivo_rejeicao
                    : 'Entre em contato com a franquia responsável.';

                Auth::logout();
                return redirect()->route('login')
                    ->withErrors([
                        'email' => 'Seu cadastro foi rejeitado. ' . $motivo
                    ]);
            }

            // Se chegou aqui, empresa está aprovada
            if ($empresa->status_aprovacao === 'aprovado') {
                return $next($request);
            }

            // Status desconhecido
            Auth::logout();
            return redirect()->route('login')
                ->withErrors(['email' => 'Status de aprovação inválido. Entre em contato com o suporte.']);
        }

        // Tipo de empresa desconhecido
        return $next($request);
    }
}
