<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

/**
 * Middleware para restringir acesso apenas a usuários de clientes
 *
 * Este middleware valida:
 * 1. Usuário está autenticado
 * 2. Usuário NÃO é SuperAdmin
 * 3. Usuário está vinculado a uma empresa
 * 4. Empresa é do tipo 'cliente'
 *
 * Criado em: 08/11/2025
 * OWASP A01:2021 - Broken Access Control Prevention
 */
class ClientOnly
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // OWASP A01:2021 - Verificar autenticação
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Você precisa estar autenticado.');
        }

        $user = Auth::user();

        // SuperAdmin não acessa rotas de cliente (tem acesso a tudo via outros módulos)
        if ($user->is_superadmin) {
            return redirect()->route('empresas.index')
                ->with('info', 'SuperAdmin deve usar o módulo de Empresas.');
        }

        // Usuário deve estar vinculado a uma empresa
        if (!$user->empresa_id) {
            abort(403, 'Você não está vinculado a nenhuma empresa.');
        }

        // Carregar empresa se ainda não carregada
        if (!$user->empresa) {
            abort(403, 'Empresa não encontrada.');
        }

        // OWASP A01:2021 - Verificar tipo de empresa
        if ($user->empresa->tipo_empresa !== 'cliente') {
            // Se for franquia, redirecionar para clientes
            if ($user->empresa->tipo_empresa === 'franquia') {
                return redirect()->route('clientes.index')
                    ->with('info', 'Usuários de franquia devem usar o módulo Meus Clientes.');
            }

            abort(403, 'Esta área é exclusiva para usuários de clientes.');
        }

        return $next($request);
    }
}
