<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

/**
 * Middleware para verificar acesso de usuários de franquia
 * Suporta usuários com múltiplas franquias (many-to-many via empresa_user)
 * e também usuários com franquia única (legado via empresa_id)
 */
class FranchiseMultipleAccess
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // 1. Verificar se está autenticado
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Você precisa estar autenticado.');
        }

        $user = Auth::user();

        // 2. SuperAdmin não deve acessar rotas de franquia (tem rotas próprias)
        if ($user->is_superadmin) {
            return redirect()->route('empresas.index')
                ->with('info', 'SuperAdmins devem usar o módulo de Empresas.');
        }

        // 3. Verificar se é usuário de franquia (novo método: many-to-many)
        $franquiasIds = $user->empresas()
            ->where('tipo_empresa', 'franquia')
            ->pluck('empresas.id')
            ->toArray();

        // 4. Fallback: verificar empresa_id legado (para retrocompatibilidade)
        if (empty($franquiasIds)) {
            if ($user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'franquia') {
                // Usuário tem franquia via campo legado empresa_id
                // Permitir acesso
                return $next($request);
            }

            // 5. Não é de franquia nem tem vinculação via pivot
            // Verificar se é cliente
            if ($user->empresa_id && $user->empresa && $user->empresa->tipo_empresa === 'cliente') {
                return redirect()->route('dashboard')
                    ->with('error', 'Área restrita para franquias. Usuários de clientes não têm acesso.');
            }

            // 6. Usuário sem vinculação de empresa
            abort(403, 'Você não está vinculado a nenhuma franquia. Entre em contato com o administrador.');
        }

        // 7. Usuário tem pelo menos uma franquia vinculada via pivot
        // Permitir acesso
        return $next($request);
    }
}
