<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

/**
 * Middleware para garantir que apenas usuários de franquia acessem determinadas rotas
 *
 * Usuário de Franquia:
 * - is_superadmin = 0
 * - empresa_id = franquia (tipo_empresa = 'franquia')
 * - status_cadastro = 'aprovado'
 *
 * Características:
 * - Gerencia seus clientes
 * - Upload de criativos (aguarda aprovação)
 * - NÃO vê outras franquias
 *
 * Criado em: 08/11/2025
 * Documentação: tests/documentation/ACESSO-USUARIO-FRANQUIA-08-11-2025.md
 */
class FranchiseOnly
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Verifica se o usuário está autenticado
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        // SuperAdmin não acessa rotas de franquia (tem rotas próprias)
        if (Auth::user()->is_superadmin) {
            return redirect()->route('empresas.index')
                           ->with('info', 'SuperAdministradores devem usar o menu "Empresas".');
        }

        $user = Auth::user();

        // Verifica se é usuário multi-franquia (sem empresa_id, mas com franquias via pivot)
        if (!$user->empresa_id) {
            // Verificar se tem franquias vinculadas via pivot table
            if ($user->isFranchiseUser()) {
                return $next($request);
            }
            abort(403, 'Acesso negado. Você não está vinculado a nenhuma empresa.');
        }

        // Verifica se a empresa vinculada existe
        $empresa = $user->empresa;
        if (!$empresa) {
            abort(403, 'Acesso negado. Empresa vinculada não encontrada.');
        }

        // Verifica se a empresa é uma franquia
        if ($empresa->tipo_empresa !== 'franquia') {
            abort(403, 'Acesso negado. Esta área é exclusiva para usuários de franquia.');
        }

        return $next($request);
    }
}
