<?php

namespace App\Http\Requests\Auth;

use App\Models\Empresa;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class RegisterRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true; // Qualquer um pode tentar se registrar
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation()
    {
        // Converter código para maiúsculo
        if ($this->has('codigo_convite')) {
            $this->merge([
                'codigo_convite' => strtoupper($this->codigo_convite),
            ]);
        }
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'codigo_convite' => [
                'required',
                'string',
                'size:10',
                Rule::exists('empresas', 'codigo_convite')->where(function ($query) {
                    $query->where('tipo_empresa', 'franquia')
                          ->where('status', 0); // ativa
                }),
            ],
            'name' => [
                'required',
                'string',
                'max:255',
                'min:3',
            ],
            'email' => [
                'required',
                'email',
                'max:255',
                'unique:users,email',
            ],
            'password' => [
                'required',
                'string',
                'min:8',
                'confirmed',
            ],
            'phone' => [
                'nullable',
                'string',
                'max:20',
            ],
        ];
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'codigo_convite.required' => 'O código de convite é obrigatório.',
            'codigo_convite.size' => 'O código de convite deve ter exatamente 10 caracteres.',
            'codigo_convite.exists' => 'Código de convite inválido ou franquia desativada.',
            
            'name.required' => 'O nome é obrigatório.',
            'name.min' => 'O nome deve ter no mínimo 3 caracteres.',
            
            'email.required' => 'O email é obrigatório.',
            'email.email' => 'Digite um email válido.',
            'email.unique' => 'Este email já está cadastrado no sistema.',
            
            'password.required' => 'A senha é obrigatória.',
            'password.min' => 'A senha deve ter no mínimo 8 caracteres.',
            'password.confirmed' => 'As senhas não coincidem.',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     */
    public function attributes(): array
    {
        return [
            'codigo_convite' => 'código de convite',
            'name' => 'nome',
            'email' => 'email',
            'password' => 'senha',
            'phone' => 'telefone',
        ];
    }
}
