<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\Userstamps;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Publicacao extends Model
{
    use HasFactory, SoftDeletes, Userstamps;

    /**
     * Nome da tabela no banco de dados
     */
    protected $table = 'publicacoes';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'criativo_id',
        'franquia_id',
        'cliente_id',
        'titulo',
        'data_inicio',
        'data_fim',
        'link_destino',
        'habilitar_clique',
        'track_impressoes',
        'track_cliques',
        'valor_contrato',
        'impressoes_contratadas',
        'total_visualizacoes',
        'tipo_midia',
        'posicao',
        'status',
        'tipo_alcance',
        'equipment_serial',
        'empresa_alcance_id',
        'publicada_por',
        'publicada_em',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data_inicio' => 'datetime',
        'data_fim' => 'datetime',
        'habilitar_clique' => 'boolean',
        'track_impressoes' => 'boolean',
        'track_cliques' => 'boolean',
        'valor_contrato' => 'decimal:2',
        'impressoes_contratadas' => 'integer',
        'total_visualizacoes' => 'integer',
        'posicao' => 'integer',
        'publicada_em' => 'datetime',
    ];

    /**
     * Relacionamento com Criativo
     * Criativo que será exibido nesta publicação
     */
    public function criativo(): BelongsTo
    {
        return $this->belongsTo(Criativo::class, 'criativo_id');
    }

    /**
     * Relacionamento com Franquia
     * Franquia que está vendendo/publicando
     */
    public function franquia(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'franquia_id');
    }

    /**
     * Relacionamento com Cliente
     * Cliente que comprou/contratou a publicação
     */
    public function cliente(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'cliente_id');
    }

    /**
     * Relacionamento com User que publicou
     */
    public function publicadoPor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'publicada_por');
    }

    /**
     * Relacionamento com Empresa de alcance
     * Empresa específica onde a publicação será exibida
     */
    public function empresaAlcance(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'empresa_alcance_id');
    }

    /**
     * Relacionamento: Publicação tem muitos Walled Gardens
     */
    public function walledGardens(): HasMany
    {
        return $this->hasMany(WalledGarden::class);
    }

    /**
     * Relacionamento: Publicação tem muitas impressões
     */
    public function impressoes(): HasMany
    {
        return $this->hasMany(PublicacaoImpressao::class);
    }

    /**
     * Relacionamento: Publicação tem muitos cliques
     */
    public function cliques(): HasMany
    {
        return $this->hasMany(PublicacaoClique::class);
    }

    /**
     * Relacionamento: Logs de acesso que visualizaram esta publicação
     */
    public function logsAcesso(): HasMany
    {
        return $this->hasMany(LogAcessoWifi::class, 'publicacao_visualizada_id');
    }

    /**
     * Relacionamento: Auditoria Splash (dados de cliques do splash page)
     * Conecta através do criativo (image field = criativo_id)
     */
    public function auditoriaSplash()
    {
        return \DB::table('auditoria_splash')
            ->where('imagem', 'like', '%' . $this->criativo_id . '%')
            ->orWhere('imagem', 'like', '%' . ($this->criativo->title ?? '') . '%');
    }

    /**
     * Scope: Publicações ativas
     */
    public function scopeAtivas($query)
    {
        return $query->where('status', 'ativa');
    }

    /**
     * Scope: Publicações expiradas
     */
    public function scopeExpiradas($query)
    {
        return $query->where('status', 'expirada');
    }

    /**
     * Scope: Publicações pausadas
     */
    public function scopePausadas($query)
    {
        return $query->where('status', 'pausada');
    }

    /**
     * Scope: Publicações canceladas
     */
    public function scopeCanceladas($query)
    {
        return $query->where('status', 'cancelada');
    }

    /**
     * Scope: Publicações por período
     */
    public function scopePorPeriodo($query, $dataInicio, $dataFim)
    {
        return $query->where(function ($q) use ($dataInicio, $dataFim) {
            $q->whereBetween('data_inicio', [$dataInicio, $dataFim])
              ->orWhereBetween('data_fim', [$dataInicio, $dataFim])
              ->orWhere(function ($q2) use ($dataInicio, $dataFim) {
                  $q2->where('data_inicio', '<=', $dataInicio)
                     ->where('data_fim', '>=', $dataFim);
              });
        });
    }

    /**
     * Scope: Publicações vigentes (período atual)
     */
    public function scopeVigentes($query)
    {
        return $query->where('data_inicio', '<=', now())
                     ->where('data_fim', '>=', now())
                     ->where('status', 'ativa');
    }

    /**
     * Verificar se a publicação está no período vigente
     */
    public function isVigente(): bool
    {
        return $this->status === 'ativa' 
            && $this->data_inicio <= now() 
            && $this->data_fim >= now();
    }

    /**
     * Verificar se a publicação expirou
     */
    public function isExpirada(): bool
    {
        return $this->data_fim < now();
    }

    /**
     * Calcular dias restantes
     */
    public function diasRestantes(): int
    {
        if ($this->isExpirada()) {
            return 0;
        }
        
        return now()->diffInDays($this->data_fim, false);
    }

    /**
     * Calcular duração total em dias
     */
    public function duracaoTotal(): int
    {
        return $this->data_inicio->diffInDays($this->data_fim);
    }

    /**
     * Scope: Publicações para todos os equipamentos
     */
    public function scopeParaTodos($query)
    {
        return $query->where('tipo_alcance', 'todos');
    }

    /**
     * Scope: Publicações para serial específico
     */
    public function scopeSerialEspecifico($query, $serial = null)
    {
        $q = $query->where('tipo_alcance', 'serial_especifico');
        
        if ($serial) {
            $q->where('equipment_serial', $serial);
        }
        
        return $q;
    }

    /**
     * Scope: Publicações para uma empresa específica
     */
    public function scopeParaEmpresa($query, $empresaId = null)
    {
        $q = $query->where('tipo_alcance', 'empresa');
        
        if ($empresaId) {
            $q->where('empresa_alcance_id', $empresaId);
        }
        
        return $q;
    }

    /**
     * Verificar se a publicação se aplica a um equipamento específico
     */
    public function aplicavelParaEquipamento($serial, $empresaId = null): bool
    {
        // Todos os equipamentos
        if ($this->tipo_alcance === 'todos') {
            return true;
        }
        
        // Serial específico
        if ($this->tipo_alcance === 'serial_especifico') {
            return $this->equipment_serial === $serial;
        }
        
        // Empresa específica
        if ($this->tipo_alcance === 'empresa' && $empresaId) {
            return $this->empresa_alcance_id == $empresaId;
        }
        
        return false;
    }

    /**
     * Get label do tipo de alcance
     */
    public function getTipoAlcanceLabel(): string
    {
        return match($this->tipo_alcance) {
            'todos' => 'Todos os equipamentos',
            'serial_especifico' => 'Serial específico',
            'empresa' => 'Empresa específica',
            default => 'Não definido'
        };
    }

    /**
     * Incrementar visualizações (usado pela API externa)
     */
    public function incrementarVisualizacoes(int $quantidade = 1): bool
    {
        return $this->increment('total_visualizacoes', $quantidade);
    }

    /**
     * Get label do tipo de mídia com ícone
     */
    public function getTipoMidiaLabel(): string
    {
        return match($this->tipo_midia) {
            'imagem' => '🖼️ Imagem',
            'video' => '🎬 Vídeo',
            default => '❓ Não definido'
        };
    }

    /**
     * Get ícone do tipo de mídia
     */
    public function getTipoMidiaIcon(): string
    {
        return match($this->tipo_midia) {
            'imagem' => 'ph-duotone ph-image',
            'video' => 'ph-duotone ph-video',
            default => 'ph-duotone ph-question'
        };
    }

    /**
     * Get badge color baseado na posição
     */
    public function getPosicaoBadgeColor(): string
    {
        return match($this->posicao) {
            1 => 'primary',
            2 => 'success',
            3 => 'info',
            4 => 'warning',
            5 => 'danger',
            6 => 'secondary',
            default => 'dark'
        };
    }

    /**
     * Scope: Filtrar por posição
     */
    public function scopePorPosicao($query, int $posicao)
    {
        return $query->where('posicao', $posicao);
    }

    /**
     * Scope: Filtrar por tipo de mídia
     */
    public function scopePorTipoMidia($query, string $tipo)
    {
        return $query->where('tipo_midia', $tipo);
    }

    /**
     * Verificar se a posição está disponível para um equipamento/empresa
     */
    public static function posicaoDisponivel(int $posicao, ?string $serial = null, ?int $empresaId = null): bool
    {
        $query = self::vigentes()->where('posicao', $posicao);

        if ($serial) {
            $query->where(function($q) use ($serial) {
                $q->where('tipo_alcance', 'serial_especifico')
                  ->where('equipment_serial', $serial);
            });
        } elseif ($empresaId) {
            $query->where(function($q) use ($empresaId) {
                $q->where('tipo_alcance', 'empresa')
                  ->where('empresa_alcance_id', $empresaId);
            });
        }

        return $query->count() === 0;
    }
}
