<?php

namespace App\Models;

use App\Traits\Userstamps;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class TawktoConfig extends Model
{
    use HasFactory, SoftDeletes, Userstamps;

    /**
     * Nome da tabela
     */
    protected $table = 'tawkto_configs';

    /**
     * Campos que podem ser preenchidos em massa
     */
    protected $fillable = [
        'empresa_id',
        'property_id',
        'widget_id',
        'ativo',
        'paginas_habilitadas',
        'customizacao',
        'metadata_personalizado',
    ];

    /**
     * Casting de atributos
     */
    protected $casts = [
        'ativo' => 'boolean',
        'paginas_habilitadas' => 'array',
        'customizacao' => 'array',
        'metadata_personalizado' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Valores padrão
     */
    protected $attributes = [
        'ativo' => true,
        'paginas_habilitadas' => '["index","cadastro","login","welcome"]',
    ];

    /**
     * Relacionamento: Config pertence a uma Empresa
     */
    public function empresa(): BelongsTo
    {
        return $this->belongsTo(Empresa::class);
    }

    /**
     * Scopes para consultas comuns
     */
    public function scopeAtivos($query)
    {
        return $query->where('ativo', true);
    }

    public function scopePorEmpresa($query, int $empresaId)
    {
        return $query->where('empresa_id', $empresaId);
    }

    /**
     * Verificar se uma página está habilitada
     */
    public function isPaginaHabilitada(string $pagina): bool
    {
        if (!$this->paginas_habilitadas) {
            return false;
        }

        return in_array($pagina, $this->paginas_habilitadas);
    }

    /**
     * Retornar script do Tawk.to para inserir no HTML
     */
    public function getScriptEmbed(): string
    {
        if (!$this->ativo || !$this->property_id || !$this->widget_id) {
            return '';
        }

        return <<<HTML
<!--Start of Tawk.to Script-->
<script type="text/javascript">
var Tawk_API=Tawk_API||{}, Tawk_LoadStart=new Date();
(function(){
var s1=document.createElement("script"),s0=document.getElementsByTagName("script")[0];
s1.async=true;
s1.src='https://embed.tawk.to/{$this->property_id}/{$this->widget_id}';
s1.charset='UTF-8';
s1.setAttribute('crossorigin','*');
s0.parentNode.insertBefore(s1,s0);
})();
</script>
<!--End of Tawk.to Script-->
HTML;
    }
}
