<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use Wildside\Userstamps\Userstamps;

class TransacaoFinanceira extends Model
{
    use SoftDeletes, Userstamps;

    /**
     * Nome da tabela
     */
    protected $table = 'transacoes_financeiras';

    /**
     * Campos que podem ser preenchidos em massa
     */
    protected $fillable = [
        'pedido_id',
        'empresa_id',
        'franquia_id',
        'codigo_transacao',
        'transaction_id',
        'tipo',
        'canal',
        'forma_pagamento',
        'valor_bruto',
        'taxa_gateway',
        'valor_liquido',
        'status',
        'data_transacao',
        'data_aprovacao',
        'data_estorno',
        'data_liquidacao',
        'dados_gateway',
        'comprovante',
        'observacoes',
        'motivo_estorno',
        'conciliado',
        'conciliado_em',
        'conciliado_por',
    ];

    /**
     * Casts de atributos
     */
    protected $casts = [
        'valor_bruto' => 'decimal:2',
        'taxa_gateway' => 'decimal:2',
        'valor_liquido' => 'decimal:2',
        'data_transacao' => 'datetime',
        'data_aprovacao' => 'datetime',
        'data_estorno' => 'datetime',
        'data_liquidacao' => 'datetime',
        'dados_gateway' => 'json',
        'conciliado' => 'boolean',
        'conciliado_em' => 'datetime',
    ];

    /**
     * Relacionamento: Pedido de origem
     */
    public function pedido(): BelongsTo
    {
        return $this->belongsTo(PedidoPublicidade::class, 'pedido_id');
    }

    /**
     * Relacionamento: Empresa (cliente que pagou)
     */
    public function empresa(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'empresa_id');
    }

    /**
     * Relacionamento: Franquia que receberá
     */
    public function franquia(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'franquia_id');
    }

    /**
     * Relacionamento: Usuário que conciliou
     */
    public function conciliador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'conciliado_por');
    }

    /**
     * Gera código único de transação (TRX-XXXXXXXX)
     */
    public static function gerarCodigoTransacao(): string
    {
        do {
            $codigo = 'TRX-' . strtoupper(Str::random(8));
        } while (self::where('codigo_transacao', $codigo)->exists());

        return $codigo;
    }

    /**
     * Calcula o valor líquido com base no valor bruto e taxa
     */
    public static function calcularValorLiquido(float $valorBruto, float $taxaGateway): float
    {
        return $valorBruto - $taxaGateway;
    }

    /**
     * Verifica se a transação está pendente
     */
    public function isPendente(): bool
    {
        return $this->status === 'pendente';
    }

    /**
     * Verifica se a transação está em processamento
     */
    public function isProcessando(): bool
    {
        return $this->status === 'processando';
    }

    /**
     * Verifica se a transação foi aprovada
     */
    public function isAprovado(): bool
    {
        return $this->status === 'aprovado';
    }

    /**
     * Verifica se a transação foi recusada
     */
    public function isRecusado(): bool
    {
        return $this->status === 'recusado';
    }

    /**
     * Verifica se a transação foi estornada
     */
    public function isEstornado(): bool
    {
        return $this->status === 'estornado';
    }

    /**
     * Verifica se a transação foi cancelada
     */
    public function isCancelado(): bool
    {
        return $this->status === 'cancelado';
    }

    /**
     * Verifica se a transação foi conciliada
     */
    public function isConciliado(): bool
    {
        return $this->conciliado === true;
    }

    /**
     * Aprova a transação
     */
    public function aprovar(array $dadosGateway = []): void
    {
        $this->update([
            'status' => 'aprovado',
            'data_aprovacao' => now(),
            'dados_gateway' => $dadosGateway,
        ]);
    }

    /**
     * Recusa a transação
     */
    public function recusar(string $motivo): void
    {
        $this->update([
            'status' => 'recusado',
            'observacoes' => $motivo,
        ]);
    }

    /**
     * Estorna a transação
     */
    public function estornar(string $motivo): void
    {
        $this->update([
            'status' => 'estornado',
            'data_estorno' => now(),
            'motivo_estorno' => $motivo,
        ]);
    }

    /**
     * Cancela a transação
     */
    public function cancelar(string $motivo): void
    {
        $this->update([
            'status' => 'cancelado',
            'observacoes' => $motivo,
        ]);
    }

    /**
     * Concilia a transação
     */
    public function conciliar(int $usuarioId): void
    {
        $this->update([
            'conciliado' => true,
            'conciliado_em' => now(),
            'conciliado_por' => $usuarioId,
        ]);
    }

    /**
     * Remove a conciliação
     */
    public function desconciliar(): void
    {
        $this->update([
            'conciliado' => false,
            'conciliado_em' => null,
            'conciliado_por' => null,
        ]);
    }

    /**
     * Retorna a URL do comprovante
     */
    public function getComprovanteUrlAttribute(): ?string
    {
        if (!$this->comprovante) {
            return null;
        }

        return asset('storage/' . $this->comprovante);
    }

    /**
     * Retorna o nome formatado do canal
     */
    public function getCanalNomeAttribute(): string
    {
        $canais = [
            'mercado_pago' => 'Mercado Pago',
            'pagseguro' => 'PagSeguro',
            'stripe' => 'Stripe',
            'pix_manual' => 'PIX Manual',
            'transferencia' => 'Transferência',
            'outro' => 'Outro',
        ];

        return $canais[$this->canal] ?? $this->canal;
    }

    /**
     * Retorna o nome formatado da forma de pagamento
     */
    public function getFormaPagamentoNomeAttribute(): string
    {
        $formas = [
            'pix' => 'PIX',
            'cartao_credito' => 'Cartão de Crédito',
            'cartao_debito' => 'Cartão de Débito',
            'boleto' => 'Boleto',
            'transferencia' => 'Transferência',
        ];

        return $formas[$this->forma_pagamento] ?? $this->forma_pagamento;
    }

    /**
     * Retorna o status formatado com cor
     */
    public function getStatusBadgeAttribute(): array
    {
        $badges = [
            'pendente' => ['text' => 'Pendente', 'color' => 'warning'],
            'processando' => ['text' => 'Processando', 'color' => 'info'],
            'aprovado' => ['text' => 'Aprovado', 'color' => 'success'],
            'recusado' => ['text' => 'Recusado', 'color' => 'danger'],
            'estornado' => ['text' => 'Estornado', 'color' => 'dark'],
            'cancelado' => ['text' => 'Cancelado', 'color' => 'secondary'],
        ];

        return $badges[$this->status] ?? ['text' => $this->status, 'color' => 'secondary'];
    }
}
