<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use App\Traits\Userstamps;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, SoftDeletes, Userstamps;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'avatar',
        'empresa_id',
        'status_cadastro',
        'aprovado_por',
        'aprovado_em',
        'motivo_rejeicao',
        'is_superadmin',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'aprovado_em' => 'datetime',
            'is_superadmin' => 'boolean',
        ];
    }

    public function groups()
    {
        return $this->belongsToMany(Group::class, 'group_user');
    }
    
    // ==========================================
    // RELACIONAMENTOS
    // ==========================================
    
    /**
     * Empresa do usuário (legado - manter para retrocompatibilidade)
     */
    public function empresa()
    {
        return $this->belongsTo(Empresa::class, 'empresa_id');
    }

    /**
     * Múltiplas empresas/franquias do usuário (novo - many-to-many)
     * Permite que um usuário tenha acesso a várias franquias
     */
    public function empresas()
    {
        return $this->belongsToMany(Empresa::class, 'empresa_user')
            ->withPivot('role')
            ->withTimestamps();
    }

    /**
     * Apenas as franquias do usuário (filtra por tipo_empresa = 'franquia')
     */
    public function franquias()
    {
        return $this->belongsToMany(Empresa::class, 'empresa_user')
            ->wherePivot('empresa_id', function($query) {
                $query->select('id')
                      ->from('empresas')
                      ->where('tipo_empresa', 'franquia');
            })
            ->withPivot('role')
            ->withTimestamps();
    }

    /**
     * Retorna IDs de todas as franquias do usuário
     */
    public function getFranquiasIdsAttribute()
    {
        return $this->empresas()
            ->where('tipo_empresa', 'franquia')
            ->pluck('empresas.id')
            ->toArray();
    }

    /**
     * Verifica se o usuário tem acesso a múltiplas franquias
     */
    public function hasMultipleFranchises()
    {
        return $this->empresas()->where('tipo_empresa', 'franquia')->count() > 1;
    }

    /**
     * Verifica se o usuário é de franquia (tem pelo menos uma franquia vinculada)
     */
    public function isFranchiseUser()
    {
        // Verifica se tem franquias vinculadas via pivot
        if ($this->empresas()->where('tipo_empresa', 'franquia')->exists()) {
            return true;
        }

        // Fallback: verifica empresa_id legado
        if ($this->empresa_id && $this->empresa && $this->empresa->tipo_empresa === 'franquia') {
            return true;
        }

        return false;
    }

    /**
     * Quem aprovou este usuário
     */
    public function aprovador()
    {
        return $this->belongsTo(User::class, 'aprovado_por');
    }

    /**
     * Usuários aprovados por este usuário
     */
    public function usuariosAprovados()
    {
        return $this->hasMany(User::class, 'aprovado_por');
    }
}