<?php

namespace App\Policies;

use App\Models\Empresa;
use App\Models\User;

/**
 * Policy para controle de acesso a empresas/clientes
 *
 * Define quem pode visualizar, criar, editar e deletar empresas/clientes
 *
 * Regras:
 * - SuperAdmin: acesso total
 * - Usuário de Franquia: apenas seus clientes
 * - Usuário de Cliente: apenas visualização do próprio cliente
 *
 * Criado em: 08/11/2025
 * Documentação: tests/documentation/ACESSO-USUARIO-FRANQUIA-08-11-2025.md
 */
class EmpresaPolicy
{
    /**
     * Determine if the user can view any empresas.
     */
    public function viewAny(User $user): bool
    {
        // Todos usuários aprovados podem ver empresas (mas serão filtradas)
        return $user->status_cadastro === 'aprovado';
    }

    /**
     * Determine if the user can view the empresa.
     */
    public function view(User $user, Empresa $empresa): bool
    {
        // SuperAdmin vê tudo
        if ($user->is_superadmin) {
            return true;
        }

        // Usuário vê sua própria empresa
        if ($user->empresa_id === $empresa->id) {
            return true;
        }

        // Usuário de franquia vê seus clientes
        if ($user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            // Verifica se o cliente pertence à franquia
            return $empresa->empresa_pai_id === $user->empresa_id;
        }

        // Usuário multi-franquia (sem empresa_id, mas com franquias via pivot)
        if ($user->isFranchiseUser()) {
            $franquiasIds = $user->empresas()->where('tipo_empresa', 'franquia')
                ->pluck('empresas.id')->toArray();
            // Verifica se o cliente pertence a alguma das franquias do usuário
            return in_array($empresa->empresa_pai_id, $franquiasIds);
        }

        return false;
    }

    /**
     * Determine if the user can create empresas.
     */
    public function create(User $user): bool
    {
        // SuperAdmin pode criar qualquer tipo
        if ($user->is_superadmin) {
            return true;
        }

        // Usuário de franquia pode criar apenas clientes
        if ($user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            return true;
        }

        return false;
    }

    /**
     * Determine if the user can update the empresa.
     */
    public function update(User $user, Empresa $empresa): bool
    {
        // SuperAdmin pode editar tudo
        if ($user->is_superadmin) {
            return true;
        }

        // Usuário pode editar sua própria empresa (para configurações como MP)
        if ($user->empresa_id === $empresa->id) {
            return true;
        }

        // Usuário de franquia pode editar seus clientes
        if ($user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            // Verifica se o cliente pertence à franquia
            return $empresa->tipo_empresa === 'cliente'
                   && $empresa->empresa_pai_id === $user->empresa_id;
        }

        // Usuário multi-franquia pode editar clientes de suas franquias
        if ($user->isFranchiseUser()) {
            $franquiasIds = $user->empresas()->where('tipo_empresa', 'franquia')
                ->pluck('empresas.id')->toArray();
            return $empresa->tipo_empresa === 'cliente'
                   && in_array($empresa->empresa_pai_id, $franquiasIds);
        }

        return false;
    }

    /**
     * Determine if the user can delete the empresa.
     */
    public function delete(User $user, Empresa $empresa): bool
    {
        // Apenas SuperAdmin pode deletar
        if ($user->is_superadmin) {
            return true;
        }

        // Franquia pode inativar (não deletar) seus clientes
        if ($user->empresa && $user->empresa->tipo_empresa === 'franquia') {
            return $empresa->tipo_empresa === 'cliente'
                   && $empresa->empresa_pai_id === $user->empresa_id;
        }

        // Usuário multi-franquia pode deletar clientes de suas franquias
        if ($user->isFranchiseUser()) {
            $franquiasIds = $user->empresas()->where('tipo_empresa', 'franquia')
                ->pluck('empresas.id')->toArray();
            return $empresa->tipo_empresa === 'cliente'
                   && in_array($empresa->empresa_pai_id, $franquiasIds);
        }

        return false;
    }
}
