<?php

namespace App\Policies;

use App\Models\TawktoConfig;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class TawktoConfigPolicy
{
    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        // Super admin (is_superadmin OU tipo), franquias e clientes podem ver listagens
        return $user->is_superadmin || in_array($user->tipo, ['super_admin', 'franquia', 'cliente']);
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, TawktoConfig $tawktoConfig): bool
    {
        // Super admin pode ver tudo
        if ($user->is_superadmin || $user->tipo === 'super_admin') {
            return true;
        }

        // Franquia pode ver suas configs e de seus clientes
        if ($user->tipo === 'franquia') {
            return $tawktoConfig->empresa_id === $user->empresa_id
                || $tawktoConfig->empresa->empresa_pai_id === $user->empresa_id;
        }

        // Cliente pode ver apenas sua config
        if ($user->tipo === 'cliente') {
            return $tawktoConfig->empresa_id === $user->empresa_id;
        }

        return false;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        // Super admin (is_superadmin OU tipo) e franquias podem criar
        return $user->is_superadmin || in_array($user->tipo, ['super_admin', 'franquia']);
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, TawktoConfig $tawktoConfig): bool
    {
        // Super admin pode atualizar tudo
        if ($user->is_superadmin || $user->tipo === 'super_admin') {
            return true;
        }

        // Franquia pode atualizar suas configs e de seus clientes
        if ($user->tipo === 'franquia') {
            return $tawktoConfig->empresa_id === $user->empresa_id
                || $tawktoConfig->empresa->empresa_pai_id === $user->empresa_id;
        }

        // Cliente pode atualizar apenas sua config (se tiver permissão)
        if ($user->tipo === 'cliente') {
            return $tawktoConfig->empresa_id === $user->empresa_id;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, TawktoConfig $tawktoConfig): bool
    {
        // Super admin pode deletar tudo
        if ($user->is_superadmin || $user->tipo === 'super_admin') {
            return true;
        }

        // Franquia pode deletar suas configs e de seus clientes
        if ($user->tipo === 'franquia') {
            return $tawktoConfig->empresa_id === $user->empresa_id
                || $tawktoConfig->empresa->empresa_pai_id === $user->empresa_id;
        }

        // Clientes não podem deletar
        return false;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, TawktoConfig $tawktoConfig): bool
    {
        return $this->delete($user, $tawktoConfig);
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, TawktoConfig $tawktoConfig): bool
    {
        // Apenas super admin pode forçar exclusão permanente
        return $user->is_superadmin || $user->tipo === 'super_admin';
    }
}
