<?php

namespace Database\Factories;

use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Criativo>
 */
class CriativoFactory extends Factory
{
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $tipo = fake()->randomElement(['imagem', 'video', 'html']);
        $formato = match($tipo) {
            'imagem' => fake()->randomElement(['jpg', 'png', 'gif']),
            'video' => 'mp4',
            'html' => 'html',
        };
        
        return [
            'franquia_id' => fake()->optional(0.3)->randomElement(\App\Models\Empresa::pluck('id')->toArray()),
            'tipo' => $tipo,
            'title' => fake()->sentence(3),
            'formato' => $formato,
            'path' => $tipo === 'html' ? null : 'uploads/criativos/' . fake()->uuid() . '.' . $formato,
            'conteudo_html' => $tipo === 'html' ? '<div>' . fake()->paragraph() . '</div>' : null,
            'status_aprovacao' => fake()->randomElement(['pendente', 'aprovado', 'rejeitado']),
            'status' => fake()->boolean(80), // 80% ativo
            'position' => fake()->numberBetween(1, 6),
            'start_datetime' => now()->subDays(10),
            'end_datetime' => now()->addDays(20),
            'views_count' => fake()->numberBetween(0, 10000),
        ];
    }

    /**
     * Estado: Imagem
     */
    public function imagem(): static
    {
        return $this->state(fn (array $attributes) => [
            'tipo' => 'imagem',
            'formato' => fake()->randomElement(['jpg', 'png', 'gif']),
            'path' => 'uploads/criativos/' . fake()->uuid() . '.jpg',
            'conteudo_html' => null,
        ]);
    }

    /**
     * Estado: Vídeo
     */
    public function video(): static
    {
        return $this->state(fn (array $attributes) => [
            'tipo' => 'video',
            'formato' => 'mp4',
            'path' => 'uploads/criativos/' . fake()->uuid() . '.mp4',
            'conteudo_html' => null,
        ]);
    }

    /**
     * Estado: HTML (Quiz/NPS)
     */
    public function html(): static
    {
        return $this->state(fn (array $attributes) => [
            'tipo' => 'html',
            'formato' => 'html',
            'path' => null,
            'conteudo_html' => '<div class="quiz">' . fake()->paragraph() . '</div>',
        ]);
    }

    /**
     * Estado: Aprovado
     */
    public function aprovado(): static
    {
        return $this->state(fn (array $attributes) => [
            'status_aprovacao' => 'aprovado',
            'aprovado_por' => \App\Models\User::factory(),
            'aprovado_em' => now(),
        ]);
    }

    /**
     * Estado: Pendente
     */
    public function pendente(): static
    {
        return $this->state(fn (array $attributes) => [
            'status_aprovacao' => 'pendente',
            'aprovado_por' => null,
            'aprovado_em' => null,
        ]);
    }

    /**
     * Estado: Do catálogo (FreeFi)
     */
    public function catalogo(): static
    {
        return $this->state(fn (array $attributes) => [
            'franquia_id' => null,
        ]);
    }
}
