<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('transacoes_financeiras', function (Blueprint $table) {
            $table->id();

            // Relacionamentos
            $table->foreignId('pedido_id')->constrained('pedidos_publicidade')->onDelete('cascade');
            $table->foreignId('empresa_id')->constrained('empresas')->onDelete('cascade')
                ->comment('Cliente que pagou');
            $table->foreignId('franquia_id')->nullable()->constrained('empresas')->onDelete('set null')
                ->comment('Franquia que receberá');

            // Identificação da Transação
            $table->string('codigo_transacao')->unique()->comment('TRX-XXXXXXXX');
            $table->string('transaction_id')->nullable()->comment('ID do gateway (MP, PagSeguro, etc)');

            // Tipo de Transação
            $table->enum('tipo', ['pagamento', 'estorno', 'cancelamento', 'reembolso'])->default('pagamento');

            // Canal de Pagamento
            $table->enum('canal', ['mercado_pago', 'pagseguro', 'stripe', 'pix_manual', 'transferencia', 'outro']);

            // Forma de Pagamento
            $table->enum('forma_pagamento', ['pix', 'cartao_credito', 'cartao_debito', 'boleto', 'transferencia']);

            // Valores
            $table->decimal('valor_bruto', 10, 2)->comment('Valor total');
            $table->decimal('taxa_gateway', 10, 2)->default(0)->comment('Taxa cobrada pelo gateway');
            $table->decimal('valor_liquido', 10, 2)->comment('Valor líquido (bruto - taxa)');

            // Status
            $table->enum('status', ['pendente', 'processando', 'aprovado', 'recusado', 'estornado', 'cancelado'])
                ->default('pendente');

            // Datas
            $table->timestamp('data_transacao')->nullable()->comment('Data/hora da transação');
            $table->timestamp('data_aprovacao')->nullable()->comment('Data/hora da aprovação');
            $table->timestamp('data_estorno')->nullable()->comment('Data/hora do estorno');
            $table->timestamp('data_liquidacao')->nullable()->comment('Data que cai na conta');

            // Informações Adicionais
            $table->text('dados_gateway')->nullable()->comment('JSON com dados completos do gateway');
            $table->string('comprovante')->nullable()->comment('Caminho do comprovante');
            $table->text('observacoes')->nullable();
            $table->text('motivo_estorno')->nullable();

            // Conciliação
            $table->boolean('conciliado')->default(false);
            $table->timestamp('conciliado_em')->nullable();
            $table->foreignId('conciliado_por')->nullable()->constrained('users')->onDelete('set null');

            // Auditoria
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('updated_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('deleted_by')->nullable()->constrained('users')->onDelete('set null');

            $table->timestamps();
            $table->softDeletes();

            // Índices
            $table->index('pedido_id');
            $table->index('empresa_id');
            $table->index('franquia_id');
            $table->index('canal');
            $table->index('status');
            $table->index('data_transacao');
            $table->index('conciliado');
            $table->index('transaction_id');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('transacoes_financeiras');
    }
};
