#!/bin/bash

# =============================================================================
# DEPLOY REMOTO - CORREÇÃO DE PAGINAÇÃO
# Executa deploy no servidor de produção via comandos HTTP
# =============================================================================

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

SERVER_IP="172.233.21.57"
SERVER_PORT="8082"

echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}  DEPLOY REMOTO - PAGINAÇÃO${NC}"
echo -e "${BLUE}========================================${NC}"
echo ""

log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[OK]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERRO]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[AVISO]${NC} $1"
}

# Verificar conectividade
log_info "Testando conectividade com servidor..."

if curl -s -o /dev/null -w "%{http_code}" "http://$SERVER_IP:$SERVER_PORT" | grep -q "200\|302"; then
    log_success "Servidor respondendo: http://$SERVER_IP:$SERVER_PORT"
else
    log_error "Servidor não está acessível!"
    exit 1
fi

echo ""
log_warning "⚠️  ATENÇÃO: Deploy será feito no servidor de PRODUÇÃO!"
log_warning "    IP: $SERVER_IP:$SERVER_PORT"
echo ""

read -p "Confirma o deploy? (digite 'sim' para continuar): " -r
if [[ ! $REPLY == "sim" ]]; then
    log_info "Deploy cancelado pelo usuário."
    exit 0
fi

echo ""
log_info "Iniciando deploy..."

# Instruções para executar no servidor
cat > /tmp/deploy_commands.txt << 'EOF'
#!/bin/bash

echo "=== INICIANDO DEPLOY ==="

# Navegar para diretório da aplicação
cd /var/www/html/laravel/admin/freefi_admin || cd /root/freefi_admin || { echo "Diretório não encontrado!"; exit 1; }

echo "📁 Diretório atual: $(pwd)"

# 1. Fazer backup
echo ""
echo "📦 Criando backup..."
BACKUP_DIR="backups/deploy-$(date +%Y%m%d_%H%M%S)"
mkdir -p "$BACKUP_DIR"
cp app/Providers/AppServiceProvider.php "$BACKUP_DIR/" 2>/dev/null || true
cp resources/views/publicacoes/index.blade.php "$BACKUP_DIR/" 2>/dev/null || true
echo "✅ Backup criado em: $BACKUP_DIR"

# 2. Git pull
echo ""
echo "📥 Atualizando código..."
git fetch --all
git reset --hard origin/main
git pull origin main
echo "✅ Código atualizado"

# 3. Verificar arquivos
echo ""
echo "🔍 Verificando arquivos..."
if grep -q "Paginator::useBootstrapFive()" app/Providers/AppServiceProvider.php; then
    echo "✅ AppServiceProvider.php: OK"
else
    echo "❌ AppServiceProvider.php: PROBLEMA - Paginator::useBootstrapFive() não encontrado!"
fi

if grep -q "publicacoes->firstItem()" resources/views/publicacoes/index.blade.php; then
    echo "✅ index.blade.php: OK"
else
    echo "⚠️  index.blade.php: Pode precisar de verificação manual"
fi

# 4. Detectar PHP
PHP_CMD="php"
if command -v php8.3 &> /dev/null; then
    PHP_CMD="php8.3"
    echo "✅ Usando PHP 8.3"
elif command -v php8.2 &> /dev/null; then
    PHP_CMD="php8.2"
    echo "✅ Usando PHP 8.2"
fi

# 5. Limpar caches
echo ""
echo "🗑️  Limpando caches..."
$PHP_CMD artisan config:clear 2>&1 | grep -v "Composer detected\|deprecated" || true
$PHP_CMD artisan view:clear 2>&1 | grep -v "Composer detected\|deprecated" || true
$PHP_CMD artisan cache:clear 2>&1 | grep -v "Composer detected\|deprecated" || true
echo "✅ Caches limpos"

# 6. Otimizar
echo ""
echo "⚡ Otimizando..."
$PHP_CMD artisan config:cache 2>&1 | grep -v "Composer detected\|deprecated" || true
$PHP_CMD artisan route:cache 2>&1 | grep -v "Composer detected\|deprecated" || true
$PHP_CMD artisan view:cache 2>&1 | grep -v "Composer detected\|deprecated" || true
echo "✅ Otimizações aplicadas"

# 7. Reiniciar serviço
echo ""
echo "🔄 Reiniciando serviços..."
if [ -f "docker-compose.yml" ]; then
    docker-compose restart app 2>&1
    echo "✅ Container reiniciado"
else
    echo "⚠️  docker-compose.yml não encontrado, tentando PHP-FPM..."
    sudo systemctl restart php8.3-fpm 2>&1 || sudo systemctl restart php-fpm 2>&1 || echo "⚠️  Reinicie manualmente"
fi

echo ""
echo "========================================="
echo "✅ DEPLOY CONCLUÍDO COM SUCESSO!"
echo "========================================="
echo ""
echo "Acesse: http://172.233.21.57:8082/publicacoes"
echo ""

EOF

log_success "Script de deploy preparado!"
echo ""
log_info "📋 Para executar o deploy, você precisa:"
echo ""
echo "1. Copiar o script para o servidor:"
echo "   ${YELLOW}scp /tmp/deploy_commands.txt root@$SERVER_IP:/tmp/${NC}"
echo ""
echo "2. Executar no servidor:"
echo "   ${YELLOW}ssh root@$SERVER_IP 'bash /tmp/deploy_commands.txt'${NC}"
echo ""
echo "Ou executar manualmente via SSH:"
echo ""
cat << 'COMMANDS'
ssh root@172.233.21.57

# No servidor:
cd /var/www/html/laravel/admin/freefi_admin
git pull origin main
php8.3 artisan config:clear
php8.3 artisan view:clear
php8.3 artisan cache:clear
php8.3 artisan config:cache
php8.3 artisan view:cache
docker-compose restart app

COMMANDS

echo ""
log_warning "⚠️  Sem acesso SSH configurado. Comandos acima devem ser executados manualmente."
echo ""
log_info "Script salvo em: /tmp/deploy_commands.txt"

# Mostrar script
echo ""
log_info "📄 Conteúdo do script de deploy:"
echo -e "${YELLOW}----------------------------------------${NC}"
cat /tmp/deploy_commands.txt
echo -e "${YELLOW}----------------------------------------${NC}"
echo ""

log_info "Para executar automaticamente (se tiver SSH configurado):"
echo "  ${BLUE}ssh root@$SERVER_IP 'bash -s' < /tmp/deploy_commands.txt${NC}"
echo ""
