# 🚀 Guia de Deploy Automatizado - MikroTik Hotspot

## 📋 Índice

1. [Pré-requisitos](#pré-requisitos)
2. [Configuração Inicial](#configuração-inicial)
3. [Autenticação SSH](#autenticação-ssh)
4. [Como Usar](#como-usar)
5. [Troubleshooting](#troubleshooting)
6. [Deploy via GitHub Actions (CI/CD)](#deploy-via-github-actions)

---

## 🔧 Pré-requisitos

### No seu computador:

```bash
# Verificar se tem SSH e SCP instalados
ssh -V
scp -V

# Instalar rsync (opcional, mas recomendado)
# Ubuntu/Debian:
sudo apt install rsync

# macOS:
brew install rsync
```

### No MikroTik:

1. **Habilitar SSH:**
   ```
   /ip service
   set ssh port=22
   enable ssh
   ```

2. **Verificar diretório hotspot:**
   ```
   /file print
   # Deve existir: /flash/hotspot/
   ```

3. **Criar usuário com permissões:**
   ```
   /user add name=deploy group=full password=SUA_SENHA
   ```

---

## ⚙️ Configuração Inicial

### 1. Configurar VPN

Certifique-se de que seu PC pode alcançar o MikroTik via VPN:

```bash
# Testar conectividade
ping 10.x.x.x  # IP do MikroTik na VPN

# Testar SSH
ssh admin@10.x.x.x
```

### 2. Configurar arquivo `.env.deploy`

Edite o arquivo `.env.deploy` e ajuste as configurações:

```bash
cd /var/www/html/laravel/admin/freefi_admin
cp .env.deploy.example .env.deploy  # Se houver exemplo
nano .env.deploy
```

**Configurações mínimas obrigatórias:**

```bash
MIKROTIK_IP=10.x.x.x      # ⚠️ AJUSTAR
MIKROTIK_USER=admin       # ⚠️ AJUSTAR
MIKROTIK_PORT=22
MIKROTIK_DEST=/flash/hotspot
```

### 3. Dar permissão de execução ao script

```bash
chmod +x deploy-mikrotik.sh
```

---

## 🔐 Autenticação SSH

### Opção 1: Autenticação por Chave SSH (Recomendado)

**Passo 1: Gerar chave SSH**

```bash
ssh-keygen -t rsa -b 4096 -f ~/.ssh/mikrotik_rsa -C "deploy@mikrotik"
```

**Passo 2: Copiar chave pública para o MikroTik**

```bash
# Método 1: Via ssh-copy-id
ssh-copy-id -i ~/.ssh/mikrotik_rsa.pub admin@10.x.x.x

# Método 2: Manual
cat ~/.ssh/mikrotik_rsa.pub
# Copie o conteúdo e adicione no MikroTik:
# /user ssh-keys import user=admin public-key-file=...
```

**Passo 3: Configurar no .env.deploy**

```bash
SSH_KEY=/home/seu_usuario/.ssh/mikrotik_rsa
```

**Testar:**

```bash
ssh -i ~/.ssh/mikrotik_rsa admin@10.x.x.x
```

### Opção 2: Autenticação por Senha

Simplesmente deixe `SSH_KEY` vazio no `.env.deploy`. A senha será solicitada durante o deploy.

⚠️ **Não recomendado para automação!**

---

## 🚀 Como Usar

### Deploy Manual

```bash
# Deploy básico
./deploy-mikrotik.sh

# Deploy especificando ambiente
./deploy-mikrotik.sh production
./deploy-mikrotik.sh staging
```

### O que o script faz:

1. ✅ Verifica requisitos (SSH, SCP, rsync)
2. ✅ Testa conexão com MikroTik via VPN
3. ✅ Cria backup automático no MikroTik
4. ✅ Exibe lista de arquivos que serão enviados
5. ✅ Pede confirmação do usuário
6. ✅ Transfere arquivos (usando rsync ou SCP)
7. ✅ Verifica se o deploy foi bem-sucedido
8. ✅ Exibe resumo

**Exemplo de saída:**

```
╔════════════════════════════════════════════════════════════════════╗
║  DEPLOY AUTOMATIZADO - MIKROTIK HOTSPOT                            ║
╚════════════════════════════════════════════════════════════════════╝

[STEP] Verificando requisitos...
✓ Requisitos verificados

[STEP] Testando conexão com MikroTik (10.x.x.x)...
✓ Conexão estabelecida com sucesso

[STEP] Criando backup no MikroTik...
✓ Backup criado: /flash/hotspot_backup_20251021_203045

[STEP] Arquivos que serão enviados:
  📄 index.html
  📄 cadastro.html
  📄 css/style.css
  ...

════════════════════════════════════════════════════════════════════
  ATENÇÃO: Você está prestes a fazer deploy em PRODUCTION
  Destino: admin@10.x.x.x:/flash/hotspot
════════════════════════════════════════════════════════════════════

Deseja continuar? (s/n): s

[STEP] Iniciando deploy dos arquivos...
[STEP] Usando rsync para transferência...
✓ Arquivos transferidos com sucesso!

[STEP] Verificando deployment...
✓ Verificação concluída - arquivo atualizado corretamente

╔════════════════════════════════════════════════════════════════════╗
║  DEPLOY CONCLUÍDO COM SUCESSO!                                     ║
╚════════════════════════════════════════════════════════════════════╝

Ambiente: production
Destino: admin@10.x.x.x:/flash/hotspot
Data: 2025-10-21 20:30:45
```

---

## 🔍 Troubleshooting

### Problema: "Conexão recusada"

**Possíveis causas:**
1. VPN não está ativa
2. IP do MikroTik incorreto
3. SSH desabilitado no MikroTik
4. Firewall bloqueando porta 22

**Solução:**
```bash
# 1. Verificar VPN
ip route | grep vpn

# 2. Testar conectividade
ping 10.x.x.x

# 3. Verificar se porta SSH está aberta
nmap -p 22 10.x.x.x

# 4. Testar SSH manualmente
ssh -v admin@10.x.x.x
```

### Problema: "Permission denied (publickey)"

**Causa:** Chave SSH não configurada corretamente

**Solução:**
```bash
# Verificar permissões da chave
chmod 600 ~/.ssh/mikrotik_rsa

# Testar conexão verbosa
ssh -i ~/.ssh/mikrotik_rsa -v admin@10.x.x.x

# Verificar se chave pública está no MikroTik
ssh admin@10.x.x.x "/user ssh-keys print"
```

### Problema: "Diretório não encontrado"

**Causa:** Caminho `/flash/hotspot` não existe no MikroTik

**Solução:**
```bash
# Conectar no MikroTik e criar diretório
ssh admin@10.x.x.x
/file print
# Se não existir /flash/hotspot:
# Criar via FTP ou WebFig
```

### Problema: Arquivos transferidos mas não funcionam

**Verificar:**

1. **Permissões dos arquivos:**
   ```bash
   ssh admin@10.x.x.x "ls -la /flash/hotspot/"
   ```

2. **Cache do MikroTik:**
   ```bash
   # Limpar cache do hotspot
   ssh admin@10.x.x.x "/ip hotspot walled-garden flush"
   ```

3. **Encoding dos arquivos:**
   ```bash
   # Verificar se arquivos estão em UTF-8
   file -bi flash/hotspot/index.html
   ```

---

## 🤖 Deploy via GitHub Actions (CI/CD)

### Criar workflow automático

Crie `.github/workflows/deploy-mikrotik.yml`:

```yaml
name: Deploy to MikroTik

on:
  push:
    branches:
      - main
    paths:
      - 'flash/hotspot/**'

  workflow_dispatch:  # Permite deploy manual

jobs:
  deploy:
    runs-on: ubuntu-latest

    steps:
      - name: Checkout code
        uses: actions/checkout@v3

      - name: Setup SSH
        run: |
          mkdir -p ~/.ssh
          echo "${{ secrets.MIKROTIK_SSH_KEY }}" > ~/.ssh/mikrotik_rsa
          chmod 600 ~/.ssh/mikrotik_rsa
          ssh-keyscan -H ${{ secrets.MIKROTIK_IP }} >> ~/.ssh/known_hosts

      - name: Install dependencies
        run: |
          sudo apt-get update
          sudo apt-get install -y rsync

      - name: Deploy to MikroTik
        env:
          MIKROTIK_IP: ${{ secrets.MIKROTIK_IP }}
          MIKROTIK_USER: ${{ secrets.MIKROTIK_USER }}
        run: |
          chmod +x deploy-mikrotik.sh
          ./deploy-mikrotik.sh production

      - name: Notify on success
        if: success()
        run: echo "Deploy concluído com sucesso!"

      - name: Notify on failure
        if: failure()
        run: echo "Deploy falhou!"
```

### Configurar Secrets no GitHub

Em `Settings > Secrets and variables > Actions`, adicione:

- `MIKROTIK_IP` - IP do MikroTik na VPN
- `MIKROTIK_USER` - Usuário SSH
- `MIKROTIK_SSH_KEY` - Conteúdo da chave privada

⚠️ **Importante:** O runner do GitHub precisa ter acesso à VPN!

---

## 📊 Verificação Pós-Deploy

Após o deploy, verificar:

### 1. Arquivos no MikroTik

```bash
ssh admin@10.x.x.x "ls -lh /flash/hotspot/"
```

### 2. Testar Hotspot

1. Conectar em uma rede WiFi gerenciada pelo MikroTik
2. Abrir navegador (deve redirecionar para o hotspot)
3. Verificar se:
   - ✅ Página carrega corretamente
   - ✅ Imagens de publicidade aparecem
   - ✅ Modal de publicidade funciona
   - ✅ Formulários de cadastro funcionam

### 3. Verificar Logs

```bash
# No servidor Laravel
ssh root@172.233.21.57 "cd /var/www/freefi && docker compose logs -f app | grep splash"

# Verificar auditoria
ssh root@172.233.21.57 "cd /var/www/freefi && docker compose exec -T app php artisan tinker --execute='echo DB::table(\"auditoria_splash\")->count();'"
```

---

## 📝 Comandos Úteis

```bash
# Listar backups no MikroTik
ssh admin@10.x.x.x "ls -lh /flash/ | grep hotspot_backup"

# Restaurar backup
ssh admin@10.x.x.x "cp -r /flash/hotspot_backup_20251021_203045 /flash/hotspot"

# Ver logs do MikroTik
ssh admin@10.x.x.x "/log print where topics~\"hotspot\""

# Reiniciar serviço hotspot
ssh admin@10.x.x.x "/ip hotspot disable [find]"
ssh admin@10.x.x.x "/ip hotspot enable [find]"

# Limpar cache do navegador dos clientes
ssh admin@10.x.x.x "/ip hotspot user remove [find]"
```

---

## 🔄 Rollback (Reverter Deploy)

Se algo der errado, você pode reverter para o backup:

```bash
# Listar backups disponíveis
ssh admin@10.x.x.x "ls -d /flash/hotspot_backup_*"

# Reverter para backup específico
BACKUP="hotspot_backup_20251021_203045"
ssh admin@10.x.x.x "
    rm -rf /flash/hotspot_old
    mv /flash/hotspot /flash/hotspot_old
    cp -r /flash/$BACKUP /flash/hotspot
    echo 'Rollback concluído!'
"
```

---

## 🛡️ Segurança

### Checklist de Segurança

- [ ] Use chave SSH ao invés de senha
- [ ] Não versione `.env.deploy` no Git
- [ ] Use usuário específico para deploy (não admin)
- [ ] Limite permissões SSH no MikroTik
- [ ] Configure firewall para permitir SSH apenas da VPN
- [ ] Rotate chaves SSH periodicamente
- [ ] Mantenha backups regulares
- [ ] Monitore logs de acesso SSH

### Adicionar ao .gitignore

```bash
echo ".env.deploy" >> .gitignore
echo "*.backup" >> .gitignore
```

---

**Última atualização:** 21/10/2025
**Versão:** 1.0
**Autor:** FreeFi Admin Team
