#!/bin/bash

###############################################################################
# Script de Deploy Automatizado - MikroTik Hotspot (CONFIGURADO)
# Servidor: 172.233.21.57
# Uso: bash deploy-mikrotik-configured.sh
###############################################################################

set -e

# Cores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

# ============================================================================
# CONFIGURAÇÕES - JÁ PRÉ-CONFIGURADAS
# ============================================================================

MIKROTIK_IP="1.2.0.237"
MIKROTIK_USER="admin"
MIKROTIK_PORT="22"
MIKROTIK_DEST="flash/hotspot"
LOCAL_DIR="./flash/hotspot"
MIKROTIK_PASSWORD="Delta46.capiba2!"

# ============================================================================
# FUNÇÕES
# ============================================================================

print_header() {
    echo -e "${BLUE}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BLUE}║  DEPLOY MIKROTIK - FREEFI HOTSPOT                                 ║${NC}"
    echo -e "${BLUE}╚════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo -e "${CYAN}[STEP]${NC} $1"
}

print_success() {
    echo -e "${GREEN}✓${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠${NC} $1"
}

print_error() {
    echo -e "${RED}✗${NC} $1"
}

test_connection() {
    print_step "Testando conexão com MikroTik ($MIKROTIK_IP)..."

    if sshpass -p "Delta46.capiba2!" ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 "$MIKROTIK_USER@$MIKROTIK_IP" "echo 'OK'" &>/dev/null; then
        print_success "Conexão estabelecida com sucesso"
        return 0
    else
        print_error "Falha na conexão. Verifique se sshpass está instalado:"
        echo "  sudo apt install sshpass"
        return 1
    fi
}

list_remote_files() {
    print_step "Listando arquivos atuais no MikroTik..."

    sshpass -p "Delta46.capiba2!" ssh -o StrictHostKeyChecking=no "$MIKROTIK_USER@$MIKROTIK_IP" "
        echo '=== Arquivos em /$MIKROTIK_DEST/ ==='
        ls -lh /$MIKROTIK_DEST/ 2>/dev/null || echo 'Diretório não existe!'
    "
}

create_backup() {
    print_step "Criando backup no MikroTik..."

    BACKUP_NAME="hotspot_backup_$(date +%Y%m%d_%H%M%S).tar.gz"

    sshpass -p "Delta46.capiba2!" ssh -o StrictHostKeyChecking=no "$MIKROTIK_USER@$MIKROTIK_IP" "
        if [ -d '/$MIKROTIK_DEST' ]; then
            tar -czf /flash/${BACKUP_NAME} -C /flash hotspot
            echo 'Backup criado: /flash/${BACKUP_NAME}'
        else
            echo 'AVISO: Diretório não existe, criando...'
            mkdir -p /$MIKROTIK_DEST
        fi
    "

    print_success "Backup concluído"
}

show_changes() {
    print_step "Arquivos que serão enviados:"
    echo ""

    if [ ! -d "$LOCAL_DIR" ]; then
        print_error "Diretório local não encontrado: $LOCAL_DIR"
        exit 1
    fi

    find "$LOCAL_DIR" -type f | while read file; do
        rel_path="${file#$LOCAL_DIR/}"
        size=$(du -h "$file" | cut -f1)
        echo "  📄 $rel_path ($size)"
    done

    echo ""
}

confirm_deploy() {
    echo -e "${YELLOW}════════════════════════════════════════════════════════════════════${NC}"
    echo -e "${YELLOW}  Destino: $MIKROTIK_USER@$MIKROTIK_IP:/$MIKROTIK_DEST${NC}"
    echo -e "${YELLOW}  Total de arquivos: $(find "$LOCAL_DIR" -type f | wc -l)${NC}"
    echo -e "${YELLOW}════════════════════════════════════════════════════════════════════${NC}"
    echo ""

    read -p "Deseja continuar com o deploy? (s/n): " -n 1 -r
    echo

    if [[ ! $REPLY =~ ^[Ss]$ ]]; then
        print_warning "Deploy cancelado"
        exit 0
    fi
}

deploy_files() {
    print_step "Transferindo arquivos..."

    # Usar rsync se disponível
    if command -v rsync &> /dev/null; then
        print_step "Usando rsync (modo eficiente)..."

        sshpass -p "Delta46.capiba2!" rsync -avz --progress \
            --exclude="*.md" \
            --exclude="*.backup" \
            --exclude=".git" \
            --exclude="_errors.txt" \
            -e "ssh -o StrictHostKeyChecking=no" \
            "$LOCAL_DIR/" "$MIKROTIK_USER@$MIKROTIK_IP:/$MIKROTIK_DEST/"

    else
        print_step "Usando SCP..."

        # Criar diretório remoto
        sshpass -p "Delta46.capiba2!" ssh -o StrictHostKeyChecking=no "$MIKROTIK_USER@$MIKROTIK_IP" "mkdir -p /$MIKROTIK_DEST"

        # Copiar arquivos recursivamente
        sshpass -p "Delta46.capiba2!" scp -r -o StrictHostKeyChecking=no "$LOCAL_DIR"/* "$MIKROTIK_USER@$MIKROTIK_IP:/$MIKROTIK_DEST/"
    fi

    if [ $? -eq 0 ]; then
        print_success "Arquivos transferidos com sucesso!"
    else
        print_error "Falha na transferência"
        exit 1
    fi
}

verify_deployment() {
    print_step "Verificando deployment..."

    # Verificar se index.html contém a atualização
    CHECK=$(sshpass -p "Delta46.capiba2!" ssh -o StrictHostKeyChecking=no "$MIKROTIK_USER@$MIKROTIK_IP" "
        if [ -f '/$MIKROTIK_DEST/index.html' ]; then
            if grep -q '8082/storage' '/$MIKROTIK_DEST/index.html'; then
                echo 'UPDATED'
            else
                echo 'OLD_VERSION'
            fi
        else
            echo 'MISSING'
        fi
    " 2>/dev/null)

    case "$CHECK" in
        "UPDATED")
            print_success "✓ Arquivo atualizado com sucesso!"
            ;;
        "OLD_VERSION")
            print_warning "Arquivo transferido mas ainda com código antigo"
            ;;
        "MISSING")
            print_error "Arquivo index.html não encontrado!"
            ;;
        *)
            print_warning "Não foi possível verificar"
            ;;
    esac
}

show_summary() {
    echo ""
    echo -e "${GREEN}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║  DEPLOY CONCLUÍDO!                                                 ║${NC}"
    echo -e "${GREEN}╚════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "${CYAN}Servidor:${NC} $MIKROTIK_IP"
    echo -e "${CYAN}Destino:${NC} /$MIKROTIK_DEST"
    echo -e "${CYAN}Data:${NC} $(date '+%Y-%m-%d %H:%M:%S')"
    echo ""
    echo -e "${YELLOW}Próximos passos:${NC}"
    echo "  1. Teste o hotspot conectando na rede WiFi"
    echo "  2. Verifique se as imagens carregam"
    echo "  3. Confirme se a auditoria está funcionando"
    echo ""
}

# ============================================================================
# MAIN
# ============================================================================

main() {
    print_header

    # Verificar se sshpass está instalado
    if ! command -v sshpass &> /dev/null; then
        print_error "sshpass não encontrado!"
        echo "Instale com: sudo apt install sshpass"
        exit 1
    fi

    # Verificar se diretório local existe
    if [ ! -d "$LOCAL_DIR" ]; then
        print_error "Diretório local não encontrado: $LOCAL_DIR"
        echo "Execute este script do diretório raiz do projeto"
        exit 1
    fi

    # Testes
    test_connection || exit 1
    list_remote_files

    # Exibir mudanças
    show_changes

    # Confirmar
    confirm_deploy

    # Backup
    create_backup

    # Deploy
    deploy_files

    # Verificar
    verify_deployment

    # Resumo
    show_summary
}

main "$@"
