@extends('layouts.master')
@section('title', 'Editar Cliente')
@section('main-content')
    <div class="container-fluid">
        <!-- Breadcrumb start -->
        <div class="row m-1">
            <div class="col-12">
                <h5><i class="ph-duotone ph-buildings f-s-16 me-2"></i>Editar Cliente</h5>
                <ul class="app-line-breadcrumbs mb-3">
                    <li class="">
                        <a href="{{ route('clientes.index') }}" class="f-s-14 f-w-500">
                            <span>Meus Clientes</span>
                        </a>
                    </li>
                    <li class="active">
                        <a href="#" class="f-s-14 f-w-500">Editar Cliente</a>
                    </li>
                </ul>
            </div>
        </div>
        <!-- Breadcrumb end -->

        <div class="row">
            <div class="col-xl-8">
                <div class="card">
                    <div class="card-header">
                        <h5>Editar Dados do Cliente</h5>
                        <p class="text-muted small mb-0">
                            <i class="ph ph-info me-1"></i>
                            Cliente: <strong>{{ $cliente->name }}</strong> | Franquia: <strong>{{ $franquia->name }}</strong>
                        </p>
                    </div>
                    <div class="card-body">
                        {{-- OWASP A02:2021 - CSRF Protection --}}
                        {{-- OWASP A03:2021 - Validação no backend (ClienteController) --}}
                        <form action="{{ route('clientes.update', $cliente->id) }}"
                              method="POST"
                              enctype="multipart/form-data"
                              id="formCliente">
                            @csrf
                            @method('PUT')

                            <!-- Nome -->
                            <div class="mb-3">
                                <label for="name" class="form-label">
                                    Nome do Cliente <span class="text-danger">*</span>
                                </label>
                                <input type="text"
                                       class="form-control @error('name') is-invalid @enderror"
                                       id="name"
                                       name="name"
                                       value="{{ old('name', $cliente->name) }}"
                                       required
                                       minlength="3"
                                       maxlength="255"
                                       placeholder="Ex: Restaurante Sabor da Casa"
                                       autofocus>
                                @error('name')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">Mínimo 3 caracteres, apenas letras, espaços e hífens</small>
                            </div>

                            <!-- Email -->
                            <div class="mb-3">
                                <label for="email" class="form-label">E-mail</label>
                                <input type="email"
                                       class="form-control @error('email') is-invalid @enderror"
                                       id="email"
                                       name="email"
                                       value="{{ old('email', $cliente->email) }}"
                                       maxlength="255"
                                       placeholder="contato@cliente.com.br">
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">Email deve ser válido (verificação RFC + DNS)</small>
                            </div>

                            <!-- Telefone -->
                            <div class="mb-3">
                                <label for="phone" class="form-label">Telefone</label>
                                <input type="text"
                                       class="form-control @error('phone') is-invalid @enderror"
                                       id="phone"
                                       name="phone"
                                       value="{{ old('phone', $cliente->phone) }}"
                                       maxlength="20"
                                       placeholder="(00) 0000-0000">
                                @error('phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">Apenas números, espaços, parênteses, hífens e +</small>
                            </div>

                            <!-- Endereço -->
                            <div class="mb-3">
                                <label for="address" class="form-label">Endereço</label>
                                <textarea class="form-control @error('address') is-invalid @enderror"
                                          id="address"
                                          name="address"
                                          rows="3"
                                          maxlength="500"
                                          placeholder="Rua, número, bairro, cidade - UF">{{ old('address', $cliente->address) }}</textarea>
                                @error('address')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">Máximo 500 caracteres</small>
                            </div>

                            <!-- Status -->
                            <div class="mb-3">
                                <label for="status" class="form-label">
                                    Status <span class="text-danger">*</span>
                                </label>
                                <select class="form-select @error('status') is-invalid @enderror"
                                        id="status"
                                        name="status"
                                        required>
                                    <option value="0" {{ old('status', $cliente->status) == 0 ? 'selected' : '' }}>Ativo</option>
                                    <option value="1" {{ old('status', $cliente->status) == 1 ? 'selected' : '' }}>Inativo</option>
                                </select>
                                @error('status')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">
                                    <i class="ph ph-warning me-1"></i>
                                    Clientes inativos não poderão acessar o sistema
                                </small>
                            </div>

                            <!-- Logomarca -->
                            {{-- OWASP A05:2021 - Upload seguro com validações --}}
                            <div class="mb-3">
                                <label for="logomarca" class="form-label">Logomarca</label>

                                @if($cliente->logomarca && Storage::disk('public')->exists($cliente->logomarca))
                                    <div class="mb-2">
                                        <p class="text-muted small mb-1">Logomarca atual:</p>
                                        <img src="{{ asset('storage/' . $cliente->logomarca) }}"
                                             alt="{{ $cliente->name }}"
                                             class="rounded border"
                                             style="max-width: 200px; max-height: 200px; object-fit: cover;"
                                             id="currentLogo">
                                        <div class="form-check mt-2">
                                            <input class="form-check-input"
                                                   type="checkbox"
                                                   id="removeLogo"
                                                   name="remove_logo"
                                                   value="1">
                                            <label class="form-check-label text-danger" for="removeLogo">
                                                <i class="ph ph-trash me-1"></i>Remover logomarca atual
                                            </label>
                                        </div>
                                    </div>
                                @endif

                                <input type="file"
                                       class="form-control @error('logomarca') is-invalid @enderror"
                                       id="logomarca"
                                       name="logomarca"
                                       accept="image/jpeg,image/png,image/jpg,image/gif,image/webp">
                                @error('logomarca')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">
                                    <i class="ph ph-info me-1"></i>
                                    Formatos: JPG, PNG, GIF, WEBP | Tamanho máx: 2MB | Dimensões máx: 4000x4000px
                                </small>

                                <!-- Preview da nova imagem -->
                                <div id="imagePreview" class="mt-3" style="display: none;">
                                    <p class="text-muted small mb-1">Nova logomarca:</p>
                                    <img id="preview" src="#" alt="Preview" class="rounded border" style="max-width: 200px; max-height: 200px;">
                                </div>
                            </div>

                            <hr class="my-4">

                            <!-- Botões -->
                            <div class="d-flex justify-content-between">
                                <a href="{{ route('clientes.index') }}" class="btn btn-secondary">
                                    <i class="ph ph-arrow-left me-1"></i>Voltar
                                </a>
                                <button type="submit" class="btn btn-primary" id="btnSubmit">
                                    <i class="ph ph-check me-1"></i>Salvar Alterações
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Card de Informações -->
            <div class="col-xl-4">
                <div class="card bg-light-info">
                    <div class="card-header bg-info text-white">
                        <h5 class="mb-0"><i class="ph ph-info me-2"></i>Informações</h5>
                    </div>
                    <div class="card-body">
                        <h6>Dados do Cliente</h6>
                        <ul class="small mb-3">
                            <li><strong>ID:</strong> {{ $cliente->id }}</li>
                            <li><strong>Criado em:</strong> {{ $cliente->created_at->format('d/m/Y H:i') }}</li>
                            <li><strong>Última atualização:</strong> {{ $cliente->updated_at->format('d/m/Y H:i') }}</li>
                        </ul>

                        <h6>Validações de Segurança</h6>
                        <ul class="small mb-0">
                            <li>Nome: apenas letras e hífens</li>
                            <li>Email: validação RFC + DNS</li>
                            <li>Telefone: apenas números e símbolos</li>
                            <li>Imagem: máx 2MB, 4000x4000px</li>
                            <li>Todos dados são sanitizados</li>
                        </ul>

                        <hr>

                        <h6>Ações Disponíveis</h6>
                        <div class="d-grid gap-2">
                            <a href="{{ route('clientes.show', $cliente->id) }}" class="btn btn-sm btn-outline-info">
                                <i class="ph ph-eye me-1"></i>Visualizar Detalhes
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    {{-- OWASP A03:2021 - Client-side validation (complementar, não substitui server-side) --}}
    {{-- OWASP A05:2021 - CSP-safe inline scripts com nonce (se implementado) --}}
    @push('scripts')
    <script>
        // Preview de nova imagem
        document.getElementById('logomarca').addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                // Validação client-side básica (não confiável, apenas UX)
                if (file.size > 2048000) { // 2MB
                    alert('Arquivo muito grande! Máximo 2MB.');
                    this.value = '';
                    return;
                }

                const reader = new FileReader();
                reader.onload = function(event) {
                    document.getElementById('preview').src = event.target.result;
                    document.getElementById('imagePreview').style.display = 'block';
                };
                reader.readAsDataURL(file);
            }
        });

        // Prevenir duplo submit (OWASP A04:2021 - Insecure Design)
        document.getElementById('formCliente').addEventListener('submit', function() {
            const btn = document.getElementById('btnSubmit');
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Salvando...';
        });

        // Máscara de telefone (opcional, melhora UX)
        const phoneInput = document.getElementById('phone');
        if (phoneInput) {
            phoneInput.addEventListener('input', function(e) {
                let value = e.target.value.replace(/\D/g, '');
                if (value.length > 11) value = value.substring(0, 11);

                if (value.length > 6) {
                    value = value.replace(/^(\d{2})(\d{4,5})(\d{4})$/, '($1) $2-$3');
                } else if (value.length > 2) {
                    value = value.replace(/^(\d{2})(\d{0,5})/, '($1) $2');
                } else if (value.length > 0) {
                    value = value.replace(/^(\d*)/, '($1');
                }

                e.target.value = value;
            });
        }

        // Ocultar preview se checkbox de remover for marcado
        const removeLogoCheckbox = document.getElementById('removeLogo');
        const currentLogo = document.getElementById('currentLogo');
        if (removeLogoCheckbox && currentLogo) {
            removeLogoCheckbox.addEventListener('change', function() {
                if (this.checked) {
                    currentLogo.style.opacity = '0.3';
                    currentLogo.style.filter = 'grayscale(100%)';
                } else {
                    currentLogo.style.opacity = '1';
                    currentLogo.style.filter = 'none';
                }
            });
        }
    </script>
    @endpush
@endsection
