<?php

use App\Http\Controllers\ProfileController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\LocaleController;
use App\Http\Controllers\GroupController;
use App\Http\Controllers\PermissionController;
use App\Http\Controllers\EmpresaController;
use App\Http\Controllers\ClienteController;
use App\Http\Controllers\MidiaController;
use App\Http\Controllers\HotspotController;
use App\Http\Controllers\HotspotCustomizationController;
use App\Http\Controllers\PublicacaoController;
use App\Http\Controllers\EstatisticasController;
use App\Http\Controllers\DebugController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\Auth\RegisterController;
use App\Http\Controllers\PublicClienteRegisterController;
use App\Http\Controllers\PublicidadeCompraController;
use App\Http\Controllers\MercadoPagoConfigController;
use App\Http\Controllers\TawktoConfigController;
use Illuminate\Support\Facades\Route;

Route::get('/', function () {
    return view('welcome');
});

// Página de testes de interface (apenas em desenvolvimento)
Route::get('/teste-interface', function () {
    return view('teste-interface');
})->name('teste.interface');

Route::get('locale/{locale}', [LocaleController::class, 'setLocale'])->name('locale.setting');

// ==========================================
// ROTAS PÚBLICAS (sem autenticação)
// ==========================================

// Cadastro público de cliente via link da franquia
Route::get('/cadastro/cliente/{franquiaToken}', [PublicClienteRegisterController::class, 'showRegistrationForm'])
    ->name('public.register.form');
Route::post('/cadastro/cliente/{franquiaToken}', [PublicClienteRegisterController::class, 'register'])
    ->name('public.register.submit');

// Landing Page de Compra de Publicidade (Público)
Route::prefix('comprar-publicidade')->name('publicidade.compra.')->group(function () {
    Route::get('/', [PublicidadeCompraController::class, 'index'])->name('index');

    // Passo 1: Código da empresa
    Route::get('/passo-1', [PublicidadeCompraController::class, 'passo1'])->name('passo1');
    Route::post('/validar-codigo', [PublicidadeCompraController::class, 'validarCodigo'])->name('validar-codigo');

    // Passo 2: Upload de imagem
    Route::get('/passo-2', [PublicidadeCompraController::class, 'passo2'])->name('passo2');
    Route::post('/upload-imagem', [PublicidadeCompraController::class, 'uploadImagem'])->name('upload-imagem');

    // Passo 3: Seleção de hotspots (oculto por enquanto)
    Route::get('/passo-3', [PublicidadeCompraController::class, 'passo3'])->name('passo3');

    // Passo 4: Datas sugeridas
    Route::get('/passo-4', [PublicidadeCompraController::class, 'passo4'])->name('passo4');
    Route::post('/selecionar-data', [PublicidadeCompraController::class, 'selecionarData'])->name('selecionar-data');

    // Passo 5: Posição (oculto por enquanto)
    Route::get('/passo-5', [PublicidadeCompraController::class, 'passo5'])->name('passo5');

    // Passo 6: Periodicidade + Resumo
    Route::get('/passo-6', [PublicidadeCompraController::class, 'passo6'])->name('passo6');
    Route::post('/selecionar-periodicidade', [PublicidadeCompraController::class, 'selecionarPeriodicidade'])->name('selecionar-periodicidade');

    // Passo 7: Pagamento
    Route::get('/passo-7', [PublicidadeCompraController::class, 'passo7'])->name('passo7');
    Route::post('/processar-pagamento', [PublicidadeCompraController::class, 'processarPagamento'])->name('processar-pagamento');

    // Página de sucesso
    Route::get('/sucesso/{pedido}', [PublicidadeCompraController::class, 'sucesso'])->name('sucesso');
});

// ==========================================
// REGISTRO COM CÓDIGO DE CONVITE (Guest)
// ==========================================
Route::middleware('guest')->group(function () {
    Route::get('/register', [RegisterController::class, 'showForm'])->name('register');
    Route::post('/register', [RegisterController::class, 'register'])->name('register.store');
    Route::post('/register/validate-code', [RegisterController::class, 'validateCode'])->name('register.validate-code');
});

Route::get('/dashboard', [DashboardController::class, 'index'])
    ->middleware(['auth', 'verified', 'check.approval', 'check.empresa.aprovada'])
    ->name('dashboard');

Route::get('/hierarquia', [App\Http\Controllers\HierarchyController::class, 'index'])
    ->middleware(['auth', 'verified', 'check.approval', 'check.empresa.aprovada'])
    ->name('hierarchy.index');

Route::middleware(['auth', 'check.approval', 'check.empresa.aprovada'])->group(function () {
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');

    // User Management
    Route::get('users/trash', [UserController::class, 'trash'])->name('users.trash');
    Route::patch('users/{id}/restore', [UserController::class, 'restore'])->name('users.restore');
    Route::patch('users/{user}/approve', [UserController::class, 'approve'])->name('users.approve');
    Route::patch('users/{user}/reject', [UserController::class, 'reject'])->name('users.reject');
    Route::resource('users', UserController::class);

    // RBAC Routes
    Route::resource('permissions', PermissionController::class);
    Route::resource('groups', GroupController::class);
    Route::get('groups/{group}/permissions', [GroupController::class,'editPermissions'])->name('groups.permissions.edit');
    Route::post('groups/{group}/permissions', [GroupController::class,'updatePermissions'])->name('groups.permissions.update');

    // ==========================================
    // EMPRESAS - APENAS SUPERADMIN
    // OWASP A01:2021 – Broken Access Control
    // ==========================================
    Route::middleware('superadmin')->group(function () {
        Route::get('empresas/trash', [EmpresaController::class, 'trash'])->name('empresas.trash');
        Route::patch('empresas/{id}/restore', [EmpresaController::class, 'restore'])->name('empresas.restore');
        Route::patch('empresas/{empresa}/status', [EmpresaController::class, 'toggleStatus'])->name('empresas.toggleStatus');
        Route::resource('empresas', EmpresaController::class);

        // Configuração Mercado Pago por empresa
        Route::prefix('empresas/{empresa}')->name('empresas.mercadopago.')->group(function () {
            Route::get('/mercadopago', [MercadoPagoConfigController::class, 'edit'])->name('edit');
            Route::put('/mercadopago', [MercadoPagoConfigController::class, 'update'])->name('update');
            Route::post('/mercadopago/testar', [MercadoPagoConfigController::class, 'testarConexao'])->name('testar');
            Route::post('/mercadopago/desativar', [MercadoPagoConfigController::class, 'desativar'])->name('desativar');
        });
    });

    // ==========================================
    // CLIENTES - APENAS FRANQUIAS
    // OWASP A01:2021 – Broken Access Control
    // OWASP A07:2021 – Rate Limiting
    // ==========================================
    Route::middleware(['franchise', 'throttle:60,1'])->prefix('clientes')->name('clientes.')->group(function () {
        // Lista de clientes (GET)
        Route::get('/', [ClienteController::class, 'index'])->name('index');

        // Criar cliente (GET + POST com rate limit mais restritivo)
        Route::get('/create', [ClienteController::class, 'create'])->name('create');
        Route::post('/', [ClienteController::class, 'store'])
            ->middleware('throttle:10,1') // Máximo 10 criações por minuto
            ->name('store');

        // Aprovação de cadastros de clientes (Franqueado OU SuperAdmin)
        Route::post('/{cliente}/approve', [ClienteController::class, 'approve'])->name('approve');
        Route::post('/{cliente}/reject', [ClienteController::class, 'reject'])->name('reject');

        // Visualizar cliente (GET)
        Route::get('/{cliente}', [ClienteController::class, 'show'])->name('show');

        // Editar cliente (GET + PUT/PATCH)
        Route::get('/{cliente}/edit', [ClienteController::class, 'edit'])->name('edit');
        Route::match(['put', 'patch'], '/{cliente}', [ClienteController::class, 'update'])
            ->middleware('throttle:20,1') // Máximo 20 atualizações por minuto
            ->name('update');

        // Deletar/Inativar cliente (DELETE)
        Route::delete('/{cliente}', [ClienteController::class, 'destroy'])
            ->middleware('throttle:10,1') // Máximo 10 exclusões por minuto
            ->name('destroy');
    });
    Route::get('midias/trash', [MidiaController::class, 'trash'])->name('midias.trash');
    Route::patch('midias/{midia}/restore', [MidiaController::class, 'restore'])->name('midias.restore');
    Route::patch('midias/{midia}/status', [MidiaController::class, 'toggleStatus'])->name('midias.toggleStatus');
    Route::resource('midias', MidiaController::class);
    
    // Hotspots Management
    Route::get('hotspots/trash', [HotspotController::class, 'trash'])->name('hotspots.trash');
    Route::patch('hotspots/{id}/restore', [HotspotController::class, 'restore'])->name('hotspots.restore');
    Route::patch('hotspots/{hotspot}/status', [HotspotController::class, 'toggleStatus'])->name('hotspots.toggleStatus');
    Route::patch('hotspots/{hotspot}/situacao', [HotspotController::class, 'updateSituacao'])->name('hotspots.updateSituacao');
    Route::post('hotspots/{hotspot}/approve', [HotspotController::class, 'approve'])->name('hotspots.approve');
    Route::post('hotspots/{hotspot}/reject', [HotspotController::class, 'reject'])->name('hotspots.reject');
    Route::resource('hotspots', HotspotController::class);

    // Hotspot Customizations Management
    Route::get('customizations/{customization}/preview', [HotspotCustomizationController::class, 'preview'])->name('customizations.preview');
    Route::patch('customizations/{customization}/toggle', [HotspotCustomizationController::class, 'toggle'])->name('customizations.toggle');
    Route::resource('customizations', HotspotCustomizationController::class);

    // Tawk.to Configurations Management
    Route::resource('tawkto', TawktoConfigController::class);

    // Publicações Management
    Route::get('publicacoes/criativo/{id}', [PublicacaoController::class, 'buscarCriativo'])->name('publicacoes.buscarCriativo');
    Route::get('publicacoes/{publicacao}/estatisticas', [PublicacaoController::class, 'estatisticas'])->name('publicacoes.estatisticas');
    Route::get('publicacoes/{publicacao}/analytics-json', [PublicacaoController::class, 'analyticsJson'])->name('publicacoes.analyticsJson');
    Route::patch('publicacoes/{publicacao}/status', [PublicacaoController::class, 'toggleStatus'])->name('publicacoes.toggleStatus');
    Route::resource('publicacoes', PublicacaoController::class)->parameters(['publicacoes' => 'publicacao']);
    
    // Estatísticas
    Route::prefix('estatisticas')->name('estatisticas.')->group(function () {
        Route::get('dashboard', [EstatisticasController::class, 'dashboard'])->name('dashboard');
        Route::get('por-publicacao', [EstatisticasController::class, 'porPublicacao'])->name('por-publicacao');
        Route::get('por-hotspot', [EstatisticasController::class, 'porHotspot'])->name('por-hotspot');
        Route::get('graficos-json', [EstatisticasController::class, 'graficosJson'])->name('graficos-json');
    });
    
    // API para incrementar visualizações (pode ser movida para api.php se necessário)
    Route::post('api/publicacoes/{id}/visualizacoes', [PublicacaoController::class, 'incrementarVisualizacoes'])->name('api.publicacoes.incrementarVisualizacoes');

    // Debug - Auditoria Splash (remover em produção)
    Route::get('debug/auditoria-splash', [DebugController::class, 'auditoriaSplash'])->name('debug.auditoria-splash');
});

require __DIR__.'/auth.php';
