#!/bin/bash

###############################################################################
# Script para Configuração de Secrets do GitHub Actions
# FreeFi Admin - Deploy Pipeline
###############################################################################

echo "╔════════════════════════════════════════════════════════════════════╗"
echo "║  CONFIGURAÇÃO DE SECRETS - GITHUB ACTIONS                          ║"
echo "║  FreeFi Admin Deploy Pipeline                                      ║"
echo "╚════════════════════════════════════════════════════════════════════╝"
echo ""

# Cores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Função para printar com cores
print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_info() {
    echo -e "${BLUE}ℹ $1${NC}"
}

echo ""
print_info "Este script irá ajudá-lo a coletar as informações necessárias"
print_info "para configurar os secrets no GitHub Actions."
echo ""
print_warning "ATENÇÃO: As informações coletadas são SENSÍVEIS!"
print_warning "Não compartilhe o arquivo gerado com ninguém."
echo ""

# Perguntar se quer continuar
read -p "Deseja continuar? (s/n): " continuar
if [ "$continuar" != "s" ] && [ "$continuar" != "S" ]; then
    print_warning "Operação cancelada."
    exit 0
fi

echo ""
echo "════════════════════════════════════════════════════════════════════"
echo " COLETANDO INFORMAÇÕES"
echo "════════════════════════════════════════════════════════════════════"
echo ""

# 1. SSH_HOST
print_info "1/6 - SSH_HOST"
echo "   Qual é o IP ou hostname do servidor Linode?"
echo "   Exemplo: 192.168.1.100 ou freefi.seudominio.com.br"
read -p "   SSH_HOST: " SSH_HOST

# 2. SSH_USERNAME
print_info "2/6 - SSH_USERNAME"
echo "   Qual é o usuário SSH do servidor?"
echo "   Exemplos: cloud, deploy, root"
read -p "   SSH_USERNAME: " SSH_USERNAME

# 3. SSH_PORT
print_info "3/6 - SSH_PORT"
echo "   Qual é a porta SSH do servidor? (padrão: 22)"
read -p "   SSH_PORT [22]: " SSH_PORT
SSH_PORT=${SSH_PORT:-22}

# 4. SSH_PRIVATE_KEY
print_info "4/6 - SSH_PRIVATE_KEY"
echo "   Onde está a chave privada SSH?"
echo "   Caminho padrão: ~/.ssh/id_rsa"
read -p "   Caminho da chave [~/.ssh/id_rsa]: " SSH_KEY_PATH
SSH_KEY_PATH=${SSH_KEY_PATH:-~/.ssh/id_rsa}
SSH_KEY_PATH="${SSH_KEY_PATH/#\~/$HOME}"

if [ ! -f "$SSH_KEY_PATH" ]; then
    print_error "Chave não encontrada: $SSH_KEY_PATH"
    print_warning "Você precisa:"
    echo "   1. Gerar uma chave: ssh-keygen -t rsa -b 4096"
    echo "   2. Copiar para o servidor: ssh-copy-id $SSH_USERNAME@$SSH_HOST"
    echo "   3. Executar este script novamente"
    exit 1
fi

SSH_PRIVATE_KEY=$(cat "$SSH_KEY_PATH")
print_success "Chave privada carregada!"

# 5. PROJECT_PATH
print_info "5/6 - PROJECT_PATH"
echo "   Qual é o caminho do projeto no servidor?"
echo "   Exemplos:"
echo "   - /var/www/freefi"
echo "   - /home/cloud/freefi"
echo "   - /var/www/html/laravel/admin/freefi_admin"
read -p "   PROJECT_PATH: " PROJECT_PATH

# 6. GH_PAT
print_info "6/6 - GH_PAT (GitHub Personal Access Token)"
echo "   Se você já tem um token, cole aqui."
echo "   Se não tem, acesse:"
echo "   https://github.com/settings/tokens/new"
echo ""
echo "   Configure o token com:"
echo "   - Nome: FreeFi Deploy"
echo "   - Scopes: repo (all), write:packages, read:packages"
echo ""
read -p "   GH_PAT: " GH_PAT

# Testar conexão SSH
echo ""
echo "════════════════════════════════════════════════════════════════════"
echo " TESTANDO CONEXÃO SSH"
echo "════════════════════════════════════════════════════════════════════"
echo ""

print_info "Testando conexão SSH com $SSH_USERNAME@$SSH_HOST:$SSH_PORT..."

# Salvar chave temporariamente
TEMP_KEY=$(mktemp)
echo "$SSH_PRIVATE_KEY" > "$TEMP_KEY"
chmod 600 "$TEMP_KEY"

# Testar conexão
if ssh -i "$TEMP_KEY" -p "$SSH_PORT" -o StrictHostKeyChecking=no -o ConnectTimeout=10 \
   "$SSH_USERNAME@$SSH_HOST" "echo 'Conexão OK'" &>/dev/null; then
    print_success "Conexão SSH bem-sucedida!"
else
    print_error "Falha na conexão SSH!"
    print_warning "Verifique:"
    echo "   - IP/hostname correto: $SSH_HOST"
    echo "   - Usuário correto: $SSH_USERNAME"
    echo "   - Porta correta: $SSH_PORT"
    echo "   - Chave pública está no servidor: ~/.ssh/authorized_keys"
    rm "$TEMP_KEY"
    exit 1
fi

# Verificar se projeto existe no servidor
print_info "Verificando se o projeto existe em $PROJECT_PATH..."

if ssh -i "$TEMP_KEY" -p "$SSH_PORT" -o StrictHostKeyChecking=no \
   "$SSH_USERNAME@$SSH_HOST" "[ -d '$PROJECT_PATH' ]" &>/dev/null; then
    print_success "Diretório encontrado!"
else
    print_warning "Diretório NÃO encontrado: $PROJECT_PATH"
    print_info "Você precisará criar o diretório e clonar o repositório:"
    echo "   ssh $SSH_USERNAME@$SSH_HOST"
    echo "   mkdir -p $PROJECT_PATH"
    echo "   cd $PROJECT_PATH"
    echo "   git clone git@github.com:freefi-com-br/freefiadm.git ."
fi

# Limpar chave temporária
rm "$TEMP_KEY"

# Gerar arquivo de configuração
echo ""
echo "════════════════════════════════════════════════════════════════════"
echo " GERANDO ARQUIVO DE CONFIGURAÇÃO"
echo "════════════════════════════════════════════════════════════════════"
echo ""

OUTPUT_FILE="github-secrets-$(date +%Y%m%d-%H%M%S).txt"

cat > "$OUTPUT_FILE" << EOF
╔════════════════════════════════════════════════════════════════════╗
║  GITHUB ACTIONS SECRETS - FREEFI ADMIN                             ║
║  Gerado em: $(date '+%d/%m/%Y %H:%M:%S')                                      ║
╚════════════════════════════════════════════════════════════════════╝

⚠️  ATENÇÃO: Este arquivo contém informações SENSÍVEIS!
    - NÃO commite este arquivo no Git
    - NÃO compartilhe com ninguém
    - DELETE após configurar os secrets
    - Mantenha em local seguro (ex: 1Password, LastPass)

════════════════════════════════════════════════════════════════════

INSTRUÇÕES PARA CONFIGURAR SECRETS NO GITHUB:

1. Acesse: https://github.com/freefi-com-br/freefiadm/settings/secrets/actions

2. Clique em "New repository secret" para cada secret abaixo

3. Copie e cole o valor exatamente como está (incluindo quebras de linha)

════════════════════════════════════════════════════════════════════

SECRET 1: SSH_HOST
────────────────────────────────────────────────────────────────────
$SSH_HOST
────────────────────────────────────────────────────────────────────


SECRET 2: SSH_USERNAME
────────────────────────────────────────────────────────────────────
$SSH_USERNAME
────────────────────────────────────────────────────────────────────


SECRET 3: SSH_PORT
────────────────────────────────────────────────────────────────────
$SSH_PORT
────────────────────────────────────────────────────────────────────


SECRET 4: PROJECT_PATH
────────────────────────────────────────────────────────────────────
$PROJECT_PATH
────────────────────────────────────────────────────────────────────


SECRET 5: GH_PAT
────────────────────────────────────────────────────────────────────
$GH_PAT
────────────────────────────────────────────────────────────────────


SECRET 6: SSH_PRIVATE_KEY
────────────────────────────────────────────────────────────────────
$SSH_PRIVATE_KEY
────────────────────────────────────────────────────────────────────

════════════════════════════════════════════════════════════════════

PRÓXIMOS PASSOS:

1. Configure todos os 6 secrets acima no GitHub

2. No servidor, certifique-se de que:
   [ ] Docker está instalado
   [ ] Docker Compose está instalado
   [ ] Projeto está clonado em: $PROJECT_PATH
   [ ] Arquivo .env está configurado
   [ ] Porta 8082 está livre

3. Faça um push para main para testar:
   git add .
   git commit -m "test: Deploy pipeline"
   git push origin main

4. Acompanhe em: https://github.com/freefi-com-br/freefiadm/actions

════════════════════════════════════════════════════════════════════
EOF

print_success "Arquivo gerado: $OUTPUT_FILE"
echo ""
print_warning "⚠️  LEMBRE-SE: DELETE este arquivo após configurar os secrets!"
echo ""
print_info "Para visualizar o arquivo:"
echo "   cat $OUTPUT_FILE"
echo ""
print_info "Para copiar um secret específico:"
echo "   # SSH_HOST"
echo "   echo '$SSH_HOST' | pbcopy    # Mac"
echo "   echo '$SSH_HOST' | xclip -selection clipboard    # Linux"
echo ""

# Oferecer para abrir o arquivo
if command -v xdg-open &> /dev/null; then
    read -p "Deseja abrir o arquivo agora? (s/n): " abrir
    if [ "$abrir" = "s" ] || [ "$abrir" = "S" ]; then
        xdg-open "$OUTPUT_FILE"
    fi
elif command -v open &> /dev/null; then
    read -p "Deseja abrir o arquivo agora? (s/n): " abrir
    if [ "$abrir" = "s" ] || [ "$abrir" = "S" ]; then
        open "$OUTPUT_FILE"
    fi
fi

echo ""
print_success "✓ Configuração concluída!"
echo ""
