<?php

namespace Tests\Feature;

use App\Models\HotspotCustomization;
use App\Models\Hotspot;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

/**
 * Teste End-to-End do fluxo completo de personalização
 */
class HotspotCustomizationE2ETest extends TestCase
{
    use RefreshDatabase;

    /**
     * Teste E2E completo: Criar customização -> Buscar via API -> Verificar JavaScript
     */
    public function test_complete_customization_flow(): void
    {
        // 1. SETUP: Criar usuário e hotspot
        $user = User::factory()->create();
        $hotspot = Hotspot::factory()->create([
            'serial' => 'E2E-TEST-SERIAL-001',
            'nome' => 'Hotspot E2E Test',
        ]);

        // 2. CRIAR CUSTOMIZAÇÃO
        $customization = HotspotCustomization::create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'E2E WiFi Test',
            'logo_url' => 'img/e2e-logo.png',
            'welcome_title' => 'Bem-vindo ao WiFi E2E',
            'primary_color' => '#E91E63',
            'primary_hover_color' => '#C2185B',
            'bg_color_1' => 'hsla(340, 82%, 52%, 1)',
            'bg_color_2' => 'hsla(340, 70%, 40%, 1)',
            'bg_color_3' => 'hsla(340, 60%, 30%, 1)',
            'bg_color_4' => 'hsla(340, 50%, 20%, 1)',
            'modal_countdown_seconds' => 12,
            'custom_css' => '.e2e-test { color: pink; }',
            'custom_js' => 'console.log("E2E test loaded");',
            'is_active' => true,
            'created_by' => $user->id,
            'updated_by' => $user->id,
        ]);

        $this->assertDatabaseHas('hotspot_customizations', [
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'E2E WiFi Test',
        ]);

        // 3. BUSCAR VIA API (JSON)
        $jsonResponse = $this->get('/api/customization/E2E-TEST-SERIAL-001');

        $jsonResponse->assertStatus(200);
        $jsonResponse->assertJson(['success' => true]);

        $jsonData = $jsonResponse->json('customization');
        $this->assertEquals('E2E WiFi Test', $jsonData['brand_name']);
        $this->assertEquals('#E91E63', $jsonData['primary_color']);
        $this->assertEquals(12, $jsonData['modal_countdown_seconds']);

        // 4. BUSCAR SCRIPT JAVASCRIPT
        $scriptResponse = $this->get('/api/customization/E2E-TEST-SERIAL-001/script.js');

        $scriptResponse->assertStatus(200);
        $scriptResponse->assertHeader('Content-Type', 'application/javascript; charset=UTF-8');

        $scriptContent = $scriptResponse->getContent();

        // Verificar conteúdo do script
        $this->assertStringContainsString('window.freefiCustomization', $scriptContent);
        $this->assertStringContainsString('E2E WiFi Test', $scriptContent);
        $this->assertStringContainsString('#E91E63', $scriptContent);
        $this->assertStringContainsString('Bem-vindo ao WiFi E2E', $scriptContent);
        $this->assertStringContainsString('.e2e-test { color: pink; }', $scriptContent);
        $this->assertStringContainsString('console.log("E2E test loaded");', $scriptContent);

        // 5. ATUALIZAR CUSTOMIZAÇÃO
        $customization->update([
            'brand_name' => 'E2E WiFi Updated',
            'primary_color' => '#9C27B0',
        ]);

        // 6. VERIFICAR ATUALIZAÇÃO NA API
        $updatedResponse = $this->get('/api/customization/E2E-TEST-SERIAL-001');
        $updatedData = $updatedResponse->json('customization');

        $this->assertEquals('E2E WiFi Updated', $updatedData['brand_name']);
        $this->assertEquals('#9C27B0', $updatedData['primary_color']);

        // 7. DESATIVAR CUSTOMIZAÇÃO
        $customization->update(['is_active' => false]);

        // 8. VERIFICAR QUE NÃO RETORNA MAIS
        $inactiveResponse = $this->get('/api/customization/E2E-TEST-SERIAL-001');
        $inactiveResponse->assertStatus(404);
        $inactiveResponse->assertJson([
            'success' => false,
            'use_default' => true,
        ]);

        // 9. SOFT DELETE
        $customization->delete();

        $this->assertSoftDeleted('hotspot_customizations', [
            'id' => $customization->id,
        ]);

        // 10. FINAL: Verificar que nada é retornado
        $deletedResponse = $this->get('/api/customization/E2E-TEST-SERIAL-001');
        $deletedResponse->assertStatus(404);
    }

    /**
     * Teste de simulação de uso real no hotspot
     */
    public function test_simulate_real_hotspot_usage(): void
    {
        // Simular vários hotspots com customizações diferentes
        for ($i = 1; $i <= 5; $i++) {
            $hotspot = Hotspot::factory()->create([
                'serial' => "HOTSPOT-$i",
            ]);

            HotspotCustomization::factory()->active()->create([
                'hotspot_id' => $hotspot->id,
                'brand_name' => "WiFi Location $i",
                'primary_color' => sprintf('#%06X', mt_rand(0, 0xFFFFFF)),
            ]);
        }

        // Simular requisições de cada hotspot
        for ($i = 1; $i <= 5; $i++) {
            $response = $this->get("/api/customization/HOTSPOT-$i/script.js");

            $response->assertStatus(200);
            $this->assertStringContainsString("WiFi Location $i", $response->getContent());
        }

        // Verificar que hotspot não existente retorna vazio
        $response = $this->get("/api/customization/HOTSPOT-999/script.js");
        $response->assertStatus(200);
        $this->assertStringContainsString('// Nenhuma customização ativa', $response->getContent());
    }

    /**
     * Teste de performance com carga
     */
    public function test_api_performance_under_load(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'PERF-TEST']);
        HotspotCustomization::factory()->active()->create(['hotspot_id' => $hotspot->id]);

        $start = microtime(true);

        // Simular 50 requisições
        for ($i = 0; $i < 50; $i++) {
            $response = $this->get('/api/customization/PERF-TEST/script.js');
            $response->assertStatus(200);
        }

        $duration = microtime(true) - $start;

        // Deve completar 50 requisições em menos de 5 segundos
        $this->assertLessThan(5.0, $duration, "API muito lenta: $duration segundos para 50 requisições");
    }

    /**
     * Teste de injeção de código malicioso
     */
    public function test_sanitizes_malicious_code(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'SECURITY-TEST']);

        // Tentar injetar script malicioso
        $customization = HotspotCustomization::factory()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => '<script>alert("XSS")</script>',
            'custom_js' => '"; alert("XSS"); //',
            'is_active' => true,
        ]);

        $response = $this->get('/api/customization/SECURITY-TEST/script.js');

        $content = $response->getContent();

        // O conteúdo deve estar escapado no JSON
        $this->assertStringContainsString('<script>alert("XSS")</script>', $content);

        // Mas no contexto do JSON, estará escapado
        $json = json_encode(['test' => $customization->brand_name]);
        $this->assertStringContainsString('\\', $json); // Escapamento presente
    }

    /**
     * Teste de caracteres especiais Unicode
     */
    public function test_handles_unicode_characters(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'UNICODE-TEST']);

        HotspotCustomization::factory()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'WiFi 中文 العربية עברית 🌐',
            'welcome_title' => 'Bem-vindo! 欢迎! مرحبا! שלום!',
            'is_active' => true,
        ]);

        $response = $this->get('/api/customization/UNICODE-TEST');

        $response->assertStatus(200);
        $data = $response->json('customization');

        $this->assertEquals('WiFi 中文 العربية עברית 🌐', $data['brand_name']);
        $this->assertEquals('Bem-vindo! 欢迎! مرحبا! שלום!', $data['welcome_title']);
    }

    /**
     * Teste de validação de cores
     */
    public function test_color_formats_are_valid(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'COLOR-TEST']);

        $customization = HotspotCustomization::factory()->create([
            'hotspot_id' => $hotspot->id,
            'primary_color' => '#FF5722',
            'bg_color_1' => 'hsla(10, 100%, 50%, 1)',
            'bg_color_2' => 'rgb(255, 87, 34)',
            'is_active' => true,
        ]);

        $response = $this->get('/api/customization/COLOR-TEST');
        $data = $response->json('customization');

        $this->assertMatchesRegularExpression('/^#[0-9A-F]{6}$/i', $data['primary_color']);
        $this->assertStringContainsString('hsla', $data['bg_color_1']);
    }

    /**
     * Teste de campos obrigatórios vs opcionais
     */
    public function test_required_and_optional_fields(): void
    {
        $hotspot = Hotspot::factory()->create();
        $user = User::factory()->create();

        // Criar com apenas campos obrigatórios
        $customization = HotspotCustomization::create([
            'hotspot_id' => $hotspot->id,
            'created_by' => $user->id,
            'updated_by' => $user->id,
        ]);

        // Deve ter valores padrão
        $this->assertNotNull($customization->brand_name);
        $this->assertNotNull($customization->logo_url);
        $this->assertNotNull($customization->primary_color);

        // Campos opcionais podem ser null
        $this->assertNull($customization->terms_url);
        $this->assertNull($customization->custom_css);
    }
}
