<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Midia;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

class MidiaEndToEndTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $empresa;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->empresa = Empresa::factory()->create();
        $this->actingAs($this->user);
        Storage::fake('public');
    }

    public function test_complete_midia_lifecycle()
    {
        // 1. Visualizar listagem vazia
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee('Nenhuma mídia encontrada');

        // 2. Acessar formulário de criação
        $response = $this->get(route('midias.create'));
        $response->assertStatus(200);
        $response->assertSee('Criar Mídia');

        // 3. Criar nova mídia com upload
        $file = UploadedFile::fake()->image('test.jpg', 1920, 1080);
        $response = $this->post(route('midias.store'), [
            'title' => 'Mídia de Teste E2E',
            'type' => 'image',
            'media_file' => $file,
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'E2E-001',
            'position' => 1,
            'start_datetime' => '2025-01-01 10:00:00',
            'end_datetime' => '2025-01-31 18:00:00',
        ]);

        $response->assertRedirect(route('midias.index'));
        
        // 4. Verificar mídia na listagem
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee('Mídia de Teste E2E');
        $response->assertSee($this->empresa->name);
        $response->assertSee('E2E-001');

        // 5. Acessar formulário de edição
        $midia = Midia::where('title', 'Mídia de Teste E2E')->first();
        $response = $this->get(route('midias.edit', $midia));
        $response->assertStatus(200);
        $response->assertSee('Editar Mídia');
        $response->assertSee('Mídia de Teste E2E');

        // 6. Atualizar mídia
        $response = $this->put(route('midias.update', $midia), [
            'title' => 'Mídia de Teste E2E Atualizada',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'E2E-001-UPDATED',
            'position' => 2,
            'start_datetime' => '2025-01-01 10:00:00',
            'end_datetime' => '2025-01-31 18:00:00',
        ]);

        $response->assertRedirect(route('midias.index'));

        // 7. Verificar atualização na listagem
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee('Mídia de Teste E2E Atualizada');
        $response->assertSee('E2E-001-UPDATED');

        // 8. Alternar status para inativo
        $response = $this->patch(route('midias.toggleStatus', $midia));
        $response->assertRedirect(route('midias.index'));

        // 9. Verificar mídia inativa ainda aparece na listagem (mas com estilo diferente)
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee('Mídia de Teste E2E Atualizada');

        // 10. Verificar status foi alterado no banco
        $midia->refresh();
        $this->assertEquals(1, $midia->status); // 1 = inativo

        // 11. Reativar mídia
        $response = $this->patch(route('midias.toggleStatus', $midia));
        $response->assertRedirect(route('midias.index'));

        $midia->refresh();
        $this->assertEquals(0, $midia->status); // 0 = ativo

        // 12. Fazer soft delete
        $response = $this->delete(route('midias.destroy', $midia));
        $response->assertRedirect(route('midias.index'));

        // 13. Verificar mídia não aparece mais na listagem principal
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertDontSee('Mídia de Teste E2E Atualizada');

        // 14. Verificar mídia na lixeira via API
        $response = $this->get(route('midias.trash'));
        $response->assertStatus(200);

        // Verificar diretamente no banco que foi feito soft delete
        $this->assertSoftDeleted('midias', ['id' => $midia->id]);

        // 15. Restaurar mídia da lixeira
        $response = $this->patch(route('midias.restore', $midia->id));
        $response->assertRedirect(route('midias.trash'));

        // 16. Verificar mídia restaurada aparece na listagem principal novamente
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee('Mídia de Teste E2E Atualizada');

        // 17. Verificar que arquivo foi mantido durante todo o processo
        $midia->refresh();
        $this->assertTrue(Storage::disk('public')->exists($midia->path));
    }

    public function test_search_functionality()
    {
        // Criar múltiplas mídias
        Midia::factory()->create([
            'title' => 'Propaganda de Verão',
            'empresa_id' => $this->empresa->id,
        ]);

        Midia::factory()->create([
            'title' => 'Campanha de Inverno',
            'empresa_id' => $this->empresa->id,
        ]);

        Midia::factory()->create([
            'title' => 'Promoção de Natal',
            'empresa_id' => $this->empresa->id,
        ]);

        // Teste busca com termo específico
        $response = $this->get(route('midias.index', ['q' => 'Verão']));
        $response->assertStatus(200);
        $response->assertSee('Propaganda de Verão');
        $response->assertDontSee('Campanha de Inverno');
        $response->assertDontSee('Promoção de Natal');

        // Teste busca com termo parcial
        $response = $this->get(route('midias.index', ['q' => 'Promo']));
        $response->assertStatus(200);
        $response->assertSee('Promoção de Natal');
        $response->assertDontSee('Propaganda de Verão');
        $response->assertDontSee('Campanha de Inverno');

        // Teste busca com termo não encontrado
        $response = $this->get(route('midias.index', ['q' => 'Inexistente']));
        $response->assertStatus(200);
        $response->assertDontSee('Propaganda de Verão');
        $response->assertDontSee('Campanha de Inverno');
        $response->assertDontSee('Promoção de Natal');
    }

    public function test_pagination_with_search()
    {
        // Criar 30 mídias para testar paginação (15 por página = 2 páginas)
        for ($i = 1; $i <= 30; $i++) {
            Midia::factory()->create([
                'title' => "Mídia Teste {$i}",
                'empresa_id' => $this->empresa->id,
            ]);
        }

        // Teste primeira página - verificar se há conteúdo
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        $response->assertSee('Mídia Teste'); // Pelo menos uma mídia

        // Teste segunda página - verificar se há paginação
        $response = $this->get(route('midias.index', ['page' => 2]));
        $response->assertStatus(200);
        $response->assertSee('Mídia Teste'); // Pelo menos uma mídia na segunda página

        // Teste busca específica
        $response = $this->get(route('midias.index', ['q' => 'Teste 25']));
        $response->assertStatus(200);
        $response->assertSee('Mídia Teste 25');
        $response->assertDontSee('Mídia Teste 1'); // Não deve mostrar outros
    }

    public function test_multiple_users_interaction()
    {
        // Criar outro usuário
        $otherUser = User::factory()->create();
        
        // Usuário 1 cria mídia
        $response = $this->post(route('midias.store'), [
            'title' => 'Mídia do Usuário 1',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
        ]);

        $midia = Midia::where('title', 'Mídia do Usuário 1')->first();
        $this->assertEquals($this->user->id, $midia->created_by);

        // Usuário 2 faz login e edita a mídia
        $this->actingAs($otherUser);
        
        $response = $this->put(route('midias.update', $midia), [
            'title' => 'Mídia Editada pelo Usuário 2',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
        ]);

        $midia->refresh();
        $this->assertEquals($this->user->id, $midia->created_by); // Criador original
        $this->assertEquals($otherUser->id, $midia->updated_by); // Editor atual

        // Usuário 1 volta e deleta
        $this->actingAs($this->user);
        
        $response = $this->delete(route('midias.destroy', $midia));
        
        $midia->refresh();
        $this->assertEquals($this->user->id, $midia->deleted_by);
    }

    public function test_empresa_relationship_consistency()
    {
        // Criar duas empresas
        $empresa1 = Empresa::factory()->create(['name' => 'Empresa 1']);
        $empresa2 = Empresa::factory()->create(['name' => 'Empresa 2']);

        // Criar mídia para empresa 1
        $response = $this->post(route('midias.store'), [
            'title' => 'Mídia da Empresa 1',
            'type' => 'image',
            'empresa_id' => $empresa1->id,
        ]);

        $midia = Midia::where('title', 'Mídia da Empresa 1')->first();
        $this->assertEquals($empresa1->id, $midia->empresa_id);

        // Verificar relacionamento
        $this->assertEquals('Empresa 1', $midia->empresa->name);

        // Mudar para empresa 2
        $response = $this->put(route('midias.update', $midia), [
            'title' => 'Mídia da Empresa 1',
            'type' => 'image',
            'empresa_id' => $empresa2->id,
        ]);

        $midia->refresh();
        $this->assertEquals($empresa2->id, $midia->empresa_id);
        $this->assertEquals('Empresa 2', $midia->empresa->name);

        // Verificar contadores de relacionamento
        $this->assertEquals(0, $empresa1->midias()->count());
        $this->assertEquals(1, $empresa2->midias()->count());
    }

    public function test_error_handling_and_validation_flow()
    {
        // Teste validação de campos obrigatórios
        $response = $this->post(route('midias.store'), []);
        $response->assertSessionHasErrors(['title', 'type', 'empresa_id']);
        $response->assertRedirect();

        // Teste validação de tipo inválido
        $response = $this->post(route('midias.store'), [
            'title' => 'Teste',
            'type' => 'audio', // Tipo não permitido
            'empresa_id' => $this->empresa->id,
        ]);
        $response->assertSessionHasErrors(['type']);

        // Teste validação de empresa inexistente
        $response = $this->post(route('midias.store'), [
            'title' => 'Teste',
            'type' => 'image',
            'empresa_id' => 99999, // ID inexistente
        ]);
        $response->assertSessionHasErrors(['empresa_id']);

        // Teste validação de posição inválida
        $response = $this->post(route('midias.store'), [
            'title' => 'Teste',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'position' => 10, // Fora do range 1-6
        ]);
        $response->assertSessionHasErrors(['position']);

        // Teste validação de datas
        $response = $this->post(route('midias.store'), [
            'title' => 'Teste',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'start_datetime' => '2025-12-31',
            'end_datetime' => '2025-01-01', // Data fim anterior ao início
        ]);
        $response->assertSessionHasErrors(['end_datetime']);

        // Verificar que nenhuma mídia foi criada após validações falharem
        $this->assertEquals(0, Midia::count());
    }
}