<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Models\User;
use App\Models\Midia;
use App\Models\Empresa;

class MidiaModuleTest extends TestCase
{
    use RefreshDatabase;

    private User $user;
    private Empresa $empresa;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->empresa = Empresa::factory()->create();
        $this->actingAs($this->user);
    }

    public function test_can_soft_delete_a_midia()
    {
        $midia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);

        $response = $this->delete(route('midias.destroy', $midia));

        $response->assertRedirect(route('midias.index'));
        $this->assertSoftDeleted('midias', ['id' => $midia->id]);
    }

    public function test_soft_deleted_midia_is_not_in_index_list()
    {
        $midia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);
        $midia->delete(); // Soft delete

        $response = $this->get(route('midias.index'));

        $response->assertStatus(200);
        $response->assertDontSee($midia->title);
    }

    public function test_can_toggle_status_from_active_to_inactive()
    {
        $midia = Midia::factory()->create([
            'empresa_id' => $this->empresa->id,
            'status' => 0, // Active
        ]);

        $response = $this->patch(route('midias.toggleStatus', $midia));

        $response->assertRedirect(route('midias.index'));
        $this->assertDatabaseHas('midias', [
            'id' => $midia->id,
            'status' => 1, // Inactive
        ]);
    }

    public function test_can_toggle_status_from_inactive_to_active()
    {
        $midia = Midia::factory()->create([
            'empresa_id' => $this->empresa->id,
            'status' => 1, // Inactive
        ]);

        $response = $this->patch(route('midias.toggleStatus', $midia));

        $response->assertRedirect(route('midias.index'));
        $this->assertDatabaseHas('midias', [
            'id' => $midia->id,
            'status' => 0, // Active
        ]);
    }

    public function test_inactive_midia_is_still_in_index_list()
    {
        $midia = Midia::factory()->create([
            'empresa_id' => $this->empresa->id,
            'status' => 1, // Inactive
        ]);

        $response = $this->get(route('midias.index'));

        $response->assertStatus(200);
        $response->assertSee($midia->title);
    }

    public function test_can_list_soft_deleted_midias_in_trash()
    {
        $midia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);
        $midia->delete(); // Soft delete

        $activeMidia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);

        $response = $this->get(route('midias.trash'));

        $response->assertStatus(200);
        $response->assertSee($midia->title);
        $response->assertDontSee($activeMidia->title);
    }

    public function test_can_restore_a_soft_deleted_midia()
    {
        $midia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);
        $midia->delete();

        $response = $this->patch(route('midias.restore', $midia->id));

        $response->assertRedirect(route('midias.trash'));
        $this->assertNotSoftDeleted('midias', ['id' => $midia->id]);
    }
    
    public function test_cannot_restore_a_non_soft_deleted_midia()
    {
        $midia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);

        $response = $this->patch(route('midias.restore', $midia->id));

        $response->assertStatus(404);
    }
}